/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* koverartistcase.h
 */
#ifndef koverartistcase_h
#define koverartistcase_h

#include <qsize.h>
#include <qvaluevector.h>


namespace KoverArtist
{

class CaseLoader;


/** Dimensions and details of a cd/dvd case or box.
 * The left side is the side to the left if one looks at the front of the case.
 *
 *@author Stefan Taferner <taferner@kde.org>
 */
class Case
{
public:
   Case();
   Case(const Case&);
   Case& operator=(const Case&);
   virtual ~Case();

   /** Clear the object. */
   void clear();

   /** Name of the case. (untranslated, machine readable). */
   const QString& name() const { return mName; }

   /** Set the name of the case. */
   void setName(const QString& nm) { mName = nm; }

   /** Get/set the human readable (translated) name of the case. */
   const QString& label() const { return mLabel; }
   void setLabel(const QString& l) { mLabel = l; }

   enum ConnectSide { NotConnected=0, ConnectLeft, ConnectRight };

   /** Are front and back Sides connected? Example: DVD cases are connected on the left side. */
   ConnectSide frontBackConnected() const { return mConnected; }

   /** Dimension of the front in 1/10 millimeters. Most (all?) cases have a front side. */
   const QSize& front() const { return mFront; }
   bool hasFront() const { return !mFront.isNull(); }

   /** Dimension of the back in 1/10 millimeters. Slim cases usually have no back side. */
   const QSize& back() const { return mBack; }
   bool hasBack() const { return !mBack.isNull(); }

   /** Width of the back left side in 1/10 millimeters. Usually all CD/DVD cases execpt slim CD. */
   int backLeftSide() const { return mBackLeftSide; }
   bool hasBackLeftSide() const { return mBackLeftSide>0; }

   /** Width of the back right side in 1/10 millimeters. Standard CD cases have this, for example. */
   int backRightSide() const { return mBackRightSide; }
   bool hasBackRightSide() const { return mBackRightSide>0; }

   /** Width of the front left side in 1/10 millimeters. Multi CD cases (4-6 discs) have this. */
   int frontLeftSide() const { return mFrontLeftSide; }
   bool hasFrontLeftSide() const { return mFrontLeftSide>0; }

   /** Width of the front right side in 1/10 millimeters. Multi CD cases (4-6 discs) have this. */
   int frontRightSide() const { return mFrontRightSide; }
   bool hasFrontRightSide() const { return mFrontRightSide>0; }

   /** Load the case from file fname. */
   virtual bool loadFile(const QString& fname);

   /** Compares the labels of two cases. */
   bool operator<(const Case&) const;

   /** Returns a standard cd case. */
   static Case& standardCase();

   /** Access the list of cases. */
   static QValueVector<Case>& cases() { return sCases; }

   /** Load all cases. Returns true if any cases */
   static bool loadCases();

   /** Returns the case with name nm or NULL. */
   static Case* findCase(const QString& nm);

   /** Returns the index of the case with name nm or -1 if not found. */
   static int indexOfCase(const QString& nm);

protected:
   friend class CaseLoader;

   QString mName, mLabel;
   ConnectSide mConnected;
   QSize mFront, mBack;
   int mBackLeftSide, mBackRightSide, mFrontLeftSide, mFrontRightSide;

   static QValueVector<Case> sCases;
};

}

#endif /*koverartistcase_h*/

