/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// project.cpp

#include "project.h"
#include "settings.h"

#include "../config.h"

#include <qdom.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qfont.h>
#include <qtextstream.h>
#include <qxml.h>

#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ktempfile.h>

#include <iostream>


namespace KoverArtist
{


Project::Project()
:mTitle()
,mSubTitle()
,mImgFront()
,mImgBack()
,mImgFrontWrapAround(false)
,mStyleTitle(QFont("Arial", 32, QFont::Bold, false), Qt::black, Qt::AlignCenter)
,mStyleSubTitle(QFont("Arial", 22, QFont::Bold, false), Qt::black, Qt::AlignRight)
,mStyleContents(QFont("Arial", 14, QFont::Normal, false), Qt::black, Qt::AlignLeft)
,mStyleSide(QFont("Arial", 14, QFont::Bold, false), Qt::black, Qt::AlignLeft)
,mDiscTitleFont(QFont("Arial", 14, QFont::Bold, false))
,mOutlineColor(QColor("#888888"))
,mBackgroundColor(Qt::white)
,mTitleEffect(Effect::GLOW, QColor("#f6ff42"), Qt::white, 12, 160)
,mContentsEffect(Effect::GLOW, Qt::white, QColor("#a5a5a5"), 8, 125)
,mSideEffect(Effect::NONE)
,mCase()
,mDiscs(1)
,mUrl()
,mModified(false)
{
}


Project::Project(const Project& o)
:mTitle(o.mTitle)
,mSubTitle(o.mSubTitle)
,mImgFront(o.mImgFront)
,mImgBack(o.mImgBack)
,mImgFrontWrapAround(o.mImgFrontWrapAround)
,mStyleTitle(o.mStyleTitle)
,mStyleSubTitle(o.mStyleSubTitle)
,mStyleContents(o.mStyleContents)
,mStyleSide(o.mStyleSide)
,mDiscTitleFont(o.mDiscTitleFont)
,mOutlineColor(o.mOutlineColor)
,mBackgroundColor(o.mBackgroundColor)
,mTitleEffect(o.mTitleEffect)
,mContentsEffect(o.mContentsEffect)
,mSideEffect(o.mSideEffect)
,mCase(o.mCase)
,mDiscs(o.mDiscs)
,mUrl(o.mUrl)
,mModified(o.mModified)
{
}


Project& Project::operator=(const Project& o)
{
   if (&o!=this)
   {
      mTitle = o.mTitle;
      mSubTitle = o.mSubTitle;
      mImgFront = o.mImgFront;
      mImgBack = o.mImgBack;
      mImgFrontWrapAround = o.mImgFrontWrapAround;
      mStyleTitle = o.mStyleTitle;
      mStyleSubTitle = o.mStyleSubTitle;
      mStyleContents = o.mStyleContents;
      mStyleSide = o.mStyleSide;
      mDiscTitleFont = o.mDiscTitleFont;
      mOutlineColor = o.mOutlineColor;
      mBackgroundColor = o.mBackgroundColor;
      mTitleEffect = o.mTitleEffect;
      mContentsEffect = o.mContentsEffect;
      mSideEffect = o.mSideEffect;
      mCase = o.mCase;
      mDiscs = o.mDiscs;
      mUrl = o.mUrl;
      mModified = o.mModified;
   }
   return *this;
}


Project::~Project()
{
}


void Project::clear()
{
   *this = Project();
}


QString Project::fileExt(const QString& aFileName)
{
   int idx = aFileName.findRev('.');
   if (idx>0) return aFileName.mid(idx+1);
   return QString::null;
}


bool Project::load(const KURL& aUrl, QWidget *aWindow)
{
   QString fileName;
   bool ok = false;
   bool isLocal = aUrl.isLocalFile();

   if (isLocal) fileName = aUrl.path();
   else
   {
      if (!KIO::NetAccess::download(aUrl, fileName, aWindow))
      {
         KMessageBox::error(aWindow, KIO::NetAccess::lastErrorString());
         return false;
      }
   }

   QFileInfo fi(fileName);
   if (fi.exists() && fi.isReadable() && !fi.isDir())
   {
      ok = loadFile(fileName);
      mUrl = aUrl;
   }
   else
   {
      KMessageBox::error(aWindow, i18n("Cannot open project %1\n"
         "File does not exist or is not readable").arg(aUrl.prettyURL()));
   }

   if (!isLocal)
      KIO::NetAccess::removeTempFile(fileName);

   return ok;
}


bool Project::loadFile(const QString& aFileName)
{
   QString ext = fileExt(aFileName);
   clear();

   if (ext=="koap" || ext=="kmf" || ext.isEmpty())
      return loadFileXml(aFileName);

   KMessageBox::error(0, i18n("Cannot load project from %1:\n"
      "Invalid file format %2").arg(aFileName).arg(ext));
   return false;
}


bool Project::canLoad(const QString& aFileName)
{
   QString ext = fileExt(aFileName);

   if (ext=="koap" || ext=="kmf" || ext=="koad")
      return true;

   return false;
}


bool Project::loadFileXml(const QString& aFileName)
{
   QDomDocument doc;
   QFile in(aFileName);

   if (!in.open(IO_ReadOnly)) return false;
   if (!doc.setContent(&in)) return false;

   QDomDocumentType dt = doc.doctype();
   QString type = dt.name();

   if (type=="koverartist_project") return loadFileKoap(aFileName, doc);
   else if (type=="kmf_project") return loadFileKmf(aFileName, doc);

   KMessageBox::error(0, i18n("Cannot load project from %1:\n"
      "Unknown document type %2")
      .arg(aFileName).arg(dt.name()));
   return false;
}


bool Project::loadFileKmf(const QString& aFileName, QDomDocument& aDoc)
{
   QDomElement root = aDoc.documentElement();
   QDomElement elem, elem2;
   QString str, imgNames[2];
   int imgCount=0;

   Disc& d = mDiscs[0];

   if (!d.loadFileKmf(aFileName, aDoc)) return false;

   mTitle = d.title();
   mCase = Case::standardCase();

   // Find preview images
   for (elem=root.firstChild().toElement(); !elem.isNull() && imgCount<2;
        elem=elem.nextSibling().toElement())
   {
      if (elem.tagName()=="media")
      {
         if (elem.attribute("object")=="video")
         {
            for (elem2=elem.firstChild().toElement(); !elem2.isNull() && imgCount<2;
                 elem2=elem2.nextSibling().toElement())
	    {
               if (elem2.tagName()=="video")
               {
                  str = elem2.attribute("custom_preview");
                  if (str.isEmpty()) continue;
                  if (str.left(6)=="file:/") str = str.mid(6);
                  imgNames[imgCount++] = str;
	       }
	    }
	 }
      }
   }

   if (imgCount>=1) mImgFront.load(imgNames[0]);
   if (imgCount>=2) mImgBack.load(imgNames[1]);

   return true;
}


bool Project::loadFileKoap(const QString& aFileName, QDomDocument& aDoc)
{
   QString projectDir = aFileName;
   int idx = projectDir.findRev('/');
   if (idx>0) projectDir = projectDir.left(idx);
   else projectDir = QString::null;

   QDomElement root = aDoc.documentElement();
   QDomElement elem;
   QString nm;

   for (elem=root.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="Title")
      {
         if (!mStyleTitle.fromDom(elem)) return false;
         mTitle = elem.text();
      }
      else if (nm=="SubTitle")
      {
         if (!mStyleSubTitle.fromDom(elem)) return false;
         mSubTitle = elem.text();
      }
      else if (nm=="TitleEffect")
      {
         if (!mTitleEffect.fromDom(elem)) return false;
      }
      else if (nm=="SideEffect")
      {
         if (!mSideEffect.fromDom(elem)) return false;
      }
      else if (nm=="SideStyle")
      {
         if (!mStyleSide.fromDom(elem)) return false;
      }
      else if (nm=="Case")
      {
         if (elem.hasAttribute("background")) mBackgroundColor = QColor(elem.attribute("background"));
         else mBackgroundColor = Qt::white;

         if (elem.hasAttribute("outline")) mOutlineColor = QColor(elem.attribute("outline"));
         else mOutlineColor = QColor("#888888");

         QString cname = elem.text().stripWhiteSpace();
         Case* c = Case::findCase(cname);
         if (c) mCase = *c;
         else
         {
            int rc = KMessageBox::warningContinueCancel(0, i18n("Unknown case: %1").arg(cname));
            if (rc!=KMessageBox::Continue) return false;
         }
      }
      else if (nm=="Image")
      {
         QString t = elem.attribute("type");
         bool ok = false;
         if (t=="front")
         {
            setImgFrontWrapAround(elem.attribute("wrap")=="true");
            ok = mImgFront.fromDom(elem, projectDir);
	 }
         else if (t=="back")
            ok = mImgBack.fromDom(elem, projectDir);
         else
         {
            int rc = KMessageBox::warningContinueCancel(0, i18n("Unknown image type: %1").arg(t));
            if (rc!=KMessageBox::Continue) return false;
         }
         if (!ok) return false;
      }
      else if (nm=="Discs")
      {
         resize(0);
         if (!loadFileKoapDiscs(aDoc, elem)) return false;
      }
      else
      {
         int rc = KMessageBox::warningContinueCancel(0,
                  i18n("Unknown XML element: %1").arg(nm));
         if (rc!=KMessageBox::Continue) return false;
      }
   }

   return true;
}


bool Project::loadFileKoapDiscs(QDomDocument& aDoc, QDomElement& aDiscs)
{
   QDomElement elem;
   QString nm;

   for (elem=aDiscs.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="Effect")
      {
         if (!contentsEffect().fromDom(elem)) return false;
      }
      else if (nm=="Style")
      {
         if (!mStyleContents.fromDom(elem)) return false;
      }
      else if (nm=="Disc")
      {
         if (!loadFileKoapDisc(aDoc, elem)) return false;
      }
      else
      {
         int rc = KMessageBox::warningContinueCancel(0,
                  i18n("Unknown XML element: %1").arg("Discs/"+nm));
         if (rc!=KMessageBox::Continue) return false;
      }
   }
   return true;
}


bool Project::loadFileKoapDisc(QDomDocument&, QDomElement& aDisc)
{
   QDomElement elem;
   QString nm;

   int idx = count();
   resize(idx+1);
   Disc& d = disc(idx);

   return d.fromDom(aDisc);
}


bool Project::makeImageLocal(Image& aImg, const QString& aName, const QString& aProjFileName,
                             QWidget *aWindow)
{
   const KURL& imgUrl = aImg.url();
   int idx;

   if (aImg.pixmap().isNull()) return true;

   idx = aProjFileName.findRev('/');
   QString projDir = idx>=0 ? aProjFileName.left(idx) : aProjFileName;
   if (projDir==imgUrl.directory()) return true;

   QString fname = aProjFileName;
   idx = fname.findRev('.');
   if (idx>0 && idx>=int(fname.length()-5)) fname = fname.left(idx);
   fname += "-"+aName+"."+fileExt(imgUrl.path());

   if (!KIO::NetAccess::file_copy(imgUrl, KURL(fname), -1, true, false, aWindow))
   {
      KMessageBox::error(aWindow, KIO::NetAccess::lastErrorString());
      return false;
   }

   aImg.load(KURL(fname));
   return true;
}


bool Project::save(const KURL& aUrl, QWidget *aWindow)
{
   QString filename;
   KTempFile tempFile;
   tempFile.setAutoDelete(true);

   if (aUrl.isLocalFile()) filename = aUrl.path();
   else filename = tempFile.name();

   if (!saveFile(filename, aWindow))
      return false;

   if (!aUrl.isLocalFile())
   {
      if (!KIO::NetAccess::upload(filename, aUrl, aWindow))
      {
         KMessageBox::error(aWindow, KIO::NetAccess::lastErrorString());
         return false;
      }
   }

   mUrl = aUrl;
   mModified = false;
   return true;
}


bool Project::saveFile(const QString& aFileName, QWidget *aWindow)
{
   QString ext = fileExt(aFileName);
   bool ok = false;

   if (ext=="koap" || ext=="xml" || ext.isEmpty())
      ok = saveFileKoap(aFileName, aWindow);
   else
   {
      KMessageBox::error(0, i18n("Cannot save project as %1:\n"
         "Invalid file format %2").arg(aFileName).arg(ext));
   }

   if (ok) mModified = false;

   return ok;
}


bool Project::saveFileKoap(const QString& aFileName, QWidget *aWindow)
{
   QDomDocument doc("koverartist_project");
   QString projectDir = mUrl.directory();
   QDomElement elem;

   // Copy images into project directory
   Settings* stg = Settings::instance();
   if (stg->localImages)
   {
      if (!makeImageLocal(mImgFront, "front", aFileName, aWindow)) return false;
      if (!makeImageLocal(mImgBack, "back", aFileName, aWindow)) return false;
   }

   // Root
   QDomElement root = doc.createElement("KoverArtist");
   root.setAttribute("creator", PACKAGE);
   root.setAttribute("version", VERSION);
   doc.appendChild(root);

   // Title
   QDomElement title = doc.createElement("Title");
   mStyleTitle.toDom(doc, title);
   title.appendChild(doc.createTextNode(mTitle));
   root.appendChild(title);

   // Sub-Title
   QDomElement subTitle = doc.createElement("SubTitle");
   mStyleSubTitle.toDom(doc, subTitle);
   subTitle.appendChild(doc.createTextNode(mSubTitle));
   root.appendChild(subTitle);

   // Title effect
   elem = doc.createElement("TitleEffect");
   mTitleEffect.toDom(doc, elem);
   root.appendChild(elem);

   // Case
   elem = doc.createElement("Case");
   elem.setAttribute("background", mBackgroundColor.name());
   elem.setAttribute("outline", mOutlineColor.name());
   elem.appendChild(doc.createTextNode(mCase.name()));
   root.appendChild(elem);

   // Side effect
   elem = doc.createElement("SideEffect");
   mSideEffect.toDom(doc, elem);
   root.appendChild(elem);

   // Side style
   elem = doc.createElement("SideStyle");
   mStyleSide.toDom(doc, elem);
   root.appendChild(elem);

   // Front image
   elem = doc.createElement("Image");
   mImgFront.toDom(doc, elem, projectDir);
   elem.setAttribute("type", "front");
   elem.setAttribute("wrap", mImgFrontWrapAround ? "true" : "false");
   root.appendChild(elem);

   // Back image
   elem = doc.createElement("Image");
   mImgBack.toDom(doc, elem, projectDir);
   elem.setAttribute("type", "back");
   root.appendChild(elem);

   // Discs
   QDomElement discs = doc.createElement("Discs");
   root.appendChild(discs);

   // Disc contents effects
   elem = doc.createElement("Effect");
   mContentsEffect.toDom(doc, elem);
   discs.appendChild(elem);

   // Disc contents style
   elem = doc.createElement("Style");
   mStyleContents.toDom(doc, elem);
   discs.appendChild(elem);

   // The discs
   int i, num = count();
   QDomElement disc;
   for (i=0; i<num; ++i)
   {
      disc = doc.createElement("Disc");
      mDiscs[i].toDom(doc, disc);
      discs.appendChild(disc);
   }

   // Save the DOM
   QFile out(aFileName);
   if (!out.open(IO_WriteOnly)) return false;

   QCString hdr("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
   out.writeBlock(hdr, strlen(hdr));

   QCString domStr = doc.toString().utf8();
   int domStrLen = strlen(domStr);
   return (out.writeBlock(domStr, domStrLen)==domStrLen);
}


void Project::resize(int num)
{
   mDiscs.resize(num);
}


DiscVector::iterator Project::discIt(int aIdx)
{
   DiscVector::iterator it;
   for (it=mDiscs.begin(); it!=mDiscs.end() && aIdx>0; --aIdx, ++it)
      ;
   return it;
}


void Project::modified()
{
   mModified = true;
}


} //namespace KoverArtist
