#include <stdio.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qframe.h>
#include <qlineedit.h>
#include <qhbuttongroup.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qsettings.h>
#include <qlayout.h>

#include "../dissipate2/sipcall.h"
#include "../dissipate2/sipuser.h"
#include "../dissipate2/sipregister.h"
#include "../dissipate2/sipclient.h"
#include "kphone.h"
#include "kphoneview.h"
#include "ksipauthentication.h"
#include "ksipregistrations.h"

KSipIdentityEdit::KSipIdentityEdit( QWidget *parent, const char *name, QObject *reg )
	: QDialog( parent, QString( name ) + "Identity Editor", true )
{
	QVBox *vbox = new QVBox( this );
	vbox->setMargin( 3 );
	vbox->setSpacing( 3 );
	QVBoxLayout *vboxl = new QVBoxLayout( this, 5 );
	vboxl->addWidget( vbox );

	(void) new QLabel( "Full Name:", vbox );
	fullname = new QLineEdit( vbox );
	fullname->setMinimumWidth( fontMetrics().maxWidth() * 20 );
	(void) new QLabel( "User Part of SIP URL:", vbox );
	username = new QLineEdit( vbox );
	(void) new QLabel( "Host Part of SIP URL:", vbox );
	hostname = new QLineEdit( vbox );
	(void) new QLabel( "Outbound Proxy (optional):", vbox );
	sipProxy = new QLineEdit( vbox );
	(void) new QLabel( "Authentication Username (optional):", vbox );
	sipProxyUsername = new QLineEdit( vbox );
	(void) new QLabel( "q-value between 0.0-1.0 (optional):", vbox );
	qValue = new QLineEdit( vbox );
	autoRegister = new QCheckBox( "Auto Register", vbox );
	labelRegister = new QLabel( "", vbox );
	buttonRegister = new QPushButton( "Register", vbox );
	connect( buttonRegister, SIGNAL( clicked() ), reg, SLOT( changeRegistration() ) );

	QHBoxLayout *buttonBox;
	buttonBox = new QHBoxLayout( vboxl, 6 );
	helpPushButton = new QPushButton( this, "help button" );
	helpPushButton->setText( "&help..." );
	buttonBox->addWidget( helpPushButton );
	QSpacerItem *spacer = new QSpacerItem(
		0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum );
	buttonBox->addItem( spacer );
	okPushButton = new QPushButton( this, "ok button" );
	okPushButton->setText( "OK" );
	okPushButton->setDefault( TRUE );
	buttonBox->addWidget( okPushButton );
	cancelPushButton = new QPushButton( this, "cancel button" );
	cancelPushButton->setText( "Cancel" );
	cancelPushButton->setAccel( Key_Escape );
	buttonBox->addWidget( cancelPushButton );
	connect( okPushButton, SIGNAL( clicked() ),
		this, SLOT( slotOk() ) );
	connect( cancelPushButton, SIGNAL( clicked() ),
		this, SLOT( slotCancel() ) );
	connect( helpPushButton, SIGNAL( clicked() ),
		parent, SLOT( showHelp() ) );
}

KSipIdentityEdit::~KSipIdentityEdit( void )
{
}

void KSipIdentityEdit::updateState( QString state )
{
	if( state == "off" ) {
		buttonRegister->setText( "Register" );
		buttonRegister->setEnabled( false );
	} else if( state == "Active" ) {
		labelRegister->setText( "Registration : " + state );
		buttonRegister->setText( "Unregister" );
		buttonRegister->setEnabled( true );
	} else {
		labelRegister->setText( "Registration : " + state );
		buttonRegister->setText( "Register" );
		buttonRegister->setEnabled( true );
	}
}

QString KSipIdentityEdit::getFullname( void ) const
{
	return fullname->text();
}

QString KSipIdentityEdit::getUsername( void ) const
{
	return username->text();
}

QString KSipIdentityEdit::getHostname( void ) const
{
	return hostname->text();
}

QString KSipIdentityEdit::getSipProxy( void ) const
{
	return sipProxy->text();
}

QString KSipIdentityEdit::getSipProxyUsername( void ) const
{
	return sipProxyUsername->text();
}

QString KSipIdentityEdit::getQvalue( void ) const
{
	bool ok;
	float f = qValue->text().toFloat( &ok );
	if( ok && f >= 0 && f <= 1 ) {
		return qValue->text();
	}
	return "";
}

QString KSipIdentityEdit::getUri( void ) const
{
	return "\"" + fullname->text() + "\" <sip:" + username->text() + "@" + hostname->text() + ">";
}

void KSipIdentityEdit::setFullname( const QString &newFullname )
{
	fullname->setText( newFullname );
}

void KSipIdentityEdit::setUsername( const QString &newUsername )
{
	username->setText( newUsername );
}

void KSipIdentityEdit::setHostname( const QString &newHostname )
{
	hostname->setText( newHostname );
}

void KSipIdentityEdit::setSipProxy( const QString &newSipProxy )
{
	sipProxy->setText( newSipProxy );
}

void KSipIdentityEdit::setQvalue( const QString &qvalue )
{
	qValue->setText( qvalue );
}

void KSipIdentityEdit::setSipProxyUsername( const QString &newSipProxyUsername )
{
	sipProxyUsername->setText( newSipProxyUsername );
}

void KSipIdentityEdit::slotOk( void )
{
	if( username->text() == "" || hostname->text() == "" ) {
		QDialog::reject();
	} else {
		update();
		QDialog::accept();
	}
}

void KSipIdentityEdit::slotCancel( void )
{
	QDialog::reject();
}

KSipRegistrations::KSipRegistrations(
		SipClient *client, KPhoneView *phoneView, QWidget *p, const char *prefix )
{
	parent = p;
	c = client;
	v = phoneView;
	userPrefix = prefix;
	useStun = false;

	edit = new KSipIdentityEdit( parent, userPrefix.latin1(), this );
	connect( edit, SIGNAL( update() ), this, SLOT( update() ) );
	sipreg = 0;
	QString uristr;
	QString str;
	SipUri uri;
	QSettings settings;
	QString pp = "/kphone/" + userPrefix + "/local/";
	expires = settings.readNumEntry( pp + "/RegistrationExpiresTime", constRegistrationExpiresTime );

	if( expires == 0 ) {
		expires = -1;
	}
	pp = "/kphone/" + userPrefix + "Registration/";
	if( settings.readEntry( pp + "/SipUri", "" ) != "" ) {
		uristr = settings.readEntry( pp + "/SipUri" );
		uri = SipUri( uristr );
		u = c->getUser( uri );
		if( u == NULL ) {
			u = new SipUser( c, uri );
		}
		stunSrv = "";
		if( settings.readEntry( "/kphone/STUN/UseStun", "" ) == "Yes" ) {
			useStun = true;
			stunSrv = settings.readEntry( "/kphone/STUN/StunServer", constStunServer );
			if( stunSrv.isEmpty() ) {
				QString dname = u->getMyUri()->getHostname();
				stunSrv = dname;
				QString srv = client->getSRV( QString( "_stun._udp." ) + dname );
				if( !srv.isEmpty() ) {
					stunSrv = srv;
				}
				stunSrv += ":3478";
			} else {
				if( !stunSrv.contains( ':' ) ) {
					stunSrv += ":3478";
				}
			}
		}
		uristr = "";
		if( settings.readEntry( pp + "/SipServer", "" ) != "" ) {
			uristr = settings.readEntry( pp + "/SipServer" );
		}
		QString qvalue = settings.readEntry( pp + "/qValue", "" );
		sipreg = new SipRegister( u, SipUri( uristr ), expires, qvalue );
		connect( sipreg, SIGNAL( statusUpdated() ),
			this, SLOT( registerStatusUpdated() ) );
		v->updateIdentity( u, sipreg );
		c->updateIdentity( u, sipreg->getOutboundProxy() );
		str = settings.readEntry( pp + "/UserName" );
		u->getMyUri()->setProxyUsername( str );
		str = settings.readEntry( pp + "/AutoRegister" );
		if( str == "Yes" ) {
			autoRegister = true;
			if( useStun ) {
				sipreg->setAutoRegister( true );
			} else {
				sipreg->requestRegister();
			}
		} else {
			autoRegister = false;
			sipreg->setAutoRegister( false );
		}
	} else {
		editRegistration();
	}
}

KSipRegistrations::~KSipRegistrations( void )
{
}

void KSipRegistrations::showIdentity( void )
{
	editRegistration();
}

void KSipRegistrations::save( void )
{
}

void KSipRegistrations::editRegistration( void )
{
	if( sipreg ) {
		setRegisterState();
		edit->setFullname( u->getUri().getFullname() );
		edit->setUsername( u->getUri().getUsername() );
		edit->setHostname( u->getUri().getHostname() );
		edit->setSipProxy( sipreg->getOutboundProxy() );
		edit->setSipProxyUsername( u->getUri().getProxyUsername() );
		edit->setAutoRegister( autoRegister );
		edit->setQvalue( sipreg->getQvalue() );
	} else {
		edit->updateState("off");
		edit->setFullname( "" );
		edit->setUsername( "" );
		edit->setHostname( "" );
		edit->setSipProxy( "" );
		edit->setSipProxyUsername( "" );
		edit->setQvalue( "" );
		edit->setAutoRegister( true );
	}
	edit->show();
}

void KSipRegistrations::update( void )
{
	bool isDiff = false;
	QSettings settings;
	QString p = "/kphone/" + userPrefix + "Registration/";

	QString s = edit->getSipProxy();
	if( settings.readEntry( p + "/SipUri", "" ) != edit->getUri() ||
	    settings.readEntry( p + "/SipServer", "" ) != edit->getSipProxy() ||
	    settings.readEntry( p + "/UserName", "" ) != edit->getSipProxyUsername() ||
	    settings.readEntry( p + "/qValue", "" ) != edit->getQvalue() ) {
		isDiff = true;
	}
	settings.writeEntry( p + "/SipUri", edit->getUri() );
	settings.writeEntry( p + "/SipServer", edit->getSipProxy() );
	settings.writeEntry( p + "/UserName", edit->getSipProxyUsername() );
	if( edit->getAutoRegister() ) {
		autoRegister = true;
		settings.writeEntry( p + "/AutoRegister", "Yes");
	} else {
		autoRegister = false;
		settings.writeEntry( p + "/AutoRegister", "No");
	}

	s = edit->getSipProxy();


	settings.writeEntry( p + "/qValue", edit->getQvalue() );

	if( !sipreg ) {
		QString uristr = edit->getUri();
		SipUri uri = SipUri( uristr );
		u = c->getUser( uri );
		if( u == NULL ) {
			u = new SipUser( c, uri );
		}
		uristr = edit->getSipProxy();
		QString qvalue = settings.readEntry( p + "qValue", "" );
		sipreg = new SipRegister( u, SipUri( uristr ), expires, qvalue );
		connect( sipreg, SIGNAL( statusUpdated() ),
			this, SLOT( registerStatusUpdated() ) );
		v->updateIdentity( u, sipreg );
		c->updateIdentity( u, sipreg->getOutboundProxy() );
		QString str = edit->getSipProxyUsername();
		u->getMyUri()->setProxyUsername( str );
		if( edit->getAutoRegister() ) {
			autoRegister = true;
			sipreg->requestRegister();
		} else {
			autoRegister = false;
			sipreg->setAutoRegister( false );
		}
		if( edit->getAutoRegister() ) {
			changeRegistration();
		}
	} else {
		if( isDiff ) {
			QMessageBox::information( parent, "Identity",
				"Restart Kphone to apply identity changes." );
		}
	}
}

void KSipRegistrations::changeRegistration( void )
{
	if( sipreg ) {
		if( sipreg->getRegisterState() == SipRegister::Connected ) {
			sipreg->requestClearRegistration();
		} else {
			sipreg->updateRegister();
			v->setContactsOnline();
		}
	}
}

void KSipRegistrations::unregAllRegistration( void )
{
	if( sipreg ) {
		if( sipreg->getRegisterState() == SipRegister::Connected ) {
			sipreg->requestClearRegistration();
		}
	}
}

void KSipRegistrations::setRegisterState( void )
{
	switch( sipreg->getRegisterState() ) {
		case SipRegister::NotConnected:
			edit->updateState( "Inactive" );
			break;
		case SipRegister::TryingServer:
		case SipRegister::TryingServerWithPassword:
			edit->updateState( "Trying..." );
			break;
		case SipRegister::AuthenticationRequired:
		case SipRegister::AuthenticationRequiredWithNewPassword:
			edit->updateState( "Auth Required..." );
			break;
		case SipRegister::Connected:
			edit->updateState( "Active" );
			break;
		case SipRegister::Disconnecting:
			edit->updateState( "Unregistering..." );
			break;
		case SipRegister::Reconnecting:
			edit->updateState( "Reregistering..." );
			break;
	}
}

void KSipRegistrations::registerStatusUpdated( void )
{
	setRegisterState();
	if( sipreg->getRegisterState() != SipRegister::AuthenticationRequired &&
			sipreg->getRegisterState() != SipRegister::AuthenticationRequiredWithNewPassword ) {
		return;
	}
	QString authtype;
	switch( sipreg->getAuthenticationType() ) {
		case SipRegister::DigestAuthenticationRequired:
			authtype = "Digest Authentication Request"; break;
		case SipRegister::BasicAuthenticationRequired:
			authtype = "Basic Authentication Request"; break;
		case SipRegister::ProxyDigestAuthenticationRequired:
			authtype = "Proxy Digest Authentication Request"; break;
		case SipRegister::ProxyBasicAuthenticationRequired:
			authtype = "Proxy Basic Authentication Request"; break;
	}
	QString server = sipreg->getServerUri().proxyUri();
	QString sipuri = u->getUri().uri();
	KSipAuthenticationRequest authreq( server, sipuri, userPrefix + authtype );
	QString username = sipreg->getRegisterCall()->getProxyUsername();
	authreq.setUsername( username );
	QString password = sipreg->getRegisterCall()->getPassword();
	if( password.isEmpty() || sipreg->getRegisterState() == SipRegister::AuthenticationRequiredWithNewPassword ) {
		if( authreq.exec() ) {
			if( authreq.getUsername().isEmpty() || authreq.getUsername().isEmpty() ) {
				return;
			}
			sipreg->getRegisterCall()->setProxyUsername( authreq.getUsername() );
			sipreg->getRegisterCall()->setPassword( authreq.getPassword() );
			sipreg->requestRegister( authreq.getUsername(), authreq.getPassword() );
		}
	} else {
		sipreg->requestRegister( username, password );
	}
}
