/***************************************************************************
 *   Copyright (C) 2006 by Danny Kukawka                                   *
 *                         <dkukawka@suse.de>, <danny.kukawka@web.de>      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

/*! \file 	inactivity.cpp
 * \brief 	In this file can be found the inactivity related code. 
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \version 	0.6.0
 * \date    	2006
 */

// KDE Headers
#include <klocale.h>

#include "inactivity.h"

/* needed for lXext C library linkage */
extern "C" {
	#include <X11/extensions/scrnsaver.h>
}
 
/*! The default constructor of the class autosuspend */
inactivity::inactivity() {
	myDebug ("inactivity::inactivity");

	proc = NULL;

	timeToInactivity = 0;
	blacklisted_running_last = 0;
	
	pidof_call_failed = false;
	pidof_call_started = false;
	pidof_call_returned = false;
	blacklisted_running = false;
	
	int dummy = 0;
	has_XSC_Extension = XScreenSaverQueryExtension( qt_xdisplay(), &dummy, &dummy );
	
	checkInactivity =  new QTimer( this );
	connect( checkInactivity, SIGNAL(timeout()), this, SLOT(check()));
}

/*! The default destructor of the class autosuspend */
inactivity::~inactivity() {
	myDebug ("inactivity::~inactivity");
}

/*!
 * This function start the monitoring of inactivity of user on the X-Server.
 * Here wee set the time for the signal \ref autosuspendTimeExpired() and start
 * the needed QTimer.
 * \param timeToExpire  Integer value representing the time of inactivity which need 
 *			to elapse befor send signal. The time is in seconds.
 * \param blacked	QStringList with blacklisted programs which if detected with 
 *			pidof() as running prevent the autosuspend.
 */
void inactivity::start( int timeToExpire, QStringList blacked ) {
	myDebug ("inactivity::start");

	blacklist = blacked;
 
	if(timeToExpire > 0 && has_XSC_Extension){
		stop();
		timeToInactivity = (unsigned long) (timeToExpire * 1000);
		checkInactivity->start(CHECK_for_INACTIVITY, true);
	}
}

/*!
 * \b SLOT to call check without a recheck.
 */
void inactivity::check() {
	check(false);
}

/*!
 * \b SLOT to call check as recheck inactivity if befor a running PID 
 * request was detected.
 */
void inactivity::recheck() {
	myDebug ("inactivity::recheck");
	check(true);
}

/*!
 * \b SLOT to check the current idle-time of the X-Server and if there
 * are blacklisted programs are running. If the through \ref timeToInactivity 
 * defined time is expired, this function emit signal \ref autosuspendTimeExpired() .
 * \param recheck   boolean which define if this is a recheck or not. 
 *                  \li true, if this is a recheck. In this case we didn't call
 *                      \ref checkBlacklisted() again
 *                  \li false, if this is normal check
 */
void inactivity::check( bool recheck ) {
	myDebug ("inactivity::check");
	
	checkXInactivity();
	if (!pidof_call_started && !recheck) checkBlacklisted();
	
	if( idleTime < blacklisted_running_last ) {
		blacklisted_running_last = idleTime;
	}
 
	if((idleTime - blacklisted_running_last ) >= timeToInactivity) {
		if (!pidof_call_started) {
			if (( pidof_call_returned && !blacklisted_running ) ||
			    ( pidof_call_returned && pidof_call_failed )) {
				emit inactivityTimeExpired();
			}
			else {
				checkInactivity->start(CHECK_for_INACTIVITY, true);
			}
		}
		else {
			//called if there is a getPIDs() is running
			QTimer::singleShot(500, this, SLOT(recheck()));
		}
	}
	else checkInactivity->start(CHECK_for_INACTIVITY, true);
}

/*!
 * This function stop the monitoring and reset all variables and Timer.
 */
void inactivity::stop() {
	myDebug ("inactivity::stop");

	if (checkInactivity->isActive()) checkInactivity->stop();
	timeToInactivity = 0;
	idleTime = 0;
	blacklisted_running_last = 0;
	
	pidof_call_failed = false;
	pidof_call_started = false;
	pidof_call_returned = false;
	blacklisted_running = false;
}

/*!
 * This function query the idle-time of user-imput from the X-Server and set
 * the return value to \ref idleTime.
 */
void inactivity::checkXInactivity(){
	myDebug ("inactivity::checkXInactivity");

	if(has_XSC_Extension) {
		static XScreenSaverInfo* mitInfo = 0;
		if (!mitInfo) mitInfo = XScreenSaverAllocInfo ();
		XScreenSaverQueryInfo (qt_xdisplay(), DefaultRootWindow (qt_xdisplay()), mitInfo);
		idleTime = mitInfo->idle;
	}
	else {
		idleTime = 0;
	}
	myDebug ("autosuspend::checkXInactivity - idleTime: %f", (double)idleTime);
}

/*!
 * This funtion starts the monitoring of blacklisted processes.
 */
void inactivity::checkBlacklisted(){
	myDebug ("inactivity::checkBlacklisted");

	if (proc != NULL)
		delete proc;	
	
	proc = new KProcess;
	*proc << "pidof" << blacklist;
	
	connect( proc, SIGNAL(receivedStdout(KProcess *, char *, int)),this,
		 SLOT(getPIDs(KProcess *, char *, int)));
	connect( proc, SIGNAL(processExited(KProcess *)),
		 SLOT(getPIDsExited(KProcess *)));
	
	if (!proc->start(KProcess::NotifyOnExit, KProcess::AllOutput))
		emit displayErrorMsg(i18n("Could not start 'pidof'. "
					  "Could not autosuspend the machine.\n"
					  "Please check your installation."));
 
	pidof_call_started = true;
	pidof_call_returned = false;
	pidof_call_failed = false;
}


/*!
 * \b SLOT to get the return of the command pidof and parse this to set
 * \ref blacklisted_running .
 * \param *proc     pointer to the sending KProcess
 * \param *buffer   the char pointer to the output of the process to stdout
 * \param *lenght   the length of the buffer
 */
void inactivity::getPIDs(KProcess *proc, char *buffer, int lenght) {
	myDebug ("inactivity::getPIDs");	

	if( proc == NULL || lenght == 0 ) ; // to prevent compiler warning

	QString pids(buffer);
	pids.remove(" ");
	if(pids.isEmpty() || pids == "\n" ) {
		myDebug("NO! BLACKLISTED IS RUNNING");
		blacklisted_running = false;
	} 
	else {
		if (pids.contains(QRegExp::QRegExp("[0-9]"))) {
			myDebug("BLACKLISTED IS RUNNING");
			blacklisted_running = true;
			blacklisted_running_last = idleTime;
		}
		else {
			myDebug("GET BLACKLISTED FAILED - WRONG RETURN");
			blacklisted_running = false;
			pidof_call_failed = true;
		}
	}
}


/*!
 * \b SLOT which called if the call of pidof is exited
 * \param proc the KPocess which called this SLOT
 */
void inactivity::getPIDsExited(KProcess *proc){
	myDebug ("inactivity::getPIDsExited");

	pidof_call_returned = true;
	pidof_call_started = false;
	
	
	if (proc->normalExit()){
		// if returned some pids or if pid returned nothing
		if (proc->exitStatus() == 1 || proc->exitStatus() == 0){
			pidof_call_failed = false;
			return;
		}
	}	
	// if something crashed/failed
	pidof_call_failed = true;
}
