/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "stationlistmodel.h"
#include "settings_general.h"
#include <QFile>
#include <KProgressDialog>
#include <KLocale>
#define AND  &&
#define OR  ||
#define NOT  !
#define EQUAL  ==

stationlistModel::stationlistModel(const QPointer<QObject> parent,
                                    const QPointer<QWidget> mainWidget)
                                  : QAbstractTableModel(parent)
{
//TODO: check if yet one object exists. When yes, don't create another object. Es darf
// nur ein einziges Objekt dieser Klasse gleichzeitig geben!
  // variables
  QStringList m_stringList;
  int i;  // using int because this is the return type of QStringList::size()

  // general setup
  m_mainWidget = mainWidget;

  // setup data model
  m_stringList = settings_general::streamConfigFiles();
  for (i = 0; i < m_stringList.size(); ++i) {
    m_stationlist.append(new radioStation(this, m_mainWidget, m_stringList.at(i), i));
    helper_connectSignalsAndSlots(m_stationlist.last());
  };
  helper_writeStationListToGeneralSettings();  /* This is because when radioStation can't
  open the file, it creates a new one. Probaby this won't happen, but we have to make sure
  that the list in the general config file is actual. */

  // initialize attributes
  recalculate_numberOfActiveStreams_and_bandwidth(); // now possible (after making stationlist)
  m_listOfStreamsOfWhichTheUserWantsThatTheyRip = 0;
}

stationlistModel::~stationlistModel()
{
  delete m_listOfStreamsOfWhichTheUserWantsThatTheyRip;
}

void stationlistModel::helper_connectSignalsAndSlots(QPointer<radioStation> m_stream)
{
    connect(m_stream, SIGNAL(streamNameChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadStreamName(qlonglong)));
    connect(m_stream, SIGNAL(statusChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadStatus(qlonglong)));
    connect(m_stream, SIGNAL(errorChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadStatus(qlonglong)));
    connect(m_stream, SIGNAL(songChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadSong(qlonglong)));
    connect(m_stream, SIGNAL(dataSizeChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadDataSize(qlonglong)));
    connect(m_stream, SIGNAL(bitrateChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadBitrate(qlonglong)));
    connect(m_stream, SIGNAL(bitrateChanged(qlonglong, PropertyValue)),
            this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
    connect(m_stream, SIGNAL(metaIntervalChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadMetaInterval(qlonglong)));
    connect(m_stream, SIGNAL(serverNameChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadServerName(qlonglong)));
    connect(m_stream, SIGNAL(relayPortChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadRelayPort(qlonglong)));
    connect(m_stream, SIGNAL(running()),
            this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
    connect(m_stream, SIGNAL(not_running()),
            this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
}

int stationlistModel::rowCount (const QModelIndex & parent) const
{
  return m_stationlist.size();
}

int stationlistModel::columnCount (const QModelIndex & parent) const
{
  return 8;
}

QVariant stationlistModel::columnInfo(const columnInfoType type,
                                       const int column,
                                       const qint64 row,
                                       const quint64 value) const
{
  switch (type) {
    case columnHeaderTitle:
    case columnHeaderToolTip:
    case columnHeaderWhatsThis:
    case columnWidth:
    case setColumnWidth:
    case columnVisibility:
    case setColumnVisibility:
      break;
    default:
      if ((row >= rowCount()) OR (row < 0)) {
        return QVariant();
      };
  }
  /* IF data is requested that is NOT depending on the "row"
  *  THEN no check is needed.
  *  ELSE (=when data is depending on the "row") we check if it is a valid row.
  *  If not, we return a QVariant(). This is important, because the rest of this
  *  function body assumes that "row" is valid and uses it for QList::at(row) - and
  *  when "row" is invalid, this function leeds to a crash.
  *
  *  I use the "switch" systax because this looks clearer than an "if" construct.
  *  And I test for the cases where the row is NOT important - this way (when one
  *  day the enum gets more items that depend on the row - and we forget to change
  *  it here) at least we don't get a crash.
  *
  *  We don't test if "column" is valid, because the rest of the function is a
  *  switch-case, and when there nothing matches, an invalid QVariant is returned. */

  switch (column) {

    case 0:
    {
      switch (type) {
        case columnHeaderTitle:
          return i18nc("@title:column", "stream name");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return QVariant();
        case columnWidth:
          return settings_general::columnWidth_streamName();
        case setColumnWidth:
          settings_general::setColumnWidth_streamName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_streamName();
        case setColumnVisibility:
          settings_general::setColumnVisibility_streamName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->streamName().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->streamName().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->streamName().whatsThis;
      };
      break;
    }

    case 1:
    {
      switch (type) {
        case columnHeaderTitle:
          return i18nc("@title:column", "status");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return QVariant();
        case columnWidth:
          return settings_general::columnWidth_statusAndError();
        case setColumnWidth:
          settings_general::setColumnWidth_statusAndError(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_statusAndError();
        case setColumnVisibility:
          settings_general::setColumnVisibility_statusAndError(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          if (m_stationlist.at(row)->status().formatedValue.isEmpty()) {
            return m_stationlist.at(row)->error().formatedValue;
          }
          else if (m_stationlist.at(row)->error().formatedValue.isEmpty()) {
            return m_stationlist.at(row)->status().formatedValue;
          } else {
            return QString(i18nc("@item:intable This produces the status message from the "
                                     "actual status (%1) and the error message (%3). "
                                     "Both of them are guaranteed to "
                                     "be not empty. %2 is replayed by an en-dash (Unicode "
                                     "U+2013). Use it or replace it by something what looks "
                                     "nicer in your language.",
                                   "%1 %2 %3",
                                   m_stationlist.at(row)->status().formatedValue,
                                   QString(QChar(0x2013)),
                                   m_stationlist.at(row)->error().formatedValue));
          };
        case columnDataToolTip:  // TODO provide useful return value!
        case columnDataWhatsThis:  // TODO provide useful return value!
          return QVariant();
      };
      break;
    }

    case 2:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column header of the column with the name of the actual track",
                        "track");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_song();
        case setColumnWidth:
          settings_general::setColumnWidth_song(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_song();
        case setColumnVisibility:
          settings_general::setColumnVisibility_song(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->song().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->song().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->song().whatsThis;
      };
      break;
    }

    case 3:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "track size");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_dataSize();
        case setColumnWidth:
          settings_general::setColumnWidth_dataSize(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_dataSize();
        case setColumnVisibility:
          settings_general::setColumnVisibility_dataSize(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->dataSize().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->dataSize().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->dataSize().whatsThis;
      };
      break;
    }

    case 4:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "bit rate");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_bitrate();
        case setColumnWidth:
          settings_general::setColumnWidth_bitrate(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_bitrate();
        case setColumnVisibility:
          settings_general::setColumnVisibility_bitrate(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->bitrate().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->bitrate().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->bitrate().whatsThis;
      };
      break;
    }

    case 5:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "meta data interval");
        case columnHeaderToolTip:
          return  i18nc("@info:tooltip", "interval of meta data");
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_metaInterval();
        case setColumnWidth:
          settings_general::setColumnWidth_metaInterval(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_metaInterval();
        case setColumnVisibility:
          settings_general::setColumnVisibility_metaInterval(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->metaInterval().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->metaInterval().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->metaInterval().whatsThis;
      };
      break;
    }

    case 6:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "server name");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_serverName();
        case setColumnWidth:
          settings_general::setColumnWidth_serverName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_serverName();
        case setColumnVisibility:
          settings_general::setColumnVisibility_serverName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->serverName().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->serverName().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->serverName().whatsThis;
      };
      break;
    }

    case 7:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column 'relay port' means 'the port that the relay server uses'",
                        "relay port");
        case columnHeaderToolTip:
          return  i18nc("@info:tooltip", "port of the relay server");
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_relayPort();
        case setColumnWidth:
          settings_general::setColumnWidth_relayPort(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_relayPort();
        case setColumnVisibility:
          settings_general::setColumnVisibility_relayPort(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          return m_stationlist.at(row)->relayPort().formatedValue;
        case columnDataToolTip:
          return m_stationlist.at(row)->relayPort().toolTip;
        case columnDataWhatsThis:
          return m_stationlist.at(row)->relayPort().whatsThis;
      };
      break;
    }

  /* No "default:"! This way we get a compiler warning when one day
  *  the enum type has more items and we forget to implement this here. */

  };
  return QVariant();  // Return an invalid QVariant if the given "column" was invalid.
}

void stationlistModel::recalculate_numberOfActiveStreams_and_bandwidth()
{
  // variables
  int i;
  int temp_numberOfActiveStreams = 0;
  quint64 temp_bandwidth = 0;

  // code

  // calculate new values
  for (i=0; i < m_stationlist.size(); ++i) {
    if (m_stationlist.at(i)->status().internalValue.value<ripping::statusType>() != ripping::idle) {
      temp_numberOfActiveStreams++;
      if (m_stationlist.at(i)->bitrate().type == PropertyValue::value) {
        temp_bandwidth = temp_bandwidth +
                             m_stationlist.at(i)->bitrate().internalValue.toULongLong();
      };
    };
  };

  // actualize properties if necessary
  if (internal_numberOfActiveStreams != temp_numberOfActiveStreams) {
    internal_numberOfActiveStreams = temp_numberOfActiveStreams;
    emit numberOfActiveStreamsChanged();
    if (internal_numberOfActiveStreams == 0) {
      emit numberOfActiveStreamsIsZero();
    };
  };
  if (internal_bandwidth != temp_bandwidth) {
    internal_bandwidth = temp_bandwidth;
    emit bandwidthChanged();
  };
}

void stationlistModel::reloadStreamName(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 0), index(stationIndex, 0));
}

void stationlistModel::reloadStatus(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 1), index(stationIndex, 1));
}

void stationlistModel::reloadSong(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 2), index(stationIndex, 2));
}

void stationlistModel::reloadDataSize(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 3), index(stationIndex, 3));
}

void stationlistModel::reloadBitrate(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 4), index(stationIndex, 4));
}

void stationlistModel::reloadMetaInterval(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 5), index(stationIndex, 5));
}

void stationlistModel::reloadServerName(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 6), index(stationIndex, 6));
}

void stationlistModel::reloadRelayPort(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 7), index(stationIndex, 7));
}

QVariant stationlistModel::headerData(int section, Qt::Orientation orientation, int role) const
{
     if (orientation EQUAL Qt::Vertical) {
       if (role == Qt::DisplayRole) {
         return (int(section+1));
       } else {
         return QVariant();
       };
     } else {
       switch (role) {
         case Qt::DisplayRole:
           return columnInfo(columnHeaderTitle, section);
         case Qt::ToolTipRole:
           return columnInfo(columnHeaderToolTip, section);
         case Qt::WhatsThisRole:
           return columnInfo(columnHeaderWhatsThis, section);
         default:
           return QVariant();  // other "display roles" aren't supported.
       };
     };
     return QVariant();  // This point should never be reached. It just exists
                         // to be sure to deliver a return value, also when
                         // I introduce new bugs in the code above.
}

QVariant stationlistModel::data(const QModelIndex &index, int role) const
{
  //variables
  QVariant returnValue;

  //code
  if (index.model() EQUAL this) { // test if the index belongs to THIS model
    switch (role) {
      case Qt::DisplayRole:
        returnValue = columnInfo(columnData, index.column(), index.row());
        break;
      case Qt::ToolTipRole:
        returnValue = columnInfo(columnDataToolTip, index.column(), index.row());
        break;
      case Qt::WhatsThisRole:
        returnValue = columnInfo(columnDataWhatsThis, index.column(), index.row());
        break;
      default: // other "display roles" aren't supported.
        returnValue = QVariant();
        break;
    };
  }

  //return
  return returnValue;
}

void stationlistModel::sort (int column, Qt::SortOrder order)
{
  // TODO qStableSort(m_stationlist.begin(), m_stationlist.end(), ???);
}

void stationlistModel::addNewStation()
{
  // consts
  const int size_of_list = m_stationlist.size();

  // variables
  QPointer<radioStation> m_newStation;
  int exitCode;
  QString m_fileName;

  // code
  m_newStation = new radioStation(this, m_mainWidget, QString(), size_of_list);
  exitCode = m_newStation->execSettingsDialog();
  if (exitCode == QDialog::Accepted) {  // insert the new station in our model
    beginInsertRows(QModelIndex(), size_of_list, size_of_list);
    m_stationlist.append(m_newStation);
    helper_connectSignalsAndSlots(m_stationlist.last());
    helper_writeStationListToGeneralSettings();
    endInsertRows();
  } else {  // delete the station object and the config file
    m_fileName = m_newStation->configFileName();
    delete m_newStation;
    // We must wait with constructing QFile until the station
    // object has been deleted (we shouldn't open a file that's yet open!).
    QFile theFile(m_fileName);
    theFile.remove();
  };
}

void stationlistModel::deleteStation(const int index)
{
  // test if the index is valid (if not, QList.at() would crash!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    // variables
    QString filename;
    int i;
    QFile the_file;

    // code
    // save the filename of the config file, we'll need it later to delete the file:
    filename = m_stationlist.at(index)->configFileName();

    beginRemoveRows(QModelIndex(), index, index);

    delete m_stationlist.at(index);  // delete the radioStation object

    // remove the pointer to the (now deleted) radioStation object from our internal list:
    m_stationlist.removeAt(index);
    // synchronize the general config file with our internal list:
    helper_writeStationListToGeneralSettings();
    // correct the index for all following rows (which is now changed):
    for (i=index; i<m_stationlist.size(); ++i) {
      m_stationlist.at(i)->setIndex(i);
    };

    endRemoveRows();

    // delete the config file of the stream
    the_file.setFileName(filename);
    the_file.remove();

  };
}

inline void stationlistModel::helper_writeStationListToGeneralSettings()
{
  // variables
  int i;
  QStringList helper_stringlist;

  // code
  for (i = 0; i < m_stationlist.size(); ++i) {
    helper_stringlist.append(m_stationlist.at(i)->configFileName());
  };
  settings_general::setStreamConfigFiles(helper_stringlist);
  settings_general::self()->writeConfig();
}

void stationlistModel::record(const int index)
{
  // test if the index is valid (when it is not valid, we may not use QList.at()!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    m_stationlist.at(index)->startStreamripper();
  };
}

void stationlistModel::stopRecording(const int index)
{
  // test if the index is valid (when it is not valid, we may not use QList.at()!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    m_stationlist.at(index)->shutDown();
  };
}

void stationlistModel::showConfigDialog(const int index)
{
  // test if the index is valid (when it is not valid, we may not use QList.at()!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    m_stationlist.at(index)->showSettingsDialog();
  };
}

quint64 stationlistModel::bandwidth() const
{
  return internal_bandwidth;
}

int stationlistModel::numberOfActiveStreams() const
{
  return internal_numberOfActiveStreams;
}

void stationlistModel::rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone()
{
  // variables
  int i;
  QStringList temp_stringList;

  // code
  if (m_listOfStreamsOfWhichTheUserWantsThatTheyRip == 0) {
    for (i=0; i < rowCount(); i++) {
      if (m_stationlist.at(i)->doesTheUserWantsThatTheStreamIsRipping()) {
        temp_stringList.append(m_stationlist.at(i)->configFileName());
      };
    };
    m_listOfStreamsOfWhichTheUserWantsThatTheyRip = new QStringList(temp_stringList);
  };
}

bool stationlistModel::queryClose()
{
  // variables
  int i;

  // code

  rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone();

  // start shutdown for all processes
  if (numberOfActiveStreams() > 0) {
    for (i = 0; i < m_stationlist.size(); ++i) {
      m_stationlist.at(i)->shutDown();
    };
  };

  /* Now we test a second time if there are still running streams. (This could
  have changed because we have shutted down before.) If there remain running
  streams, we display a "busy" dialog until the shutdown is finished. */
  if (numberOfActiveStreams() > 0) {
    QPointer<KProgressDialog> m_dialog = new KProgressDialog(m_mainWidget,
      i18nc("@title:window", "Saving files..."),
      i18nc("@label", "Please wait while last files are saved..."));
    connect(this, SIGNAL(numberOfActiveStreamsIsZero()), m_dialog, SLOT(reject()));
    // min AND max = 0 means: show "busy" indicator instead of progress:
    m_dialog->progressBar()->setMinimum(0);
    m_dialog->progressBar()->setMaximum(0);
    m_dialog->setModal(true);
    m_dialog->setAutoClose(false);
    m_dialog->setMinimumDuration(0);
    // The following line hasn't any effect because of a bug in KProgressDialog.
    // Is fixed in KDE 4.2 (and maybe yet in 4.1.4).
    m_dialog->setButtonText(i18nc("@action:button", "Quit without saving"));
    m_dialog->exec();
    delete m_dialog;
  };

  return true;
}

void stationlistModel::saveProperties(KConfigGroup & m_configGroup)
{
  rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone();
  m_configGroup.writePathEntry("streamsToResume",
                                * m_listOfStreamsOfWhichTheUserWantsThatTheyRip);
  // No need to save the changes with sync() - KMainWindow and friends take care of this.
}

void stationlistModel::readProperties(const KConfigGroup & m_configGroup)
{
  // variables
  QStringList temp_stringList;
  int i;

  // code
  temp_stringList = m_configGroup.readPathEntry("streamsToResume", temp_stringList);
  for (i = 0; i < rowCount(); i++) {
    if (temp_stringList.contains(m_stationlist.at(i)->configFileName())) {
      m_stationlist.at(i)->startStreamripper();
    };
  };
}
