/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef RIPPING_H
#define RIPPING_H

#include <QPointer>
#include "streamripper_base.h"
#include "propertyvalue.h"

/** \brief This abstract class uses \e %streamripper to rip radio streams.
*
*   You can use the public slots <tt>virtual void startStreamripper()</tt>
*   and <tt>void shutDown()</tt> to start and stop \e %streamripper.
*
*   This class applies the options available through \e settings_general when
*   calling \e %streamripper (see <tt>virtual QStringList parameterList() const</tt>).
*
*   The actual state of the %ripping process is available through a number of properties.
*
*   You have to implement <tt>virtual QString serverUri() const</tt> in inherited classes.
*
*   Furthermore this class provides the static functions <tt>static QString
*   format_bitrate(long bitrate)</tt>  and <tt>static QString
*   format_metaInterval(long metaInterval)</tt>.
*
*   \warning This class uses signals and slots internally. So don't disconnect generally
*   all connections of this object! */
class ripping : public streamripper_base
{

     Q_OBJECT

     /** This property holds the bitrate of the stream in \e kbit/s
     * (with SI prefix, which means 1000 bit per second, <em>not 1024</em>) that
     * is recorded by \e %streamripper. (This information is based on the information that the
     * stream provides via the http header. This must not necessaryly be correct.)
     *
     * Meaning of the internal value:
     * \li <b> bitrate >= 1:</b> The bitrate has been recognized.
     * \   The property holds the bitrate.
     * \li <b> (bitrate = 0) OR (bitrate = -1):</b> An error occurred
     *     while recognizing the bitrate. (Maybe a \e %streamripper error
     *     or maybe a %KRadioRipper error while recognzing.)
     * \li <b> bitrate <= -2:</b> The property isn't defined. The bitrate
     *     has not (yet) been recognized.
     *
     * <b> Initialization value: </b> \e -2
     *
     * This property guards it's last value also when \e %streamripper terminates.
     *
     * Formated value:
     *   \li If the parameter "bitrate" is >= 1, this is a nicly
     *   format QString version of bitrate in the form "23 kbit/s".
     *   \li If the parameter is  0 or -1, then this is a message that there
     *   was an error while recognizing the bitrate.
     *   \li If the parameter is < -1, then this is an empty string.
     *
     *   All returned strings are fully localised: They are translated and use the
     *   locally used number format.
     *
     *
     * \li \e read: \code virtual PropertyValue bitrate() const \endcode
     * \li \e notify: \code void bitrateChanged(qlonglong index, PropertyValue newBitrate) \endcode
     *
     * \sa \code virtual void setBitrate(qint64 bitrate) \endcode
     * \sa \code static qint64 default_value_of_bitrate() \endcode
     * \sa \code static PropertyValue formatedBitrate(const qint64 theBitrate) \endcode
     * \sa \code PropertyValue internal_bitrate \endcode */
     Q_PROPERTY(PropertyValue bitrate READ bitrate)

     /** This property holds the amount of data of the actual song in <em>byte</em>,
     *   that has been recorded by \e %streamripper until now.
     *
     * \warning The formated version contains the value in MiB (with binary prefix which means
     * 1024 * 1024 byte, <em> not 1000 * 1000 byte</em>).
     *
     * Meaning:
     * \li <b> dataSize >= 0:</b> The dataSize has been recognized. The
     *     property holds the dataSize.
     * \li <b> dataSize = -1:</b> An error occurred while recognizing the
     *     dataSize. (Maybe a \e %streamripper error or maybe a %KRadioRipper
     *     error while recognzing.)
     * \li <b> dataSize <= -2:</b> The property isn't defined. The dataSize
     *     has not (yet) been recognized.
     *
     * <b> Initialization value: </b> \e -2
     *
     * This property gets reseted when \e %streamripper terminates.
     *
     * \li \e read: \code virtual PropertyValue dataSize() const \endcode
     * \li \e notify:
     *     \code void dataSizeChanged(qlonglong index, PropertyValue newDataSize) \endcode
     *
     * \sa \code virtual void setDataSize(qint64 dataSize) \endcode
     * \sa \code static qint64 default_value_of_dataSize() \endcode
     * \sa \code static PropertyValue formatedDataSize(const qint64 theDataSize) \endcode
     * \sa \code PropertyValue internal_dataSize \endcode */
     Q_PROPERTY(PropertyValue dataSize READ dataSize)

     /** This property holds the error message of the last error that has occurred.
     *   It holds error messages from \e %streamripper or an error messages from
     *   KRadioRipper (e.g. that the \e %streamripper command wasn't found or the
     *   "save directory" isn't writable).
     *
     *   Formated error message:
     *   - <i>If error doesn't start with <tt>"error -"</tt>:</i> The error message
     *     is left as is.
     *   - <i>If error starts with <tt>"error -"</tt>:</i> The error is expected to
     *     be of the format <tt>"error -123 foo bar"</tt>. \e 123 stands for the ID
     *     number of the bug.
     *     - <i>If the ID is known:</i> The formated error message is fully human
     *       readable and localised. The following bug description (here: <i>foo bar</i>)
     *       is ignored.
     *     - <i>If the bug ID in unknown:</i> A message of the form
     *       <tt>"Error 123: foo bar"</tt> is used.
     *   Error IDs < 1000 should be \e streamripper errors and error IDs >= 1000 should
     *   be \e KRadioRipper errors.
     *
     * Meaning:
     * \li <b>an empty string:</b> No error since the last startStreamripper().
     * \li <b>else:</b> An error occurred since the last startStreamripper and the
     *     property holds the error message.
     *
     * <b> Initialization value: </b> <EM> an empty string </EM>
     *
     * This property guards it's last value also when \e %streamripper terminates.
     *
     * This property gets reseted when you startStreamripper().
     *
     * \li \e read: \code virtual QString error() const \endcode
     * \li \e notify: \code void errorChanged(qlonglong index, PropertyValue newError) \endcode
     *
     * \sa \code virtual void setError(const QString newError) \endcode
     * \sa \code static QString default_value_of_error() \endcode
     * \sa \code static PropertyValue formatedError(const QString theError) \endcode
     * \sa \code PropertyValue internal_error \endcode */
     Q_PROPERTY(PropertyValue error READ error)

     /** This property holds the #status of \e %streamripper in a simplified way:
     *   \e false when the #status is ripping::idle, and \e true in all other
     *   cases.
     *
     * <b> Initialization value: </b> <EM> false </EM>
     *
     * <ul>
     * <li> \e read: \code virtual bool isRunning() const \endcode </li>
     * <li> \e notify: There are 2 signals to which you can connect:
     *      <ul>
     *           <li> \code void not_running() \endcode
     *                This signal is emitted when isRunning changes to \e false.
     *
     *                This means that this signal is emitted when the \e %streamripper
     *                process enters in the state <tt> QProcess::NotRunning</tt>.
     *
     *                This differs from <tt>QProcess:finished()</tt>, which is only
     *                emitted when the process was yet running. But <tt>not_running</tt>
     *                is also emitted after you've tried to start \e %streamripper
     *                (#m_process has yet entered in state <tt>QProcess::Starting</tt>)
     *                but the binary could than not be invoced (because is doesn't exit
     *                or the user hasn't execution rights or whatever).
     *           </li>
     *           <li> \code void running() \endcode
     *                This signal is emitted when isRunning changes to \e true.
     *
     *                This means that this signal is emitted when the \e %streamripper
     *                process has entered in the state <tt> QProcess::Starting</tt>
     *                or immediately before it will enter.
     *
     *                This differs from <tt>QProcess:started()</tt>, which is only
     *                emitted when the binary has been successfully invoced. But
     *                <tt>running</tt> is emitted after you start to try to invoce
     *                \e %streamripper - still not knowing if this will work or not.
     *           </li>
     *      </ul>
     * </li>
     * </ul>
     *
     * \sa #status
     * \sa \code virtual void setStatus(const statusType newStatus) \endcode
     * \sa \code static QString default_value_of_isRunning() \endcode
     * \sa \code bool internal_isRunning \endcode */
     Q_PROPERTY(bool isRunning READ isRunning)

     /** This property holds the interval in \e byte in which metadata is send by the stream.
     *
     * Meaning of the internal value:
     * \li <b> metaInterval >= 1:</b> The metaInterval has been recognized. The
     *     property holds the metaInterval.
     * \li <b> (metaInterval = 0) OR (metaInterval = -1):</b> An error occurred
     *     while recognizing the metaInterval. (Maybe a \e %streamripper error or
     *     maybe a %KRadioRipper error while recognzing.)
     * \li <b> metaInterval <= -2:</b> The property isn't defined. The metaInterval
     *     has not (yet) been recognized.
     *
     * <b> Initialization value: </b> \e -2
     *
     * This property guards it's last value also when \e %streamripper terminates.
     *
     * Formated value:
     *   \li If the parameter "metaInterval" is >= 1, this is a nicly
     *   format QString version of metaInterval.
     *   \li If the parameter is  0 or -1, then this is a message that there
     *   was an error while recognizing the meta data interval.
     *   \li If the parameter is < -1, then this is an empty string.
     *
     *   All returned strings are fully localised: They are translated and use the
     *   locally used number format.
     *
     * \li \e read: \code virtual PropertyValue metaInterval() const \endcode
     * \li \e notify:
     *     \code void metaIntervalChanged(qlonglong index, PropertyValue newMetaInterval) \endcode
     *
     * \sa \code virtual void setMetaInterval(qint64 newMetaInterval) \endcode
     * \sa \code static qint64 default_value_of_metaInterval() \endcode
     * \sa \code void resetMetaInterval() \endcode
     * \sa \code static PropertyValue formatedMetaInterval(const qint64 theMetaInterval) \endcode */
     Q_PROPERTY(PropertyValue metaInterval READ metaInterval)

     /** This property holds the interval in \e milliseconds in which metadata is send by the
     * stream. It doesn't exist on it's own, but is calculated from the properties #metaInterval
     * and #bitrate.
     *
     * Meaning of the internal value:
     * \li <b> metaInterval_milliSeconds >= 1:</b> The #bitrate and the #metaInterval have been
     *     recognized. The property holds the metaInterval in milliseconds.
     * \li <b> (metaInterval_milliSeconds = 0) OR (metaInterval_milliSeconds = -1):</b> An error
     *     occurred while recognizing the #bitrate or the #metaInterval. (Maybe a \e %streamripper
     *     error or maybe a %KRadioRipper error while recognzing.)
     * \li <b> metaInterval_milliSeconds <= -2:</b> The property isn't defined. The #bitrate or the
     *     #metaInterval has not (yet) been recognized.
     *
     * <b> Initialization value: </b> <i>depending on #bitrate and #metaInterval</i>
     *
     * This property guards it's last value also when \e %streamripper terminates, as it depends on
     * #bitrate and #metaInterval which guards it's values both.
     *
     * Formated value:
     *   \li If the parameter "metaInterval" is >= 1, this is a nicly
     *   format QString version of metaInterval.
     *   \li If the parameter is  0 or -1, then this is a message that there
     *   was an error while recognizing the meta data interval.
     *   \li If the parameter is < -1, then this is an empty string.
     *
     *   All returned strings are fully localised: They are translated and use the
     *   locally used number format.
     *
     * \li \e read: \code virtual PropertyValue metaInterval_milliSeconds() const \endcode
     * \li \e notify: \code void metaInterval_milliSecondsChanged(
     *   qlonglong index,
     *   PropertyValue newMetaInterval_milliSeconds) \endcode
     *
     * \sa \code static PropertyValue formatedMetaInterval_milliSeconds(
     *   const qint64 theMetaInterval_milliSeconds) \endcode
     * \sa The constructor: There, the emitting of the notify signal is organized. */
     Q_PROPERTY(PropertyValue metaInterval_milliSeconds READ metaInterval_milliSeconds STORED false)

     /** \e %streamripper can start a local relay server which provides
     *   the stream that is actually recorded with some seconds of delay
     *   to the local network. This relay server is also used by KRadioRipper
     *   to implement to hear to the streams. This property holds the port
     *   of the relay server - but only if #status is #is_skipping or
     *   #is_ripping.
     *
     * Meaning:
     * \li <b> relayPort >= 0:</b> There exists a relay server. The property
     *     holds the port of the relay server.
     * \li <b> relayPort = -1:</b> An error occurred while recognizing the
     *     relay server port. (Maybe a \e %streamripper error or maybe a
     *     %KRadioRipper error while recognzing.)
     * \li <b> relayPort <= -2:</b> There doesn't exist a relay server.
     *
     * <b> Initialization value: </b> \e -2
     *
     * This property gets reseted when \e %streamripper terminates.
     *
     * \li \e read: \code virtual PropertyValue relayPort() const \endcode
     * \li \e notify:
     *     \code void relayPortChanged(qlonglong index, PropertyValue newRelayPort) \endcode
     *
     * \sa \code virtual void setRelayPort(qint64 newRelayPort) \endcode
     * \sa \code static qint64 default_value_of_relayPort() \endcode
     * \sa \code static PropertyValue formatedRelayPort(const qint64 theRelayPort) \endcode
     * \sa \code PropertyValue internal_relayPort \endcode
     * \sa #lastRecognizedRelayPort()
     * \sa #refreshRelayPort() */
     Q_PROPERTY(PropertyValue relayPort READ relayPort)

     /** This property holds the name of server from which the stream comes.
     *
     * Meaning:
     * \li <b>an empty string:</b> The name of the server hasn't (yet) been recognized.
     * \li <b>else:</b> The name of the server has been recognized and the property holds it.
     *
     * <b> Initialization value: </b> <EM> an empty string </EM>
     *
     * This property guards it's last value also when \e %streamripper terminates.
     *
     * \li \e read: \code virtual PropertyValue serverName() const \endcode
     * \li \e notify:
     *     \code void serverNameChanged(qlonglong index, PropertyValue newServerName) \endcode
     *
     * \sa \code virtual void settod_pudel_1920x1200ServerName(const QString newServerName) \endcode
     * \sa \code static QString default_value_of_serverName() \endcode
     * \sa \code static PropertyValue formatedServerName(const QString theServerName) \endcode
     * \sa \code PropertyValue internal_serverName \endcode */
     Q_PROPERTY(PropertyValue serverName READ serverName)

     /** This property holds the name of the song that is actually recorded by \e %streamripper.
     *
     * Meaning:
     * \li <b>an empty string:</b> Actually \e %streamripper isn't recordeding.
     * \li <b>else:</b> Actually \e %streamripper is recordeding and the property
     *     holds the name of the song who's actually recorded.
     *
     * <b> Initialization value: </b> <EM> an empty string </EM>
     *
     * \li \e read: \code virtual PropertyValue song() const \endcode
     * \li \e notify: \code void songChanged(qlonglong index, PropertyValue newSong) \endcode
     *
     * \sa \code virtual void setSong(const QString song) \endcode
     * \sa \code static QString default_value_of_song() \endcode
     * \sa \code static PropertyValue formatedSong(const QString theSong) \endcode
     * \sa \code PropertyValue internal_song \endcode */
     Q_PROPERTY(PropertyValue song READ song)

     /** This property holds the status of \e %streamripper (like "Buffering...", "Ripping...",
     *   "Shutting down..." ...).
     *
     * <b> Initialization value: </b> <EM> ripping::idle </EM>
     *
     * \li \e read: \code virtual PropertyValue status() const \endcode
     * \li \e notify: \code void statusChanged(qlonglong index, PropertyValue newStatus); \endcode
     *
     * \sa #isRunning
     * \sa \code bool doesTheUserWantsThatTheStreamIsRipping() \endcode
     * \sa \code virtual void setStatus(const statusType newStatus) \endcode
     * \sa \code static statusType default_value_of_status() \endcode
     * \sa \code static PropertyValue formatedStatus(const statusType theStatus) \endcode
     * \sa \code PropertyValue internal_status \endcode */
     Q_PROPERTY(PropertyValue status READ status)

     /** This property holds the name of the stream that is recorded by \e %streamripper.
     *
     * Meaning:
     * \li <b>an empty string:</b> The name of the stream hasn't (yet) been recognized.
     * \li <b>else:</b> The name of the stream has been recognized and the property holds it.
     *
     * <b> Initialization value: </b> <EM> an empty string </EM>
     *
     * This property guards it's last value also when \e %streamripper terminates.
     *
     * \li \e read: \code virtual PropertyValue streamName() const \endcode
     * \li \e notify:
     *     \code void streamNameChanged(qlonglong index, PropertyValue newStreamName) \endcode
     *
     * \sa \code virtual void setStreamName(const QString newStreamName) \endcode
     * \sa \code static QString default_value_of_streamName() \endcode
     * \sa \code static PropertyValue formatedStreamName(const QString theStreamName) \endcode
     * \sa \code PropertyValue internal_streamName \endcode */
     Q_PROPERTY(PropertyValue streamName READ streamName)

  public:
     // typedefs
     /** This enum can hold the actual #status of %streamripper.
     *   <ul>
     *   <li>\e idle: #m_process is not running.</li>
     *   <li>\e is_starting: <ol>
     *                       <li>#m_process will be started immediately</li>
     *                       <li>#m_process is <tt>QProcess::Starting</tt>.</li>
     *                       <li>#m_process is <tt>QProcess::Running</tt>, but
     *                           %streamripper still hasn't reached the state
     *                           <tt>ripping::is_connecting</tt>.</li>
     *                       </ol>
     *   </li>
     *   <li>\e is_connecting: %streamripper connects to the server.</li>
     *   <li>\e is_buffering: %streamripper is filling his buffer.</li>
     *   <li>\e is_skipping: %streamripper doesn't record the actual song (in splitted files).</li>
     *   <li>\e is_ripping: %streamripper is %ripping the actual song.</li>
     *   <li>\e is_saving: %streamripper is shutting down. This takes usually
     *                     some time, but only "shutting down" is an annoying message.
     *                     The user will find it more useful if the status message
     *                     describes something important. So it's called "saving files".
     *                     This sounds important and also is true.</li>
     *   </ul> */
     enum statusType {
       idle,
       is_starting,
       is_connecting,
       is_buffering,
       is_skipping,
       is_ripping,
       is_saving };

     // contructor(s) and destructor(s)
     /** Constructor of the class.
     * @param parent Sets the parent of this object. */
     ripping(QObject *parent);
     /** The destructor. */
     virtual ~ripping();

     // properties
     /** See property #bitrate. */
     virtual PropertyValue bitrate() const;
     /** See property #dataSize. */
     virtual PropertyValue dataSize() const;
     /** Performs doesTheUserWantsThatTheStreamIsRipping(const ripping::statusType theStatus)
     * with the #status of this object - so it provides a sort of view
     * to the propyerty #status. */
     bool doesTheUserWantsThatTheStreamIsRipping();
     /** See property #error. */
     virtual PropertyValue error() const;
     /** See property #isRunning. */
     virtual bool isRunning() const;
     /** See property #metaInterval. */
     virtual PropertyValue metaInterval() const;
     /** See property #metaInterval_milliSeconds. */
     virtual PropertyValue metaInterval_milliSeconds() const;
     /** See property #relayPort. */
     virtual PropertyValue relayPort() const;
     /** See property #serverName. */
     virtual PropertyValue serverName() const;
     /** See property #song. */
     virtual PropertyValue song() const;
     /** See property #status. */
     virtual PropertyValue status() const;
     /** See property #streamName. */
     virtual PropertyValue streamName() const;

     // static functions
     /** \returns The default value of property #bitrate. */
     static qint64 default_value_of_bitrate();
     /** \returns The default value of property #dataSize. */
     static qint64 default_value_of_dataSize();
     /** \returns The default value of property #error. */
     static QString default_value_of_error();
     /** \returns The default value of property #isRunning. */
     static bool default_value_of_isRunning();
     /** \returns The default value of property #metaInterval. */
     static qint64 default_value_of_metaInterval();
     /** \returns The default value of property #relayPort. */
     static qint64 default_value_of_relayPort();
     /** \returns The default value of property #serverName. */
     static QString default_value_of_serverName();
     /** \returns The default value of property #song. */
     static QString default_value_of_song();
     /** \returns The default value of property #status. */
     static statusType default_value_of_status();
     /** \returns The default value of property #streamName. */
     static QString default_value_of_streamName();
     /** \param theStatus the status for which you want to determinate if the
     *                    user wants to rip the stream or not.
     *   \returns <ul><li>\e false if the user wants that the stream is \e not
     *                    %ripping. This is assumed when the stream isn't running,
     *                    (ripping::idle) or when the stream is running but the user
     *                    has clicked on stop so that the stream is shutting down
     *                    (ripping::is_saving).
     *                </li>
     *                <li>\e true in all other cases.
     *                </li>
     *            </ul>
     *   \sa doesTheUserWantsThatTheStreamIsRipping() */
     static bool doesTheUserWantsThatTheStreamIsRipping(const statusType theStatus);
     /** \returns A full-featured formated version for the property #bitrate. */
     static PropertyValue formatedBitrate(const qint64 theBitrate);
     /** \returns A full-featured formateded version for the property #dataSize. */
     static PropertyValue formatedDataSize(const qint64 theDataSize);
     /** \returns A full-featured formateded version for the property #error. */
     static PropertyValue formatedError(const QString & theError);
     /** \returns A full-featured formateded version for the property #metaInterval. */
     static PropertyValue formatedMetaInterval(const qint64 theMetaInterval);
     /** \returns A full-featured formateded version for the property
     * #metaInterval_milliSeconds. */
     static PropertyValue formatedMetaInterval_milliSeconds(
       const qint64 theMetaInterval_milliSeconds);
     /** \returns A full-featured formateded version for the property #relayPort. */
     static PropertyValue formatedRelayPort(const qint64 theRelayPort);
     /** \returns A full-featured formateded version for the property #serverName. */
     static PropertyValue formatedServerName(const QString & theServerName);
     /** \returns A full-featured formateded version for the property #song. */
     static PropertyValue formatedSong(const QString & theSong);
     /** \returns A full-featured formateded version for the property #status. */
     static PropertyValue formatedStatus(const statusType theStatus);
     /** \returns A full-featured formateded version for the property #streamName. */
     static PropertyValue formatedStreamName(const QString & theStreamName);

  signals:
     // properties
     /** See property #bitrate. */
     void bitrateChanged(qlonglong index, PropertyValue newBitrate);
     /** See property #dataSize. */
     void dataSizeChanged(qlonglong index, PropertyValue newDataSize);
     /** See property #error. */
     void errorChanged(qlonglong index, PropertyValue newError);
     /** See property #metaInterval. */
     void metaIntervalChanged(qlonglong index, PropertyValue newMetaInterval);
     /** See property #metaInterval_milliSeconds. */
     void metaInterval_milliSecondsChanged(qlonglong index,
                                           PropertyValue newMetaInterval_milliSeconds);
     /** See property #isRunning. */
     void not_running();
     /** See property #relayPort. */
     void relayPortChanged(qlonglong index, PropertyValue newRelayPort);
     /** See property #isRunning. */
     void running();
     /** See property #serverName. */
     void serverNameChanged(qlonglong index, PropertyValue newServerName);
     /** See property #song. */
     void songChanged(qlonglong index, PropertyValue newSong);
     /** See property #status. */
     void statusChanged(qlonglong index, PropertyValue newStatus);
     /** See property #streamName. */
     void streamNameChanged(qlonglong index, PropertyValue newStreamName);

  public slots:
     /** This slot sends the \e terminate signal to the \e %streamripper instance
     * running in #m_process.
     * \sa <tt>void not_running()</tt> */
     void shutDown();
     /** This function starts \e %streamripper (using streamripper_base::startStreamripper())
     *   if it isn't yet running.
     *   It uses #workingDirectory() as working directory and if this directory doesn't exist,
     *   it is created automatically (if possible).
     *   It resets the property #error and actualizes the property #status.
     *   \sa <tt>void not_running()</tt>*/
     virtual void startStreamripper();

  protected:
     /** This function reads the output of the \e %streamripper process and interpretates
     *   it. It keeps the property of this class up to date - so when you want to use
     *   this information, use the properties.
     *
     *   Internally for performance reasons it uses from the many lines it can receive
     *   only the \e last available information of each type. (For example: When it
     *   receives 15 lines <em>"[Ripping] song title [data size]"</em> it uses only the
     *   last one to set #status, #song and #dataSize properties - but controls
     *   all the lines before, to see if they contain data of other type like #bitrate
     *   information or so on, and if so, it uses it.) */
     virtual void interpretate_console_output(QStringList & stringList);
     /** This function provides the necessary parameters for calling \e %streamripper
     *   as available from serverUri() and settings_general - and some extra parameter
     *   (like to don't override temporary files). */
     virtual QStringList parameterList() const;
     /** This function resets all \e streamripper properties to it's default values (=unset).
     *   It uses the reset functions of the properties, so for all changes a notify signal
     *   will be emitted. */
     virtual void resetStreamripperProperties();
     /** This function provides the URI which should be ripped and is used by
     *   parameterList(). Implement it in inherited classes! */
     virtual QString serverUri() const = 0;
     /** Used internally to write the property #bitrate. */
     virtual void setBitrate(const qint64 newBitrate);
     /** Used internally to write the property #dataSize. */
     virtual void setDataSize(const qint64 newDataSize);
     /** Used internally to write the property #error. */
     virtual void setError(const QString & newError);
     /** Used internally to write the property #metaInterval. */
     virtual void setMetaInterval(const qint64 newMetaInterval);
     /** Used internally to write the property #relayPort. */
     virtual void setRelayPort(const qint64 newRelayPort);
     /** Used internally to write the property #serverName. */
     virtual void setServerName(const QString & newServerName);
     /** Used internally to write the property #song. */
     virtual void setSong(const QString & newSong);
     /** Used internally to write the properties #status and also #isRunning. */
     virtual void setStatus(const statusType newStatus);
     /** Used internally to write the property #streamName. */
     virtual void setStreamName(const QString & newStreamName);
     /** This function provides the necessary command for invoking \e %streamripper
     *   as available from settings_general::streamripperCommand. */
     virtual QString streamripperCommand() const;
     /** Provides the working directory for the Streamripper process. */
     virtual QString workingDirectory() const = 0;

  private:
     // help methods
     /** This internal helper function used in interpretate_console_output()
     *   interpretates a specific part of \e %streamrippers output and sets the properties
     *   #song and #dataSize accordingly.
     *   @param my_line A string of the form <em>song title [    1kb]</em>. The units
     *   <em>B</em>, <em>KB</em>, <em>K</em>, <em>MB</em> and <em>M</em> are recognized
     *   for the data size (not case sensitive). */
     void helper_interpretate_metainfo_and_datasize(QString my_line);

     //properties
     /** Used internally to store the property #bitrate. */
     PropertyValue internal_bitrate;
     /** Used internally to store the property #dataSize. */
     PropertyValue internal_dataSize;
     /** Used internally to store the property #error. */
     PropertyValue internal_error;
     /** Used internally to store the property #isRunning. */
     bool internal_isRunning;
     /** Used internally to store the property #metaInterval. */
     PropertyValue internal_metaInterval;
     /** Used internally to store the property #relayPort. */
     PropertyValue internal_relayPort;
     /** Used internally to store the property #serverName. */
     PropertyValue internal_serverName;
     /** Used internally to store the property #song. */
     PropertyValue internal_song;
     /** Used internally to store the property #status. */
     PropertyValue internal_status;
     /** Used internally to store the property #streamName. */
     PropertyValue internal_streamName;

     /** Here, the last recognized relay port is cached. This is necessary, because
     * the property #relayPort holds its value only when the #status is #is_skipping
     * or #is_ripping.
     * \sa #refreshRelayPort() */
     qint64 lastRecognizedRelayPort;

  private slots:
     /** This function emits the signal #metaInterval_milliSecondsChanged() */
     void emit_metaInterval_milliSecondsChanged();
     /** This slot is to catch the signal <tt> QProcess::error(QProcess::ProcessError
     *   error) </tt> from #m_process.
     *
     *   If the error is...
     *   \li <b><tt> QProcess::FailedToStart</tt>:</b> Make a test if either the
     *       invoked program is missing or you may have insufficient permissions
     *       to invoke the program. An appropriate error messessage is set in the
     *       property #error.
     *   \li <b><tt> QProcess::Crashed</tt>:</b> An appropriate error messessage
     *       is set in the property #error.
     *   \li All other errors can safly be ignored because they don't mean that
     *       \e %streamripper is terminated.
     *
     *   @param error The <tt> QProcess::ProcessError </tt> that has occurred. */
     void errorOccured(const QProcess::ProcessError error);
     /** Recalculates #internal_relayPort and emits the necessary signals.
     * \sa #lastRecognizedRelayPort
     * \sa #setError()
     * \sa #setStatus
     * \sa #setRelayPort() */
     void refreshRelayPort();
     /** This slot is to catch the signal
     *   <tt>QProcess::stateChanged(QProcess::ProcessState)</tt> from #m_process.
     *
     *   \li When the state has changed to <tt> QProcess::NotRunning </tt> than
     *       the properties get adjusted.
     *   \li When the state has changed to <tt> QProcess::Starting </tt> nothing
     *       happens because the properties get adjusted by the calling function.
     *   \li When the state has changed to <tt> QProcess::Running </tt> nothing
     *       happens because the properties will get adjusted by
     *       interpretate_console_output().
     *
     *   @param newState The new <tt> QProcess::ProcessState </tt> that has occurred. */
     void streamripperStateChange(const QProcess::ProcessState newState);
 };

Q_DECLARE_METATYPE(ripping::statusType)

#endif
