/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "stationlistwidget.h"

#include <QTimer>
#include <QContextMenuEvent>
#include <KLocale>
#include "settings_general.h"
#define AND  &&
#define OR  ||
#define NOT  !
#define EQUAL  ==

stationlistWidget::stationlistWidget(QWidget *parent, QWidget *mainWindow)
  : QTableView(parent)
{
  // variables
  QPointer<CustomizableHeaderView> theHorizontalHeader;
  int i;
  Qt::SortOrder order;

  // code
  theHorizontalHeader = new CustomizableHeaderView(Qt::Horizontal);
  // this also deletes automatically the previously used (standard) QHeaderView object:
  setHorizontalHeader(theHorizontalHeader);
  theHorizontalHeader->setContextMenuTitle(
    i18nc("@title:menu Title of the context menu where you can change visibility of columns",
          "Columns"));
  if (settings_general::sortingAscendingly()) {
    order = Qt::AscendingOrder;
  } else {
    order = Qt::DescendingOrder;
  };
  theHorizontalHeader->setSortIndicator(settings_general::sortByColumn(), order);
  theHorizontalHeader->setSortIndicatorShown(true);
  theHorizontalHeader->setClickable(true);
  /* We have to save the width of a column _befor_ it is hidden (after hiding,
  *  we wouln't be able to get the width anymore). */
  connect(theHorizontalHeader,
          SIGNAL(sectionAboutToBeHidden(int, Qt::Orientation)),
          this,
          SLOT(saveColumnSize(int)));

  setSelectionBehavior(QAbstractItemView::SelectRows);  // don't change this.
  // Much code relys on the assumption that only hole rows can be selected.

  setShowGrid(false);

  m_stationlistModel = new stationlistModel(this, mainWindow);
  setModel(m_stationlistModel);

  // restore column width
  for (i=0; i < m_stationlistModel->columnCount(); ++i) {
    setColumnHidden(
      i,
      !m_stationlistModel->columnInfo(stationlistModel::columnVisibility, i).toBool());
    setColumnWidth(
      i,
      m_stationlistModel->columnInfo(stationlistModel::columnWidth, i).toLongLong());
  };

  // various settings
  setVerticalScrollMode(QAbstractItemView::ScrollPerPixel);
  setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
  setDragDropMode(QAbstractItemView::DropOnly);
  setDragDropOverwriteMode(false); // don't override items, but insert new ones!
  setAcceptDrops(true);
  setDropIndicatorShown(false);
  setSortingEnabled(true);
}

stationlistWidget::~stationlistWidget()
{
}

void stationlistWidget::saveAllColumnSizes()
{
  // variables
  int i;

  // code
  for (i=0; i < m_stationlistModel->columnCount(); i++) {
    if (NOT isColumnHidden(i)) {
      saveColumnSize(i);
    };
    m_stationlistModel->columnInfo(stationlistModel::setColumnVisibility,
                                    i,
                                    -1,
                                    NOT isColumnHidden(i));
  };
}

void stationlistWidget::saveColumnSize(const int column)
{
  m_stationlistModel->columnInfo(stationlistModel::setColumnWidth,
                                  column,
                                  -1,
                                  columnWidth(column));
}

void stationlistWidget::record()
{
  // variables
  int i;
  // initialized with a list of the selected rows (for column 0):
  QModelIndexList m_list = selectionModel()->selectedRows();

  // code
  for (i = 0; i < m_list.size(); ++i) {
    m_stationlistModel->record(m_list.at(i).row());
  };
}

void stationlistWidget::stopRecord()
{
  // variables
  int i;
  // initialized with a list of the selected rows (for column 0):
  QModelIndexList m_list = selectionModel()->selectedRows();

  // code
  for (i = 0; i < m_list.size(); ++i) {
    m_stationlistModel->stopRecording(m_list.at(i).row());
  };
}

void stationlistWidget::selectionChanged(const QItemSelection & selected,
                                          const QItemSelection & deselected)
{
  // call the original function from the base class
  QTableView::selectionChanged(selected, deselected);

  // code
  switch (selectionModel()->selectedRows().size()) {
    case 0:
      emit noneSelected(true);
      break;
    case 1:
      emit oneSelected(true);
      break;
    default:
      emit multipleSelected(true);
      break;
  };
}

void stationlistWidget::displayStreamSettings()
{
  // variables
  // initialized with a list of the selected rows (for column 0):
  QModelIndexList m_list = selectionModel()->selectedRows();

  // code
  if (m_list.size() == 1) {
    m_stationlistModel->showConfigDialog(m_list.at(0).row());
  };
}

void stationlistWidget::deleteStation()
{
  // variables
  QModelIndexList m_list;

  // code
  // initialized with a list of the selected rows (for column 0):
  m_list = selectionModel()->selectedRows();
  while (m_list.size() > 0) {
    m_stationlistModel->removeRow(m_list.at(0).row());
    m_list = selectionModel()->selectedRows();
  };
}

void stationlistWidget::addNewStation()
{
  // variables
  QModelIndex index_of_new_station;

  // code
  index_of_new_station = m_stationlistModel->addNewStation();
  if (index_of_new_station.isValid()) {
    clearSelection();
    selectionModel()->select(index_of_new_station,
                             QItemSelectionModel::SelectCurrent | QItemSelectionModel::Rows);
  };
}

void stationlistWidget::contextMenuEvent(QContextMenuEvent * e)
{
  if (indexAt(e->pos()).isValid()) {
    streamContextMenu.exec(e->globalPos());
  } else {
    clearSelection();
    globalContextMenu.exec(e->globalPos());
  };
}

QPointer<stationlistModel> stationlistWidget::stationlistmodel()
{
  return m_stationlistModel;
}

bool stationlistWidget::queryClose()
{
  saveAllColumnSizes();
  return stationlistmodel()->queryClose();
}

void stationlistWidget::saveProperties(KConfigGroup & m_configGroup)
{
  saveAllColumnSizes();
  stationlistmodel()->saveProperties(m_configGroup);
}

void stationlistWidget::readProperties(const KConfigGroup & m_configGroup)
{
  stationlistmodel()->readProperties(m_configGroup);
}

void stationlistWidget::mousePressEvent(QMouseEvent *event)
{
  if (event->button() == Qt::MidButton) {
    stationlistmodel()->pasteSelection();
  } else {
    // pass on other buttons to base class
    QTableView::mousePressEvent(event);
  };
}
