/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "Contraction.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

typedef KGenericFactory<KritaContraction> KritaContractionFactory;
K_EXPORT_COMPONENT_FACTORY( kritaContraction, KritaContractionFactory( "krita" ) )

KritaContraction::KritaContraction(QObject *parent, const char *name, const QStringList &)
: KParts::Plugin(parent, name)
{
    setInstance(KritaContractionFactory::instance());

    kdDebug(41006) << "Contraction filter plugin. Class: "
    << className()
    << ", Parent: "
    << parent -> className()
    << "\n";

    if(parent->inherits("KisFilterRegistry"))
    {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisContractionFilter());
    }
}

KritaContraction::~KritaContraction()
{
}

KisContractionFilter::KisContractionFilter() 
    : KisFilter(id(), "Contraction", i18n("&Contraction"))
{
}

void KisContractionFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);
    
    KisHLineIteratorPixel dstIt = dst->createHLineIterator(rect.x(), rect.y(), rect.width(), true );
    KisHLineIteratorPixel srcIt = src->createHLineIterator(rect.x(), rect.y(), rect.width(), false);

    int pixelsProcessed = 0;
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();
    Q_INT32 psize = cs->pixelSize();
    const Q_UINT8 *colors[9];
    Q_UINT8 weights[9];
    
    for(int y = 0; y < rect.height(); y++)
    {
        while( not srcIt.isDone() )
        {
            if(srcIt.isSelected())
            {
                const Q_UINT8* oldData = srcIt.oldRawData();
                if( cs->getAlpha(oldData) != 0 )
                {
                    KisRectIteratorPixel neighIt = src->createRectIterator( srcIt.x() - 1, srcIt.y() - 1, 3, 3, false);
                    int minAlpha = 255;
                    while( not neighIt.isDone())
                    {
                        int alpha = cs->getAlpha( neighIt.oldRawData() );
                        if(alpha < minAlpha) minAlpha = alpha;
                        ++neighIt;
                    }
                    cs->setAlpha( dstIt.rawData(), minAlpha, 1);
                }
            }
            setProgress(++pixelsProcessed);
            ++srcIt;
            ++dstIt;
        }
        srcIt.nextRow();
        dstIt.nextRow();
    }

    setProgressDone(); // Must be called even if you don't really support progression
}
