/***************************************************************************
 *
 * Copyright (C) 2005 Elad Lahav (elad_lahav@users.sourceforge.net)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

#include <qfile.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <klineedit.h>
#include <kinputdialog.h>
#include <kxmlguifactory.h>
#include <kstatusbar.h>
#include <kaction.h>
#include <kurldrag.h>
#include <kkeydialog.h>
#include "kscope.h"
#include "kscopeconfig.h"
#include "projectmanager.h"
#include "editortabs.h"
#include "fileview.h"
#include "filelist.h"
#include "querywidget.h"
#include "editormanager.h"
#include "cscopefrontend.h"
#include "ctagslist.h"
#include "newprojectdlg.h"
#include "openprojectdlg.h"
#include "preferencesdlg.h"
#include "dirscanner.h"
#include "querypage.h"
#include "calltreedlg.h"
#include "calltreemanager.h"
#include "kscopepixmaps.h"
#include "progressdlg.h"
#include "projectfilesdlg.h"
#include "cscopemsgdlg.h"
#include "symboldlg.h"
#include "symbolcompletion.h"
#include "queryviewdlg.h"
#include "graphwidget.h"
#include "welcomedlg.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
KScope::KScope(QWidget* pParent, const char* szName) :
	KParts::DockMainWindow(pParent, szName),
	m_pCscopeBuild(NULL),	
	m_sCurFilePath(""),
	m_nCurLine(0),
	m_pProgressDlg(NULL),
	m_bUpdateGUI(true)
{
	QString sPath;

	// Load configuration
	Config().load();
	
	// Create the main child widgets
	m_pEditTabs = new EditorTabs(this, NULL);
	m_pQueryWidget = new QueryWidget(this);
	m_pFileView = new FileView(this);
	m_pFileList = m_pFileView->getFileList();
	m_pMsgDlg = new CscopeMsgDlg(this);
	m_pQueryDock = createDockWidget("Query Window", QPixmap());
	m_pFileViewDock = createDockWidget("File List Window", QPixmap());

	// Connect menu and toolbar items with the object's slots
	initActions();
	
	// Create all child widgets
	initMainWindow();

	// Create control objects
	m_pProjMgr = new ProjectManager();
	m_pEditMgr = new EditorManager(this);
	m_pCallTreeMgr = new CallTreeManager(this);

	// Initialise the icon manager	
	Pixmaps().init();
	
	// Open a file for editing when selected in the project's file list or the
	// file tree
	connect(m_pFileView, SIGNAL(fileRequested(const QString&, uint)), this,
		SLOT(slotShowEditor(const QString&, uint)));

	// Delete an editor page object after it is removed
	connect(m_pEditTabs, SIGNAL(editorRemoved(EditorPage*)),
		this, SLOT(slotDeleteEditor(EditorPage*)));
	
	connect(m_pEditTabs, SIGNAL(filesDropped(QDropEvent*)), this,
		SLOT(slotDropEvent(QDropEvent*)));
	
	// Set an editor as the active part whenever its owner tab is selected
	connect(m_pEditTabs, SIGNAL(editorChanged(EditorPage*, EditorPage*)),
		this, SLOT(slotChangeEditor(EditorPage*, EditorPage*)));

	// Display a file at a specific line when selected in a query list
	connect(m_pQueryWidget, SIGNAL(lineRequested(const QString&, uint)),
		this, SLOT(slotQueryShowEditor(const QString&, uint)));
	
	// Display the symbol dialogue when the user opens a new query page
	connect(m_pQueryWidget, SIGNAL(newQuery()), 
		this, SLOT(slotQueryReference()));
	
	// Refresh the file list if files were added to or removed from a project
	connect(m_pProjMgr, SIGNAL(fileListChanged()), this,
		SLOT(slotProjectFilesChanged()));

	// Add files to the file list when they are added to the project
	connect(m_pProjMgr, SIGNAL(filesAdded(const QStringList&)), this,
		SLOT(slotFilesAdded(const QStringList&)));

	// Store the new file tree root of a project
	connect(m_pFileView, SIGNAL(rootChanged(const QString&)), this,
		SLOT(slotProjectRootChanged(const QString&)));
	
	// Launch a "Find File" query when the "Find File..." button of the
	// file tree is clicked
	connect(m_pFileView, SIGNAL(findFile()), this, 
		SLOT(slotQueryFile()));

	// Rebuild the project database after a certain time period has elapsed
	// since the last save
	connect(&m_timerRebuild, SIGNAL(timeout()), this, SLOT(slotRebuildDB()));

	// Display a file at a specific line when selected in a call tree dialogue
	connect(m_pCallTreeMgr, SIGNAL(lineRequested(const QString&, uint)),
		this, SLOT(slotQueryShowEditor(const QString&, uint)));
		
	// Store main window settings when closed
	setAutoSaveSettings();
	
	// Initialise arrow head drawing
	GraphWidget::setArrowInfo(20, 15);
	
	// Use a maximised window the first time
	if (Config().isFirstTime())
		showMaximized();

	// Show the Welcome message
	if (Config().showWelcomeDlg()) {
		show();
		slotShowWelcome();
	}

	// If this is the first time the user has launched KScope, prompt him/her
	// to configure the global parameters
	if (Config().isFirstTime())
		slotConfigure();		
}

/**
 * Class destructor.
 */
KScope::~KScope()
{
	// Save configuration
	Config().store();
	Config().storeWorkspace(this);
	
	delete m_pCallTreeMgr;
	delete m_pEditMgr;
	delete m_pCscopeBuild;
	delete m_pProjMgr;
}

/**
 * Connects menu bar and toolbar commands with class members. These members
 * handle the actions associated with each command.
 */
void KScope::initActions()
{
	KToggleAction* pLockAction;
	
	// File menu
	KStdAction::openNew(this, SLOT(slotNewFile()), actionCollection());
	KStdAction::open(this, SLOT(slotOpenFile()), actionCollection());
	KStdAction::close(this, SLOT(slotCloseEditor()), actionCollection());
	KStdAction::quit(this, SLOT(slotClose()), actionCollection());

	(void)new KAction(i18n("Go to File List"), KShortcut("Ctrl+Shift+O"),
		m_pFileList, SLOT(slotSetFocus()), actionCollection(),
		"file_open_file_from_list");
	(void)new KAction(i18n("Save Al&l"), 
		"save_all",	KShortcut("Ctrl+L"), m_pEditTabs, SLOT(slotSaveAll()),
		actionCollection(), "file_save_all");
	
	// Edit menu
	m_pExtEditAction = new KAction(i18n("Edit in E&xternal Editor"),
		KShortcut("Ctrl+E"), this,
		SLOT(slotExtEdit()), actionCollection(), "edit_external_editor");
	(void)new KAction(i18n("Complete Symbol"), KShortcut("Ctrl+Space"), this,
		SLOT(slotCompleteSymbol()), actionCollection(), "edit_comp_symbol");
		
	// Project menu
	(void)new KAction(i18n("&New Project..."), 0, this, 
		SLOT(slotCreateProject()), actionCollection(), "project_new");
	(void)new KAction(i18n("&Open Project..."), "project_open", 0, this,
		SLOT(slotOpenProject()), actionCollection(), "project_open");
	(void)new KAction(i18n("Add/Remove &Files..."), 0, this,
		SLOT(slotProjectFiles()), actionCollection(),
		"project_add_rem_files");
	(void)new KAction(i18n("&Properties..."), 0, this,
		SLOT(slotProjectProps()), actionCollection(),
		"project_properties");
	(void)new KAction(i18n("Open &Cscope.out..."), 0, this,
		SLOT(slotProjectCscopeOut()), actionCollection(),
		"project_cscope_out");
	(void)new KAction(i18n("&Close Project"), 0, this,
		SLOT(slotCloseProject()), actionCollection(), "project_close");

	// Cscope menu
	(void)new KAction(i18n("&References..."), KShortcut("Ctrl+0"), this,
		SLOT(slotQueryReference()), actionCollection(), "cscope_references");
	(void)new KAction(i18n("&Definition..."), 0, KShortcut("Ctrl+1"), this,
		SLOT(slotQueryDefinition()), actionCollection(), "cscope_definition");
	(void)new KAction(i18n("&Called Functions..."), 0, KShortcut("Ctrl+2"),
		this, SLOT(slotQueryCalled()), actionCollection(), "cscope_called");
	(void)new KAction(i18n("C&alling Functions..."), 0, KShortcut("Ctrl+3"),
		this, SLOT(slotQueryCalling()),	actionCollection(), "cscope_calling");
	(void)new KAction(i18n("Find &Text..."), 0, KShortcut("Ctrl+4"), this,
		SLOT(slotQueryText()), actionCollection(), "cscope_text");
	(void)new KAction(i18n("Find &EGrep Pattern..."), 0, KShortcut("Ctrl+5"),
		this, SLOT(slotQueryPattern()), actionCollection(), "cscope_pattern");
	(void)new KAction(i18n("Find &File..."), 0, KShortcut("Ctrl+7"), this,
		SLOT(slotQueryFile()), actionCollection(), "cscope_file");
	(void)new KAction(i18n("&Including Files..."), 0, KShortcut("Ctrl+8"),
		this, SLOT(slotQueryIncluding()), actionCollection(),
		"cscope_including");
	(void)new KAction(i18n("&Quick Definition"), 0, KShortcut("Ctrl+]"), this,
		SLOT(slotQueryQuickDef()), actionCollection(), "cscope_quick_def");
	(void)new KAction(i18n("Call &Graph..."), 0, KShortcut("Ctrl+\\"), this,
		SLOT(slotCallTree()), actionCollection(), "cscope_call_tree");
	(void)new KAction(i18n("Re&build database"), "reload", 0, this,
		SLOT(slotRebuildDB()), actionCollection(),
		"cscope_rebuild");

	// Go menu
	(void)new KAction(i18n("P&revious Result"), "up", 
		KShortcut(QKeySequence(ALT + Key_Up)), m_pQueryWidget,
		SLOT(slotPrevResult()), actionCollection(), "go_prev_result");
	(void)new KAction(i18n("N&ext Result"), "down", 
		KShortcut(QKeySequence(ALT + Key_Down)), m_pQueryWidget,
		SLOT(slotNextResult()), actionCollection(),
		"go_next_result");
	(void)new KAction(i18n("&Previous Position"), "previous", 
		KShortcut(QKeySequence(ALT + Key_Left)), m_pQueryWidget,
		SLOT(slotHistoryPrev()), actionCollection(), "go_prev_pos");
	(void)new KAction(i18n("&Next Position"), "next", 
		KShortcut(QKeySequence(ALT + Key_Right)), m_pQueryWidget,
		SLOT(slotHistoryNext()), actionCollection(),
		"go_next_pos");
	(void)new KAction(i18n("Position &History"), 0, KShortcut("Ctrl+h"),
		this, SLOT(slotHistoryShow()), actionCollection(),
		"go_history");

	// View menu
	m_pToggleFileViewAction = new KToggleAction(i18n("Show/Hide File List"),
		"view_sidetree", KShortcut("Ctrl+/"), m_pFileViewDock,
		SLOT(changeHideShowState()), actionCollection(),
		"view_toggle_filelist_dock");
	m_pToggleQueryWindowAction = new KToggleAction(
		i18n("Show/Hide Query Window"), "view_top_bottom",
		KShortcut("Ctrl+."), m_pQueryDock, SLOT(changeHideShowState()),
		actionCollection(), "view_toggle_query_dock");
	m_pToggleTagListAction = new KToggleAction(i18n("Show/Hide Tag List"),
		"view_detailed", KShortcut("Ctrl+'"), m_pEditTabs,
		SLOT(slotToggleTagList()), actionCollection(),
		"view_toggle_tag_list");
	
	// Window menu
	(void)new KAction(i18n("Close &All"), 0, 0, this,
		SLOT(slotCloseAllWindows()), actionCollection(), "window_close_all");
	(void)new KAction(i18n("Go &Left"), 0, 
		KShortcut(QKeySequence(ALT + SHIFT + Key_Left)),
		m_pEditTabs, SLOT(slotGoLeft()), actionCollection(),
		"window_go_left");
	(void)new KAction(i18n("Go &Right"), 0,
		KShortcut(QKeySequence(ALT + SHIFT + Key_Right)),
		m_pEditTabs, SLOT(slotGoRight()), actionCollection(),
		"window_go_right");
		
	// Settings menu
	KStdAction::preferences(this, SLOT(slotConfigure()), actionCollection());
	KStdAction::keyBindings(this, SLOT(slotShortcuts()), actionCollection());
	
	// Help menu
	(void)new KAction(i18n("Show &Welcome Message..."), 0, 0, this,
		SLOT(slotShowWelcome()), actionCollection(), "help_welcome");
	
	// Query widget popup menu
	(void)new KAction(i18n("&New"), "filenew", 0, m_pQueryWidget,
		SLOT(slotNewQueryPage()), actionCollection(), "query_new");
	(void)new KAction(i18n("&Refresh"), "reload", 0, m_pQueryWidget,
		SLOT(slotRefreshCurrent()), actionCollection(), "query_refresh");
	pLockAction = new KToggleAction(i18n("&Lock/Unlock"), "encrypted", 0,
		m_pQueryWidget, SLOT(slotLockCurrent()), actionCollection(),
		"query_toggle_locked");
	(void)new KAction(i18n("&Close"), "fileclose", 0, m_pQueryWidget,
		SLOT(slotCloseCurrent()), actionCollection(), "query_close");
	
	m_pExtEditAction->setEnabled(Config().useExtEditor());
	
	// Show a toolbar show/hide menu
	setStandardToolBarMenuEnabled(true);
	
	// Create the initial GUI (no active part)
	setXMLFile("kscopeui.rc");
	createShellGUI();

	// Associate the "Window" menu with the editor tabs widdget
	m_pEditTabs->setWindowMenu((QPopupMenu*)factory()->container("window",
		this));

	// Associate the "Query" popup menu with the query widget
	m_pQueryWidget->setPageMenu((QPopupMenu*)factory()->container(
		"query_popup", this), pLockAction);
}

/**
 * Positions child widgets into their docking stations, and performs some
 * other main window initialisation.
 */
void KScope::initMainWindow()
{
	KStatusBar* pStatus;
	KDockWidget* pMainDock;

	// Create the status bar
	pStatus = statusBar();
	pStatus->insertItem(i18n(" Line: N/A Col: N/A "), 0, 0, true);

	// Create the main dock for the editor tabs widget
	pMainDock = createDockWidget("Editors Window", QPixmap());
	pMainDock->setWidget(m_pEditTabs);
	pMainDock->setDockSite(KDockWidget::DockCorner);
	setMainDockWidget(pMainDock);
	setView(pMainDock);
	pMainDock->setEnableDocking(KDockWidget::DockNone);

	// Create the query window dock
	m_pQueryDock->setWidget(m_pQueryWidget);
	m_pQueryDock->manualDock(pMainDock, KDockWidget::DockBottom, 65);

	// Update the relevant shell action when the dock is hidden through its
	// close button
	connect(m_pQueryDock, SIGNAL(headerCloseButtonClicked()), this,
		SLOT(slotQueryDockClosed()));
		
	// Create the file view dock
	m_pFileViewDock->setWidget(m_pFileView);
	m_pFileViewDock->manualDock(pMainDock, KDockWidget::DockRight, 80);
	
	// Update the relevant shell action when the dock is hidden through its
	// close button
	connect(m_pFileViewDock, SIGNAL(headerCloseButtonClicked()), this,
		SLOT(slotFileViewDockClosed()));
	
	// Restore dock configuration
	Config().loadWorkspace(this);
	m_bHideQueryOnSelection = m_pQueryDock->isHidden();
	m_pToggleFileViewAction->setChecked(m_pFileViewDock->isShown());
	m_pToggleQueryWindowAction->setChecked(m_pQueryDock->isShown());
	m_pToggleTagListAction->setChecked(Config().getShowTagList());
}

/**
 * Handles the "File->Quit" command. Closes the main window, which terminates
 * the application.
 */
void KScope::slotClose()
{
	// Destroy the main window
	KParts::DockMainWindow::close();
}

/**
 * Called when a request has been issued to close the main window.
 * Tries to close the active project.
 * @return	true if the main window can be closed, false otherwise
 */
bool KScope::queryClose()
{
	bool bResult;

	m_bUpdateGUI = false;
	bResult = slotCloseProject();
	m_bUpdateGUI = true;

	return true;
}

/**
 * Handles the "Project->New..." command.
 * Prompts the user for the name and folder for the project, and then creates
 * the project.
 */
void KScope::slotCreateProject()
{
	NewProjectDlg dlg(true, this);
	ProjectManager::Options opt;

	// Prompt the user to close any active projects
	if (m_pProjMgr->isOpen()) {
		if (KMessageBox::questionYesNo(0, 
			i18n("The current project needs to be closed before a new one is"
			" created.\nWould you like to close it now?")) != 
			KMessageBox::Yes) {
			return;
		}
		
		// Try to close the project.
		if (!slotCloseProject())
			return;
	}
	
	// Display the "New Project" dialog
	if (dlg.exec() != QDialog::Accepted)
		return;

	// Create the new project
	dlg.getOptions(opt);
	m_pProjMgr->create(dlg.getName(), dlg.getPath(), opt);

	// Open the project
	openProject(dlg.getPath() + "/" + dlg.getName());
}

/**
 * Handles the "Project->Open..." command.
 * Prompts the user for a project file ("cscope.proj"), and opens the
 * selected project.
 */
void KScope::slotOpenProject()
{
	OpenProjectDlg dlg;

	if (dlg.exec() == QDialog::Rejected)
		return;

	openProject(dlg.getPath());
}

/**
 * Handles the "Project->Add/Remove Files..." command.
 * Opens the project's files dialog, which allows the user to add and remove
 * source files.
 */
void KScope::slotProjectFiles()
{
	// A project must be open
	if (!m_pProjMgr->isOpen())
		return;

	// Display the files dialog
	ProjectFilesDlg dlg(m_pProjMgr, this);
	if (dlg.exec() == QDialog::Accepted)
		m_pProjMgr->writeList(&dlg);
}

/**
 * Handles the "Project->Properties..." command.
 * Opens the project's properties dialog, which allows the user to change
 * some attributes of the current project.
 * source files.
 */
void KScope::slotProjectProps()
{
	ProjectManager::Options opt;
	
	// A project must be open
	if (!m_pProjMgr->isOpen())
		return;

	// No properties for a temporary project
	if (m_pProjMgr->isTemporary()) {
		KMessageBox::error(0, i18n("The Project Properties dialogue is not "
			"available for temporary projects."));
		return;
	}
	
	// Create the properties dialog
	NewProjectDlg dlg(false, this);
	m_pProjMgr->getOptions(opt);
	dlg.setProperties(m_pProjMgr->getName(), m_pProjMgr->getPath(), opt);
	
	// Display the properties dialog
	if (dlg.exec() != QDialog::Accepted)
		return;

	// Set new properties
	dlg.getOptions(opt);
	m_pProjMgr->setOptions(opt);
	
	// Reset the CscopeFrontend class and the builder object
	initCscope();
	
	// Set auto-completion parameters
	SymbolCompletion::initAutoCompletion(opt.bACEnabled, opt.nACMinChars,
		opt.nACDelay, opt.nACMaxEntries);
}

/**
 * Handles the "Cscope->Open Cscope.out..." menu command.
 * Prompts the user for a Cscope.out file, and, if successful, opens a new
 * session for working with this file.
 */
void KScope::slotProjectCscopeOut()
{
	QString sFilePath;
	
	// Prompt for a Cscope.out file
	sFilePath = KFileDialog::getOpenFileName();
	if (sFilePath.isEmpty())
		return;

	// Open a temporary project
	openCscopeOut(sFilePath);	
}

/**
 * Handles the "Cscope->References..." menu command.
 * Prompts the user for a symbol name, and initiates a query to find all 
 * references to that symbol.
 */
void KScope::slotQueryReference()
{
	slotQuery(SymbolDlg::Reference, true);
}

/**
 * Handles the "Cscope->Definition..." menu command.
 * Prompts the user for a symbol name, and initiates a query to find the 
 * global definition of that symbol.
 */
void KScope::slotQueryDefinition()
{
	slotQuery(SymbolDlg::Definition, true);
}

/**
 * Handles the "Cscope->Called Functions..." menu command.
 * Prompts the user for a function name, and initiates a query to find all 
 * function calls from that function.
 */
void KScope::slotQueryCalled()
{
	slotQuery(SymbolDlg::Called, true);
}

/**
 * Handles the "Cscope->Calling Functions..." menu command.
 * Prompts the user for a function name, and initiates a query to find all 
 * functions calling that function.
 */
void KScope::slotQueryCalling()
{
	slotQuery(SymbolDlg::Calling, true);
}

/**
 * Handles the "Cscope->Find Text..." menu command.
 * Prompts the user for a string, and initiates a query to find all
occurances 
 * of that string.
 */
void KScope::slotQueryText()
{
	slotQuery(SymbolDlg::Text, true);
}

/**
 * Handles the "Cscope->Find EGrep Pattern..." menu command.
 * Prompts the user for a regular expression, and initiates a query to find 
 * all strings matching that pattern.
 */
void KScope::slotQueryPattern()
{
	slotQuery(SymbolDlg::Pattern, true);
}

/**
 * Handles the "Cscope->Find File..." menu command.
 * Prompts the user for a file name, and initiates a query to find all files
 * having that name.
 */
void KScope::slotQueryFile()
{
	slotQuery(SymbolDlg::FileName, true);
}

/**
 * Handles the "Cscope->Find Including Files..." menu command.
 * Prompts the user for a file name, and initiates a query to find all files
 * having an #include directive to that file.
 */
void KScope::slotQueryIncluding()
{
	slotQuery(SymbolDlg::Including, true);
}

/**
 * Handles the "Cscope->Quick Definition" menu command.
 * Initiates a query to find the global definition of the symbol currently
 * selected or under the cursor. The user is prompted only if no symbol can
 * be found.
 */
void KScope::slotQueryQuickDef()
{
	QString sSymbol;
	QueryViewDlg* pDlg;
	uint nType;
	
	// Get the requested symbol and query type
	nType = SymbolDlg::Definition;
	if (!getSymbol(nType, sSymbol, false))
		return;
		
	// Create a modeless query view dialogue
	pDlg = new QueryViewDlg(QueryViewDlg::DestroyOnSelect, this);
	
	// Display a line when it is selected in the dialogue
	connect(pDlg, SIGNAL(lineRequested(const QString&, uint)), this,
		SLOT(slotShowEditor(const QString&, uint)));
		
	// Start the query
	pDlg->query(nType, sSymbol);
}

/**
 * Handles the "Cscope->Call Tree..." menu command.
 * Displays a tree of functions calling the requested function.
 */
void KScope::slotCallTree()
{
	slotQuery(SymbolDlg::CallTree, true);
}

/**
 * Handles the "Cscope->Rebuild Database..." command.
 * Rebuilds Cscope's database for the current project.
 */
void KScope::slotRebuildDB()
{
	if (!m_pProjMgr->dbExists()) {
		m_pProgressDlg = new ProgressDlg(i18n("KScope"), i18n("Please wait "
			"while the database is built for the first time"), this);
		m_pProgressDlg->setAllowCancel(false);
		m_pProgressDlg->setValue(0);
	}
	
	m_pCscopeBuild->rebuild();
}

/**
 * Handles the "Settings->Configure Shortcuts..." command.
 * Displays the prferences dialog, which allows the user 
 * to change the shortcuts for KScope.
 */
void KScope::slotShortcuts()
{
	KKeyDialog::configure(actionCollection(), this);
}

/**
 * Handles the "Settings->Configure KScope..." command.
 * Displays the prferences dialog, which allows the user to set different
 * configuration parameters for KScope.
 */
void KScope::slotConfigure()
{
	PreferencesDlg dlg;

	// Apply the preferences if either the "Apply" or the "OK" buttons are
	// clicked
	connect(&dlg, SIGNAL(applyPref()), this, SLOT(slotApplyPref()));

	// Show the dialog
	dlg.exec();
}

/**
 * Refreshes the file list when files are added to or removed from a project,
 * and rebuilds the Cscope database.
 * This slot is attached to the fileListChanged() signal emitted by 
 * the ProjectManager object.
 */
void KScope::slotProjectFilesChanged()
{
	QStringList slArgs;
	
	// Refresh the file list
	m_pFileList->setUpdatesEnabled(false);
	m_pFileList->clear();
	m_pProjMgr->fillList(m_pFileList);
	m_pFileList->setUpdatesEnabled(true);
	
	// Rebuild the symbol database
	if (isAutoRebuildEnabled())
		slotRebuildDB();
}

/**
 * Adds a list of files to the current project.
 * This slot is connected to the filesAdded() signal of the ProjectManager
 * object. Once files are added to the project, they are also added to the
 * file list, and the project's database is rebuilt.
 * @param	slFiles	The list of file paths added to the project
 */
void KScope::slotFilesAdded(const QStringList& slFiles)
{
	QStringList::const_iterator itr;

	// Add the file paths to the project's file list
	for (itr = slFiles.begin(); itr != slFiles.end(); ++itr)
		m_pFileList->addItem(*itr);
	
	// Rebuild the database
	if (isAutoRebuildEnabled())
		slotRebuildDB();
}

/**
 * Promts the user for a symbol, an starts a new Cscope query.
 * @param	nType	The numeric query type code
 * @param	bPrompt	true to always prompt for a symbol, false to try to
 * 					obtain the symbol automatically
 */
void KScope::slotQuery(uint nType, bool bPrompt)
{
	QString sSymbol;
	CallTreeDlg* pCallTreeDlg;
	
	// Get the requested symbol and query type
	if (!getSymbol(nType, sSymbol, bPrompt))
		return;
		
	if (nType == SymbolDlg::CallTree) {
		// Create and display a call tree dialogue
		pCallTreeDlg = m_pCallTreeMgr->addDialog();
		pCallTreeDlg->setRoot(sSymbol);
		pCallTreeDlg->show();
	}
	else {
		// Run the requested query
		nType = SymbolDlg::getQueryType(nType);
		m_pQueryWidget->initQuery(nType, sSymbol);
		
		// Ensure Query Window is visible
		toggleQueryWindow(true);	
	}
}

/**
 * Opens a project.
 * If another project is currently active, it is cosed first.
 * @param	sDir	The directory of the project to open.
 */
void KScope::openProject(const QString& sDir)
{
	QStringList slQueryFiles;
	QStringList slCallTreeFiles;
	QStringList slArgs;
	ProjectManager::Options opt;
	
	// Close the current project (may return false if the user clicks on the
	// "Cancel" button while prompted to save a file)
	if (!slotCloseProject())
		return;

	// Open the project in the project manager
	if (!m_pProjMgr->open(sDir))
		return;
	
	// Change main window title
	setCaption(m_pProjMgr->getName());

	// Set the root of the file tree
	m_pFileView->setRoot(m_pProjMgr->getRoot());
	
	// Initialise Cscope and create a builder object
	initCscope();
	
	// Set auto-completion parameters
	m_pProjMgr->getOptions(opt);
	SymbolCompletion::initAutoCompletion(opt.bACEnabled, opt.nACMinChars,
		opt.nACDelay, opt.nACMaxEntries);
	
	// Create an initial query page
	m_pQueryWidget->addQueryPage();
	
	// If this is a new project (i.e., no source files are yet included), 
	// display the project files dialogue
	if (m_pProjMgr->isEmpty()) {
		slotProjectFiles();
		return;
	}
	
	// Fill the file list with all files in the project. 
	m_pFileList->setUpdatesEnabled(false);
	m_pProjMgr->fillList(m_pFileList);
	m_pFileList->setUpdatesEnabled(true);
	
	// Reload previously-open files
	loadOpenFiles();
		
	// Load previously stored queries
	m_pProjMgr->getQueryFiles(slQueryFiles);
	m_pQueryWidget->loadLockedQueries(m_pProjMgr->getPath(), slQueryFiles);
	
	// Load previously stored call trees
	m_pProjMgr->getCallTreeFiles(slCallTreeFiles);
	m_pCallTreeMgr->loadOpenDialogs(m_pProjMgr->getPath(), slCallTreeFiles);
	
	// Rebuild the cross-reference database
	if (isAutoRebuildEnabled())
		slotRebuildDB();
}

/**
 * Reopens all files which were open when the project was last closed.
 * In order to reduce the time required by this operation, the GUI of all
 * but the last editor part is not merged with that of the main window.
 */
void KScope::loadOpenFiles()
{
	OpenFileList list;
	FileLocation* pLoc;
	EditorPage* pPage;
	
	// Get the list of open files
	list.setAutoDelete(true);
	m_pProjMgr->getOpenFiles(list);
	if (list.isEmpty())
		return;
	
	// Do not update the GUI when loading the editor parts of the initially
	// hidden windows
	m_bUpdateGUI = false;
	
	for (pLoc = list.first(); pLoc != NULL; pLoc = list.next()) {
		if (QFile::exists(pLoc->m_sPath)) {
			pPage = addEditor(pLoc->m_sPath);
			pPage->setCursorPos(pLoc->m_nLine, pLoc->m_nCol);
		}
	}
	
	// Merge the GUI of the visible editor part
	m_bUpdateGUI = true;

	// Set the active editor (or choose a default one)
	if (m_pEditTabs->findEditorPage(m_pProjMgr->getLastOpenFile(), true) ==
		NULL) {
		m_pEditTabs->findEditorPage(list.last()->m_sPath, true);
	}
}

/**
 * Shows or hides the query dock window.
 * This function is only called internally, not as a result of a user's
 * workspace action (e.g., clicking the "Show/Hide Query Window" toolbar
 * button). Therefore it does not reflect the user's preference, which is
 * kept through the m_bHideQueryOnSelection variable.
 * @param	bShow	true to show the window, false to hide it
 */
void KScope::toggleQueryWindow(bool bShow)
{
	// Remember the user's preferences
	if (bShow)
		m_bHideQueryOnSelection = m_pQueryDock->isHidden();
	else
		m_bHideQueryOnSelection = false;
	
	// Change the visibility state of the widget, if required
	if (m_pQueryDock->isShown() != bShow)
		m_pQueryDock->changeHideShowState();
		
	// Make sure the action's state matches the widget's
	m_pToggleQueryWindowAction->setChecked(bShow);
}

/**
 * Opens a temporary project for a Cscope.out file.
 * @param	sFilePath	The full path of the Cscope.out file
 * @return	true if successful, false otherwise
 */
bool KScope::openCscopeOut(const QString& sFilePath)
{	
	// Close the current project (may return false if the user clicks on the
	// "Cancel" button while prompted to save a file)
	if (!slotCloseProject())
		return false;

	// Open a temporary project for this cscope.out file
	if (!m_pProjMgr->openCscopeOut(sFilePath))
		return false;
	
	// Change main window title
	setCaption(m_pProjMgr->getName());
	
	// Initialise Cscope and create a builder object
	initCscope();
	
	// Create an initial query page
	m_pQueryWidget->addQueryPage();
	return true;
}

/**
 * Parses the command line, after it was stripped of its KDE options.
 * The command line may contain one of the following options:
 * 1. A project file (named cscope.proj)
 * 2. A Cscope cross-reference database
 * 3. A list of source files
 * @param	pArgs	Command line arguments
 */
void KScope::parseCmdLine(KCmdLineArgs* pArgs)
{
	QString sArg;
	QFileInfo fi;
	int i;

	// Loop over all arguments
	for (i = 0; i < pArgs->count(); i++) {
		// Verify the argument is a file or directory name
		sArg = pArgs->arg(i);
		fi.setFile(sArg);
		if (!fi.exists())
			continue;
			
		// Handle the current argument
		if (fi.isFile()) {
			if (fi.fileName() == "cscope.proj") {
				// Open a project file
				openProject(fi.dirPath(true));
				return;
			} else if (openCscopeOut(sArg)) {
				// Opened the file as a cross-reference database
				return;
			} else {
				// Assume this is a source file
				slotShowEditor(sArg, 0);
			}
		} else if (fi.isDir()) {
			// Treat the given path as a project directory
			openProject(fi.absFilePath());
			return;
		}
	}
}

/**
 * Initialises the CscopeFrontend class with the current project arguments,
 * and creates an object used for rebuilding the symbol database.
 */
void KScope::initCscope()
{
	// Delete the current object, if one exists
	if (m_pCscopeBuild)
		delete m_pCscopeBuild;

	// Initialise CscopeFrontend
	CscopeFrontend::init(m_pProjMgr->getPath(), m_pProjMgr->getArgList());

	// Create a persistent Cscope process
	m_pCscopeBuild = new CscopeFrontend();

	// Show build progress information in the main status bar
	connect(m_pCscopeBuild, SIGNAL(progress(int, int)), this,
		SLOT(slotBuildProgress(int, int)));
	connect(m_pCscopeBuild, SIGNAL(finished(uint)), this,
		SLOT(slotBuildFinished(uint)));
	connect(m_pCscopeBuild, SIGNAL(aborted()), this,
		SLOT(slotBuildAborted()));

	// Show errors in a modeless dialogue
	connect(m_pCscopeBuild, SIGNAL(error(const QString&)), this,
		SLOT(slotCscopeError(const QString&)));
}

/**
 * Closes the active project.
 * Closing a project involves closing all of the editor windows (prompting
 * the user for unsaved changes); terminating the Cscope process; and further
 * clean-up of the project's data.
 */
bool KScope::slotCloseProject()
{
	QString sLastOpenFile;
	OpenFileList lstOpenFiles;
	QStringList slQueryFiles;
	QStringList slCallTreeFiles;
	
	// Do nothing if no project is open
	if (!m_pProjMgr->isOpen())
		return true;
	
	// Close all open editor pages
	if (m_pEditTabs->count() > 0) {
		sLastOpenFile = m_pEditTabs->getCurrentPage()->getFilePath();
		m_pEditTabs->getOpenFiles(lstOpenFiles);
		if (!m_pEditTabs->removeAllPages())
			return false;
	}
	
	// Store locked queries, and close all query pages
	m_pQueryWidget->saveLockedQueries(m_pProjMgr->getPath(), slQueryFiles);
	m_pQueryWidget->slotCloseAll();

	// Store opened call trees
	m_pCallTreeMgr->saveOpenDialogs(m_pProjMgr->getPath(), slCallTreeFiles);
	m_pCallTreeMgr->closeAll();
	
	// Save current session
	lstOpenFiles.setAutoDelete(true);
	m_pProjMgr->setLastOpenFile(sLastOpenFile);
	m_pProjMgr->setOpenFiles(lstOpenFiles);
	m_pProjMgr->setQueryFiles(slQueryFiles);
	m_pProjMgr->setCallTreeFiles(slCallTreeFiles);
	
	// Close the project in the project manager, and terminate the Cscope
	// process
	m_pProjMgr->close();
	delete m_pCscopeBuild;
	m_pCscopeBuild = NULL;
	setCaption(QString::null);

	// Clear the contents of the file list
	m_pFileView->clear();

	return true;
}

/**
 * Handles the "Edit->Edit in External Editor" menu command.
 * Invokes an external editor for the current file and line number.
 */
void KScope::slotExtEdit()
{
	QString sCmdLine;
	KProcess proc;

	// Create the command line for the external editor	
	sCmdLine = Config().getExtEditor();
	sCmdLine.replace("%F", m_sCurFilePath);
	sCmdLine.replace("%L", QString::number(m_nCurLine));
	
	// Run the external editor
	proc.setUseShell(true);
	proc << sCmdLine;
	proc.start(KProcess::DontCare);
}

/**
 * Handles the "Edit->Complete Symbol" menu command.
 * Creates a list of possible completions for the symbol currently under the
 * cursor.
 */
void KScope::slotCompleteSymbol()
{
	EditorPage* pPage;
	
	pPage = m_pEditTabs->getCurrentPage();
	if (pPage != NULL)
		pPage->slotCompleteSymbol();
}

/**
 * Handles the "Help->Show Welcome Message..." menu command.
 * Displays the "Welcome" dialogue.
 */
void KScope::slotShowWelcome()
{
	WelcomeDlg dlg;
	dlg.exec();
}

/**
 * Prompts the user for a symbol to query.
 * Shows a dialog with a line edit widget, where the user can enter a symbol
 * on which to query Cscope. The meaning of the symbol depends on the type of
 * query.
 * @param	nType	The requested type of query (may be changed in the
 *					dialogue)
 * @param	sSymbol	Holds the requested symbol, upon successful return
 * @param	bPrompt	If false, the user is prompted only if a symbol cannot be
 *					determined automatically
 * @return	true if the user hs enetered a symbol, false otherwise
 */
bool KScope::getSymbol(uint& nType, QString& sSymbol, bool bPrompt)
{
	EditorPage* pPage;
	QString sSuggested;
	QStringList slSymHistory;
	
	// Set the currently selected text, if any
	if ((pPage = m_pEditTabs->getCurrentPage()) != NULL)
		sSuggested = pPage->getSuggestedText();

	// Return if a symbol was found, and prompting is turned off
	if (!sSuggested.isEmpty() && !bPrompt) {
		sSymbol = sSuggested;
		return true;
	}
	
	// Get the list of previously requested symbols
	m_pProjMgr->getSymHistory(slSymHistory);
	
	// Show the symbol dialogue
	sSymbol = SymbolDlg::promptSymbol(this, nType, sSuggested, slSymHistory);

	// Cannot accept empty strings
	if (sSymbol.isEmpty())
		return false;
	
	// Update the symbol history list from the dialog
	m_pProjMgr->setSymHistory(slSymHistory);
	return true;
}

/**
 * Opens a file in a new editor tab.
 * If an editor page already exists for the requested file, it is selected.
 * Otherwise, a new page is created, and the requested file is loaded.
 * @param	sFilePath	The path of the file to open
 * @return	A pointer to the found or newly created editor page
 */
EditorPage* KScope::addEditor(const QString& sFilePath)
{
	EditorPage* pPage;
	QString sAbsFilePath;
		
	// If the file name is given using a relative path, we need to convert
	// it to an absolute one
	if (sFilePath[0] != '/') {
		sAbsFilePath = QDir::cleanDirPath(m_pProjMgr->getPath() + "/" +
			sFilePath);
	}
	else {
		sAbsFilePath = QDir::cleanDirPath(sFilePath);
	}
	
	// Do not open a new editor if one exists for this file
	pPage = m_pEditTabs->findEditorPage(sAbsFilePath);
	if (pPage != NULL)
		return pPage;

	// Create a new page
	pPage = createEditorPage();	
				
	// Open the requested file
	pPage->open(sAbsFilePath);
	
	return pPage;
}

/**
 * Creates a new editor page, and adds it to the editors tab widget.
 * @return	A pointer to the new page
 */
EditorPage* KScope::createEditorPage()
{
	KTextEditor::Document* pDoc;
	EditorPage* pPage;
	QPopupMenu* pMenu;
	
	// Load a new document part
	pDoc = m_pEditMgr->add();
	if (pDoc == NULL)
		return NULL;

	// Create the new editor page
	pMenu = (QPopupMenu*)factory()->container(Config().getEditorPopupName(),
		this);
	pPage = new EditorPage(pDoc, pMenu, m_pEditTabs);
	m_pEditTabs->addEditorPage(pPage);

	// Show the file's path in the main title
	connect(pPage, SIGNAL(fileOpened(EditorPage*, const QString&)), this,
		SLOT(slotFileOpened(EditorPage*, const QString&)));

	// Show cursor position in the status bar
	connect(pPage, SIGNAL(cursorPosChanged(uint, uint)), this,
		SLOT(slotShowCursorPos(uint, uint)));
	
	// Rebuild the database after a file has changed
	connect(pPage, SIGNAL(fileSaved(const QString&)), this,
		SLOT(slotSetRebuildTimer(const QString&)));
	
	// Handle file drops
	connect(pPage->getView(), SIGNAL(dropEventPass(QDropEvent*)), this,
		SLOT(slotDropEvent(QDropEvent*)));
	
	return pPage;
}
	
/**
 * @return	true if database auto-rebuild is enabled for the current project,
 *			false otherwise
 */
inline bool KScope::isAutoRebuildEnabled()
{
	return (m_pProjMgr->getAutoRebuildTime() >= 0);
}

/**
 * Deletes an editor page after it has been removed.
 * The document object associated with the page is removed from the part 
 * manager, and the view object is removed from the GUI manager.
 * This slot is connected to the editorRemoved() signal of the EditorTabs 
 * object.
 * @param	pPage	The editor page to delete
 */
void KScope::slotDeleteEditor(EditorPage* pPage)
{
	guiFactory()->removeClient(pPage->getView());
	m_pEditMgr->remove(pPage->getDocument());
	delete pPage;
}

/**
 * Sets an editor part as active when its owner tab is chosen.
 * Whenever a different editor tab is chosen, its editor part should become
 * the active part. This means that this part's GUI is merged with the
 * application's, and that it responds to actions.
 * @param	pOldPage	The editor page that has ceased to be active
 * @param	pNewPage	The newly chosen editor page
 */
void KScope::slotChangeEditor(EditorPage* pOldPage, EditorPage* pNewPage)
{
	KXMLGUIFactory* pFactory = guiFactory();
	
	// Remove the current GUI
	if (pOldPage)
		pFactory->removeClient(pOldPage->getView());

	// Set the new active part and create its GUI
	if (m_bUpdateGUI && pNewPage) {
		m_pEditMgr->setActivePart(pNewPage->getDocument());
		pFactory->addClient(pNewPage->getView());
		m_sCurFilePath = pNewPage->getFilePath();
		setCaption(m_pProjMgr->getName() + " - " + m_sCurFilePath);
	}
}

/**
 * Opens an editor for the given file and sets the cursor to the beginning of 
 * the requested line.
 * @param	sFilePath	The full path of the file to open for editing
 * @param	nLine		The number of the line on which to position the
 *						cursor, or 0 to maintain the cursor in its current
 *						position (which does not affect the position history)
 */
void KScope::slotShowEditor(const QString& sFilePath, uint nLine)
{
	EditorPage* pPage;

	// Save current position in the position history
	if (nLine != 0 && (pPage = m_pEditTabs->getCurrentPage())) {
		m_pQueryWidget->addHistoryRecord(m_sCurFilePath, m_nCurLine,
			pPage->getLineContents(m_nCurLine));
	}
	
	// Open the requested file (or select an already-open editor page)
	pPage = addEditor(sFilePath);
	if (pPage == NULL)
		return;

	if (nLine != 0) {
		// Set the cursor to the requested line
		pPage->slotGotoLine(nLine);
	
		// Add the new position to the position history
		m_pQueryWidget->addHistoryRecord(m_sCurFilePath, m_nCurLine,
			pPage->getLineContents(m_nCurLine));
	}
}

/**
 * A wrapper around slotShowEditor, that enables auto-hiding of the query
 * widget after a query result has been chosen.
 * This slot is connected to the lineRequested() signal emitted by a QueryPage
 * object.
 * @param	sFilePath	The full path of the file to open for editing
 * @param	nLine		The number of the line on which to position the cursor
 */
void KScope::slotQueryShowEditor(const QString& sFilePath, uint nLine)
{
	// Hide the query window, if it was hidden before a query was initiated
	if (m_bHideQueryOnSelection)
		toggleQueryWindow(false);
	
	// Open an editor at the requested line
	slotShowEditor(sFilePath, nLine);
}

/**
 * Handles the "Go->Position History" menu command.
 * Ensures that the query window is visible, and selects the active history
 * page.
 */
void KScope::slotHistoryShow()
{
	toggleQueryWindow(true);
	m_pQueryWidget->selectActiveHistory();
}

/**
 * Handles the "File->New" menu command.
 * Creates an editor page for a new unnamed file.
 */
void KScope::slotNewFile()
{
	EditorPage* pPage;

	// Create the new editor page
	pPage = createEditorPage();
	
	// Mark the page as containing a new file
	pPage->setNewFile();
}

/**
 * Handles the "File->Open" menu command.
 * Prompts the user for a file name, and opens it in a new editor page.
 */
void KScope::slotOpenFile()
{
	QStringList slFiles;
	QStringList::Iterator itr;
	
	// Prompt the user for the file(s) to open.
	slFiles = KFileDialog::getOpenFileNames();
	
	// Open all selected files.
	for (itr = slFiles.begin(); itr != slFiles.end(); ++itr) {
		if (!(*itr).isEmpty())
			slotShowEditor(*itr, 0);
	}
}

/**
 * Handles the "File->Close" menu command.
 * Closes the currently active editor page.
 */
void KScope::slotCloseEditor()
{
	m_pEditTabs->removeCurrentPage();
}

/**
 * Handles the "Window->Close All" menu command.
 * Closes all open editor pages.
 */
void KScope::slotCloseAllWindows()
{
	m_bUpdateGUI = false;
	m_pEditTabs->removeAllPages();
	m_bUpdateGUI = true;
}

/**
 * Displays error messages from a Cscope process.
 * This slot is connected to the progress() signal emitted by the any
 * Cscope process.
 * @param	sMsg	The error message
 */
void KScope::slotCscopeError(const QString& sMsg)
{
	m_pMsgDlg->addText(sMsg);
}

/**
 * Reports progress information from the Cscope process responsible for
 * rebuilding the cross-reference database.
 * This slot is connected to the progress() signal emitted by the builder
 * process.
 * Progress information is displayed in the status bar.
 * @param	nFiles	The number of files scanned
 * @param	nTotal	The total number of files in the project
 */
void KScope::slotBuildProgress(int nFiles, int nTotal)
{
	QString sMsg;
	
	// Use the progress dialogue, if it exists (first time builds)
	if (m_pProgressDlg) {
		m_pProgressDlg->setValue((nFiles * 100) / nTotal);
		return;
	}
	
	// Show progress information
	sMsg = i18n("Building cross reference database...") + " " +
		QString::number((nFiles * 100) / nTotal) + "%";
	statusBar()->message(sMsg);
}

/**
 * Informs the user the database rebuild process has finished.
 * This slot is connected to the finished() signal emitted by the builder
 * process.
 */
void KScope::slotBuildFinished(uint)
{
	// Delete the progress dialogue, if it exists (first time builds)
	if (m_pProgressDlg) {
		delete m_pProgressDlg;
		m_pProgressDlg = NULL;
		return;
	}
	
	// Show a message in the status bar
	statusBar()->message(i18n("Building cross reference database...Done!"),
		3000);
}

/**
 * Called if the build process failed to complete.
 * This slot is connected to the aborted() signal emitted by the builder
 * process.
 */
void KScope::slotBuildAborted()
{
	// Delete the progress dialogue, if it exists (first time builds)
	if (m_pProgressDlg) {
		delete m_pProgressDlg;
		m_pProgressDlg = NULL;
	
		// Display a failure message
		KMessageBox::error(0, i18n("The database could not be built.\n"
			"Cross-reference information will not be available for this "
			"project.\n"
			"Please ensure that the Cscope parameters were correctly "
			"entered in the \"Settings\" dialogue."));		
		return;
	}
	
	// Show a message in the status bar
	statusBar()->message(i18n("Building cross reference database...Failed"),
		3000);	
}

/**
 * Applies the selected user preferences once the "Apply" or "OK" buttons in
 * the preferences dialog is clicked.
 */
void KScope::slotApplyPref()
{
	m_pQueryWidget->applyPrefs();
	m_pFileList->applyPrefs();
	m_pEditTabs->applyPrefs();
	m_pEditMgr->applyPrefs();

	// Enable/disable the external editor menu item
	m_pExtEditAction->setEnabled(Config().useExtEditor());
}

/**
 * Displays the current cursor position, whenever it is moved by the user.
 * This slot is connected to the cursorPosChanged() signal emitted by an
 * EditorPage object.
 * @param	nLine	The new line number
 * @param	nCol	The new column number
 */
void KScope::slotShowCursorPos(uint nLine, uint nCol)
{
	KStatusBar* pStatus = statusBar();
	QString sText;
	
	/* Show the line and column numbers. */
	QTextOStream(&sText) << " Line: " << nLine << " Col: " << nCol << " ";
	pStatus->changeItem(sText, 0);
	
	/* Store the current line. */
	m_nCurLine = nLine;
}

/**
 * Stores the path of a newly opened file.
 * This slot is connected to the fileOpened() signal emitted by an
 * EditorPage object.
 * @param	sFilePath	The full path of the opened file
 */
void KScope::slotFileOpened(EditorPage*, const QString& sFilePath)
{
	m_sCurFilePath = sFilePath;
	setCaption(m_pProjMgr->getName() + " - " + m_sCurFilePath);
}

/**
 * Sets a timer for rebuilding the database after a file has been saved.
 * This slot is connected to the fileSaved() signal emitted by an EditorPage
 * object.
 * The time period before rebuilding is determined on a per-project basis.
 * @param	sPath	The full path of the modified file that caused this event
 */
void KScope::slotSetRebuildTimer(const QString& sPath)
{
	int nTime;
	
	// Get the project's auto-rebuild time
	nTime = m_pProjMgr->getAutoRebuildTime();
	
	// Do nothing if the time is set to -1
	if (nTime == -1)
		return;
		
	// Check if the file is included in the project (external files should
	// not trigger the timer)
	if (!m_pFileList->findFile(sPath))
		return;
	
	// Rebuild immediately for a time set to 0
	if (nTime == 0) {
		slotRebuildDB();
		return;
	}

	// Reset the rebuild timer
	m_timerRebuild.start(nTime * 1000, true);
}

/**
 * Sets a new file tree root for the current project.
 * This slot is connected to the rootChanged() signal of the FileView object.
 * @param	sRoot	The full path of the directory to serve as the new root
 */
void KScope::slotProjectRootChanged(const QString& sRoot)
{
	m_pProjMgr->setRoot(sRoot);
}

/**
 * Handles file drops inside the editors tab widget.
 * Opens all files dropped over the widget.
 * @param	pEvent	Pointer to an object containing the list of dropped files
 */
void KScope::slotDropEvent(QDropEvent* pEvent)
{
	KURL::List list;
	KURL::List::Iterator itr;
		
	// Create a list of file URLs
	if (!KURLDrag::decode(pEvent, list))
		return;
		
	// Open all files in the list
	for (itr = list.begin(); itr != list.end(); ++itr)
		addEditor((*itr).path());
}

/**
 * Ensures the "Show/Hide Query Window" action is unchecked when the dock
 * is closed through its close button.
 * This slot is conncted to the headerCloseButtonClicked() signal of the
 * query window dock widget.
 */
void KScope::slotQueryDockClosed()
{
	m_pToggleQueryWindowAction->setChecked(false);
}

/**
 * Ensures the "Show/Hide File View" action is unchecked when the dock
 * is closed through its close button.
 * This slot is conncted to the headerCloseButtonClicked() signal of the
 * file view dock widget.
 */
void KScope::slotFileViewDockClosed()
{
	m_pToggleFileViewAction->setChecked(false);
}

#include "kscope.moc"
