/***************************************************************************
 *
 * Copyright (C) 2006 Elad Lahav (elad_lahav@users.sf.net)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

#include <qfile.h>
#include <klocale.h>
#include "bookmarkpage.h"
#include "queryview.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
BookmarkPage::BookmarkPage(QWidget* pParent, const char* szName) :
	QueryPageBase(pParent, szName)
{
	m_pView = new QueryView(this);
	m_pView->setColumnWidth(0, 0);

	connect(m_pView, SIGNAL(lineRequested(const QString&, uint)), this,
		SIGNAL(lineRequested(const QString&, uint)));
	
	// Set colours and font
	applyPrefs();
}

/**
 * Class destructor.
 */
BookmarkPage::~BookmarkPage()
{
}

/**
 * Creates a new bookmark record at the top of the list.
 * @param	sFile	The file name associated with the record
 * @param	nLine	The line number
 * @param	sText	The text of the file at the given line
 */
void BookmarkPage::addRecord(const QString& sFile, uint nLine, 
	const QString& sText)
{
	m_pView->addRecord("", sFile, QString::number(nLine), sText, NULL);
}

/**
 * Deletes the record matching the given path name and line number.
 * @param	sPath	The path name of the item to remove
 * @param	nLine	The line number of the item to remove
 */
void BookmarkPage::removeRecord(const QString& sPath, uint nLine)
{
	QListViewItem* pItem;
	QString sLine;
	
	// Find the first item matching the given path and line number (we assume
	// there is only one - relying on the editor's bookmark system)
	pItem = m_pView->firstChild();
	sLine = QString::number(nLine);
	while (pItem) {
		if ((pItem->text(1) == sPath) && (pItem->text(2) == sLine)) {
			delete pItem;
			break;
		}
		
		pItem = pItem->nextSibling();
	}
}

/**
 * Deletes all bookmarks set for the given file.
 * @param	sPath	The file's path
 */
void BookmarkPage::removeFile(const QString& sPath)
{
	QListViewItem* pItem, * pNext;
	
	// Delete all records related to the given file
	pItem = m_pView->firstChild();
	while (pItem) {
		pNext = pItem->nextSibling();
		if (pItem->text(1) == sPath)
			delete pItem;
		
		pItem = pNext;
	}
}
	
/**
 * Sets the keyboard focus to the list widget.
 */
void BookmarkPage::setFocus()
{
	m_pView->setFocus();
}

/**
 * Fills a list with the locations of all defined bookmarks.
 * @param	list	The list to fill
 */
void BookmarkPage::fillList(FileLocationList& list)
{
	QListViewItemIterator itr(m_pView);
	FileLocation* pLoc;
	
	for (; itr.current(); ++itr) {
		QListViewItem* pItem = *itr;
		
		pLoc = new FileLocation(pItem->text(1), pItem->text(2).toUInt(), 0);
		list.append(pLoc);
	}
}

/**
 * Creates a new bookmark item.
 * This version is used when records are read from a file.
 * @param	sFile	The file name
 * @param	sLine	The line number
 * @param	sText	The contents of the line
 */
void BookmarkPage::addRecord(const QString&, const QString& sFile,
	const QString& sLine, const QString& sText)
{
	m_pView->addRecord("", sFile, sLine, sText, NULL);
}

/**
 * Creates a tab caption for this page, based on the unique page ID.
 * @param	bBrief	true to use brief caption names, false otherwise
 */
QString BookmarkPage::getCaption(bool bBrief) const
{
	return bBrief ? QString(i18n("BMK")) : QString(i18n("Boomarks"));
}

/**
 * Retrieves the name to use for the bookmarks file.
 * @return	The name of the bookmarks file name
 */
QString BookmarkPage::getFileName(const QString&) const
{
	return QString("Bookmarks");
}

#include "bookmarkpage.moc"
