/***************************************************************************
                          ksetispyview.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qclipboard.h>
#include <qfile.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qtooltip.h>
#include <qwidgetstack.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kurllabel.h>

#include "ksetispyview.h"

const QString KSetiSpyView::unknownContent = I18N_NOOP("unknown");

KSetiSpyView::KSetiSpyView(Type type, QWidget *parent, const char *name, WFlags f)
             : QWidget(parent, name, f)
{
  KIconLoader loader;

  popup = new KPopupMenu(this);

  if(type == Text)
    popup->insertItem(loader.loadIcon("editcopy", KIcon::Small),
                      i18n("&Copy to clipboard"), this, SLOT(textToClipboard()), CTRL+Key_C, 100, 100);
  else
  {
    popup->insertItem(loader.loadIcon("editcopy", KIcon::Small),
                      i18n("&Copy to clipboard"), this, SLOT(pixmapToClipboard()), CTRL+Key_C, 100, 100);
    popup->insertItem(loader.loadIcon("filesave", KIcon::Small),
                      i18n("&Save to File..."), this, SLOT(pixmapToFile()), CTRL+Key_S, 101, 101);
  }

  QWidget *window = topLevelWidget();

  if(qstrcmp(window->name(), "Main Window") != 0)
  {
    popup->insertSeparator(200);
    popup->insertItem(loader.loadIcon("exit", KIcon::Small),
                      i18n("Close &Window"), window, SLOT(hide()), Key_Escape, 201, 201);
  }

  if(kdoc != NULL)
    connect(kdoc, SIGNAL(updated()), this, SLOT(updateContent()));
}

KSetiSpyView::~KSetiSpyView()
{
  if(popup != NULL) delete popup;
}

KSetiSpyDoc *KSetiSpyView::kdoc = NULL;

void KSetiSpyView::setDocument(KSetiSpyDoc *doc)
{
  kdoc = doc;
}

void KSetiSpyView::readConfig(bool)
{
}

void KSetiSpyView::saveConfig(bool)
{
}

bool KSetiSpyView::eventFilter(QObject *obj, QEvent* e)
{
  if(e->type() == QEvent::MouseButtonPress)
  {
    QWidget *target = (QWidget *) obj;
    QMouseEvent *click = (QMouseEvent *) e;

    if (click->button() == RightButton && popup != NULL) {
      popup->popup(target->mapToGlobal(click->pos()));
      return(true);
    }
  }

  return(false);
}

void KSetiSpyView::mousePressEvent(QMouseEvent *click)
{
  if (click->button() == RightButton && popup != NULL)
    popup->popup(mapToGlobal(click->pos()));
  else
    QWidget::mousePressEvent(click);
}

QWidget *KSetiSpyView::addField(QWidget *parent, const char *name)
{
  QHBox *fieldBox = new QHBox(parent, name);
  fieldBox->setSpacing(4);

  QLabel *fieldName = new QLabel(fieldBox, "name");
  fieldName->installEventFilter(this);

  QLabel *fieldContentText = new QLabel(fieldBox, "content_text");
  fieldContentText->setAlignment(AlignLeft);
  fieldContentText->installEventFilter(this);

  QHBox *urlBox = new QHBox(fieldBox, "url_box");
  urlBox->setSpacing(4);

  KURLLabel *fieldContentURL = new KURLLabel(urlBox, "content_url");
  fieldContentURL->installEventFilter(this);

  QLabel *fieldAux = new QLabel(urlBox, "aux");
  fieldAux->setAlignment(AlignLeft);
  fieldAux->installEventFilter(this);

  urlBox->setStretchFactor(fieldAux, 1);
  urlBox->hide();
  
  fieldBox->setStretchFactor(fieldContentText, 1);
  fieldBox->setStretchFactor(urlBox, 1);

  names.prepend(name);

  return(fieldBox);
}

void KSetiSpyView::setFieldName(const char *name, const QString& text)
{
  QHBox *fieldBox = (QHBox *) this->child(name, "QHBox");

  if(fieldBox == NULL) return;

  QLabel *fieldName = (QLabel *) fieldBox->child("name", "QLabel");

  if(fieldName == NULL) return;

  fieldName->setText(text);
}

void KSetiSpyView::setFieldContentText(const char *name, const QString& text)
{
  QHBox *fieldBox = (QHBox *) this->child(name, "QHBox");

  if(fieldBox == NULL) return;
    
  QLabel *fieldContentText = (QLabel *) fieldBox->child("content_text", "QLabel");
  QHBox *urlBox = (QHBox *) fieldBox->child("url_box", "QHBox");
  KURLLabel *fieldContentURL = (KURLLabel *) fieldBox->child("content_url", "KURLLabel");

  if(fieldContentText == NULL || urlBox == NULL || fieldContentURL == NULL) return;

  fieldContentText->setText(text);

  fieldContentURL->setText(QString::null);

  fieldContentText->show();
  urlBox->hide();
}

void KSetiSpyView::setFieldContentURL(const char *name, const QString& text, const QString& url, const QString& tooltip
                                                      , const QObject *receiver, const char *member)
{
  QHBox *fieldBox = (QHBox *) this->child(name, "QHBox");

  if(fieldBox == NULL) return;
  
  QLabel *fieldContentText = (QLabel *) fieldBox->child("content_text", "QLabel");
  QHBox *urlBox = (QHBox *) fieldBox->child("url_box", "QHBox");
  KURLLabel *fieldContentURL = (KURLLabel *) fieldBox->child("content_url", "KURLLabel");

  if(fieldContentText == NULL || urlBox == NULL || fieldContentURL == NULL) return;

  fieldContentURL->setText(text);
  fieldContentURL->setURL(url);
  fieldContentURL->setUseTips(!tooltip.isEmpty());
  fieldContentURL->setTipText(tooltip);

  fieldContentText->setText(QString::null);

  fieldContentText->hide();
  urlBox->show();

  disconnect(fieldContentURL, 0, 0, 0);
  connect(fieldContentURL, SIGNAL(leftClickedURL(const QString&)), receiver, member);
}

void KSetiSpyView::setFieldAux(const char *name, const QString& text)
{
  QHBox *fieldBox = (QHBox *) this->child(name, "QHBox");

  if(fieldBox == NULL) return;
  
  QLabel *fieldContentText = (QLabel *) fieldBox->child("content_text", "QLabel");
  QHBox *urlBox = (QHBox *) fieldBox->child("url_box", "QHBox");
  QLabel *fieldAux = (QLabel *) fieldBox->child("aux", "QLabel");

  if(fieldContentText == NULL || urlBox == NULL || fieldAux == NULL) return;

  fieldAux->setText(text);

  fieldContentText->hide();
  urlBox->show();
}

void KSetiSpyView::setFieldToolTip(const char *name, const QString& tooltip)
{
  QHBox *fieldBox = (QHBox *) this->child(name, "QHBox");

  if(fieldBox == NULL) return;

  QToolTip::remove(fieldBox);
  if(!tooltip.isEmpty()) QToolTip::add(fieldBox, tooltip);
}

QStringList KSetiSpyView::text()
{
  QStringList out;

  updateContent(true);

  if(!names.isEmpty())
  {
    names.last();
    do {
      QHBox *fieldBox = (QHBox *) this->child(names.current(), "QHBox");

      if(fieldBox == NULL) continue;

      QLabel *fieldName = (QLabel *) fieldBox->child("name", "QLabel");

      if(fieldName == NULL || fieldName->text().isEmpty()) continue;

      QString fieldStr = fieldName->text();

      QLabel *fieldContentText = (QLabel *) fieldBox->child("content_text", "QLabel");
      QHBox *urlBox = (QHBox *) fieldBox->child("url_box", "QHBox");
      KURLLabel *fieldContentURL = (KURLLabel *) fieldBox->child("content_url", "KURLLabel");

      if(fieldContentText == NULL || urlBox == NULL || fieldContentURL == NULL) continue;

      if(!fieldContentText->text().isEmpty())
        fieldStr += (" " + fieldContentText->text());
      else if (!fieldContentURL->text().isEmpty()) {
        fieldStr += (" " + fieldContentURL->text());

        QLabel *fieldAux = (QLabel *) fieldBox->child("aux", "QLabel");

        if(fieldAux == NULL || fieldAux->text().isEmpty())
          fieldStr += (" " + fieldAux->text());
      }

      out << fieldStr;
    } while (names.prev());
  }

  return out;
}

void KSetiSpyView::textToClipboard()
{
  QStringList fields = text();
  QString out;

  if(fields.isEmpty()) return;

  for(QStringList::Iterator it = fields.begin(); it != fields.end(); ++it)
    out += (*it) + "\n";

  KApplication::clipboard()->setText(out);
}

void KSetiSpyView::pixmapToClipboard()
{
  KApplication::clipboard()->setPixmap(pixmap);
}

void KSetiSpyView::pixmapToFile()
{
  const QString caption = i18n("Save Picture");
  const QString filter = QString("*.bmp") + "|" + i18n("Bitmap file") + "\n" +
                         QString("*.png") + "|" + i18n("PNG file") + "\n" +
                         QString("*.jpg") + "|" + i18n("JPEG file");

  KFileDialog fileDialog(QString::null, filter, this, "save", true);
  fileDialog.setCaption(caption);
  fileDialog.setKeepLocation(true);

  if(fileDialog.exec() != 0)
  {
    QString file_name = fileDialog.selectedFile();
    QString file_filter = fileDialog.currentFilter();
    QString file_type = (  file_filter == "*.bmp"  ) ? "BMP"
                      : (  file_filter == "*.png"  ) ? "PNG"
                      : /* file_filter == "*.jpg" */   "JPEG";

    if(!QFile(file_name).exists()
       || QMessageBox::warning(this, caption, i18n("File exists. Overwrite?"),
                               QMessageBox::Ok | QMessageBox::Default, QMessageBox::Cancel) == QMessageBox::Ok)
      pixmap.save(file_name, file_type);
  }
}

void KSetiSpyView::updateContent(bool)
{
}

void KSetiSpyView::updateContent()
{
  updateContent(false);
}

#include "ksetispyview.moc"

