/***************************************************************************
                          performanceview.cpp  -  description
                             -------------------
    begin                : Wed May 30 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qcheckbox.h>
#include <qlayout.h>
#include <qwidgetstack.h>

#include <kapplication.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpushbutton.h>

#include "performanceview.h"

PerformanceView::PerformanceView(QWidget *parent, const char *name)
               : KSetiSpyView(Text, parent, name)
{
  QBoxLayout *layout = new QBoxLayout(this, QBoxLayout::TopToBottom);
  layout->setSpacing(8);

  layout->addWidget(addField(this, "PerformanceView::rate"));
  setFieldName("PerformanceView::rate", i18n("Processing rate:"));

  layout->addWidget(addField(this, "PerformanceView::speed"));
  setFieldName("PerformanceView::speed", i18n("Processing speed:"));

  layout->addWidget(addField(this, "PerformanceView::efficiency"));
  setFieldName("PerformanceView::efficiency", i18n("Processing efficiency:"));

  layout->addWidget(addField(this, "PerformanceView::flops"));
  setFieldName("PerformanceView::flops", i18n("Floating point operations to process:"));

  QCheckBox *instantCheck = new QCheckBox(i18n("&Show instantaneous performance"), this, "PerformanceView::instant");
  instantCheck->installEventFilter(this);
  layout->addWidget(instantCheck);
  connect(instantCheck, SIGNAL(toggled(bool)), this, SLOT(handleInstantCheck(bool)));

  layout->addStretch(1);

  QBoxLayout *hbox = new QBoxLayout(layout, QBoxLayout::RightToLeft);

  KPushButton *button = new KPushButton(i18n("&Compare"), this);
  button->installEventFilter(this);
  hbox->addWidget(button);

  hbox->addStretch(1);

  performanceDialog = new PerformanceDialog();
  connect(instantCheck, SIGNAL(toggled(bool)), performanceDialog, SLOT(setInstant(bool)));
  connect(button, SIGNAL(clicked()), performanceDialog, SLOT(show()));

  old.wu_name = current.wu_name = QString::null;
}

PerformanceView::~PerformanceView()
{
}

void PerformanceView::readConfig(bool readGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("Performance");

  QCheckBox *instantCheck = (QCheckBox *) child("PerformanceView::instant", "QCheckBox");
  instantCheck->setChecked(config->readBoolEntry("Instantaneous"));

  performanceDialog->readConfig(readGeometry);
  performanceDialog->setInstant(instantCheck->isChecked());
}

void PerformanceView::saveConfig(bool saveGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("Performance");

  QCheckBox *instantCheck = (QCheckBox *) child("PerformanceView::instant", "QCheckBox");
  config->writeEntry("Instantaneous", instantCheck->isChecked());

  performanceDialog->saveConfig(saveGeometry);
}

void PerformanceView::updateContent(bool force)
{
  const SetiClientMonitor::State state = kdoc->setiMonitorState();
  SetiClientMonitor *monitor = kdoc->setiMonitor();

  if(state >= SetiClientMonitor::Idle)
  {
    const seti_data *data = monitor->setiData();

    if(data->wu.name != current.wu_name || data->state.progress < current.progress) {
      old.wu_name = QString::null;
      current.wu_name = data->wu.name;
      current.cpu = data->state.cpu;
      current.progress = data->state.progress;
    } else if(data->state.progress > current.progress) {
      old = current;
      current.cpu = data->state.cpu;
      current.progress = data->state.progress;
    }
  }

  // optimization: do not update if this is not the visible widget
  const QWidgetStack *views = (QWidgetStack *) this->parent();
  if(views->visibleWidget() != this && !force) return;

  QCheckBox *instantCheck = (QCheckBox *) child("PerformanceView::instant", "QCheckBox");
  const bool instant = instantCheck->isChecked();

  KLocale *locale = KGlobal::locale();

  if(state == SetiClientMonitor::Running && !(instant && old.wu_name.isEmpty()))
  {
    const seti_data *data = monitor->setiData();
    const double teraFLOPs = SetiDataMonitor::teraFLOPs(*data);
    const sys_info *info = monitor->sysInfo();
    const int cpu = monitor->currentCPU();

    const double secs = instant ? (current.cpu - old.cpu) : current.cpu;
    const double done = instant ? (current.progress - old.progress) : current.progress;

    const double rate = (secs > 0.0) ? 1e2 * (done/secs) * 60.0 * 60.0 : 0.0;
    const double rate_inv = (rate > 0.0) ? 1e2/rate : 0.0;
    setFieldContentText("PerformanceView::rate",
                        i18n("%1 hour / work unit (%2% / hour)").arg(locale->formatNumber(rate_inv, 2))
                                                                .arg(locale->formatNumber(rate, 2)));

    const double speed = (secs > 0.0) ? 1e6 * teraFLOPs * (done/secs) : 0.0;
    setFieldContentText("PerformanceView::speed",
                        i18n("%1 MegaFLOPs / second").arg(locale->formatNumber(speed, 2)));

    if(info != NULL)
    {
      const double efficiency = (speed > 0.0) ? info->cpus[cpu].MHz/speed : 0.0;
      setFieldContentText("PerformanceView::efficiency",
                          i18n("%1 CPU cycles / FLOP").arg(locale->formatNumber(efficiency, 2)));
    }
    else
      setFieldContentText("PerformanceView::efficiency", i18n(unknownContent));
  }
  else
  {
    setFieldContentText("PerformanceView::rate", i18n(unknownContent));
    setFieldContentText("PerformanceView::speed", i18n(unknownContent));
    setFieldContentText("PerformanceView::efficiency", i18n(unknownContent));
  }

  if(state >= SetiClientMonitor::Idle) {
    const seti_data *data = monitor->setiData();
    const double teraFLOPs = SetiDataMonitor::teraFLOPs(*data);

    setFieldContentText("PerformanceView::flops",
                        i18n("%1 TeraFLOPs").arg(locale->formatNumber(teraFLOPs, 2)));
  } else
    setFieldContentText("PerformanceView::flops", i18n(unknownContent));
}

void PerformanceView::handleInstantCheck(bool)
{
  updateContent(true);
}

#include "performanceview.moc"

