/***************************************************************************
                          resultsplotview.cpp  -  description
                             -------------------
    begin                : Sun Jun 24 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>

#include <qpainter.h>
#include <qbitmap.h>
#include <qstringlist.h>

#include <kiconloader.h>
#include <klocale.h>

#include "resultsplotview.h"

const QSize ResultPlotMinSize = QSize(150, 80);
const int ResultPlotMargin = 3;
const int ResultPlotFontSize = 9;
const int ResultPlotArrowLength = 30;

const QString ResultPlotValues = "-50 -10 -3 0 3 10 50";

ResultsPlotView::ResultsPlotView(QWidget *parent, const char *name)
                : KSetiSpyView(Pixmap, parent, name)
{
  QFont font = this->font();
  font.setPointSize(ResultPlotFontSize);
  setFont(font);
}

ResultsPlotView::~ResultsPlotView()
{
}

void ResultsPlotView::updateContent(bool)
{
  repaint();
}

void ResultsPlotView::paintEvent(QPaintEvent *)
{
  QFontMetrics metrics = this->fontMetrics();
  const int lineHeight = metrics.lineSpacing();

  QPainter painter(this);

  painter.save();
  painter.translate(2*lineHeight, 0);
  {
    const QRect r(0, 0, this->width() - 2*lineHeight, lineHeight);

    painter.setPen(black);

    painter.drawText(r, AlignCenter, i18n("Signal Strength"));
  }
  painter.restore();

  painter.save();
  painter.translate(2*lineHeight, this->height() - 2*lineHeight);
  {
    const QRect r(0, 0, this->width() - 2*lineHeight, 2*lineHeight);

    painter.setPen(black);

    QStringList labels = QStringList::split(' ', ResultPlotValues);
    const uint n_labels = labels.count();
    const int lineBase = metrics.height();
    for(uint i = 0; i < n_labels; i++)
      painter.drawText((r.width() - metrics.width(labels[i])) * i/(n_labels-1), lineBase, labels[i]);

    painter.drawText(r, AlignHCenter | AlignBottom, i18n("Drift Rate (Hz/s)"));
  }
  painter.restore();

  painter.save();
  painter.translate(0, this->height() - 2*lineHeight);
  painter.rotate(-90);
  {
    const QRect r(0, 0, this->height() - 3*lineHeight, 2*lineHeight);

    painter.setPen(black);

    painter.drawText(r, AlignHCenter | AlignTop, i18n("Strength"));

    const int arrowStart = (r.width() - ResultPlotArrowLength)/2;
    const int arrowEnd = arrowStart + ResultPlotArrowLength;
    const int arrowY = lineHeight + lineHeight/2;

    painter.drawLine(arrowStart, arrowY, arrowEnd, arrowY);
    painter.drawLine(arrowEnd, arrowY, arrowEnd-3, arrowY-3);
    painter.drawLine(arrowEnd, arrowY, arrowEnd-3, arrowY+3);
  }
  painter.restore();

  QPixmap plot(QSize(this->width() - 2*lineHeight - 2*ResultPlotMargin, this->height() - 3*lineHeight));
  {
    QPainter painter(&plot);
    const QColor blue(0, 0, 127);
    const QColor gray(191, 191, 191);

    painter.fillRect(plot.rect(), black);
    painter.fillRect(0, 0, plot.width(), plot.height()/2, blue);

    painter.setPen(gray);
    painter.setFont(this->font());

    const QRect textRect(3, 3, plot.width()-2*3, plot.height()-3*3);
    painter.drawText(textRect, AlignLeft | AlignTop, i18n("Returned"));
    painter.drawText(textRect, AlignRight | AlignBottom, i18n("Not returned"));

    SetiClientMonitor *monitor = kdoc->setiMonitor();
    const SetiClientMonitor::State state = (monitor != NULL) ? monitor->currentState()
                                                           : SetiClientMonitor::No_Data;

    if(state >= SetiClientMonitor::Idle)
    {
      const seti_data *data = kdoc->setiMonitor()->setiData();

      KIconLoader loader;

      QPixmap spikeIcon = loader.loadIcon(SpikeIcon, KIcon::User);
      if(data->state.best_spike.score > 0.0)
      {
        spike_summary spike = monitor->toSummary(data->state.best_spike);

        paintResult(spike.chirp_rate, power(spike), spikeIcon, &plot);
      }
      for(uint i = 0; i < data->output.spikes.count(); i++)
      {
        spike_summary spike = data->output.spikes[i];

        paintResult(spike.chirp_rate, power(spike), spikeIcon, &plot);
      }

      QPixmap gaussianIcon = loader.loadIcon(GaussianIcon, KIcon::User);
      QPixmap interestingGaussianIcon = loader.loadIcon(InterestingGaussianIcon, KIcon::User);
      if(data->state.best_gaussian.score > 0.0)
      {
        gaussian_summary gaussian = monitor->toSummary(data->state.best_gaussian);

        if(SetiClientMonitor::isInteresting(gaussian))
          paintResult(gaussian.chirp_rate, power(gaussian), interestingGaussianIcon, &plot);
        else
          paintResult(gaussian.chirp_rate, power(gaussian), gaussianIcon, &plot);
      }
      for(uint i = 0; i < data->output.gaussians.count(); i++)
      {
        gaussian_summary gaussian = data->output.gaussians[i];

        if(SetiClientMonitor::isInteresting(gaussian))
          paintResult(gaussian.chirp_rate, power(gaussian), interestingGaussianIcon, &plot);
        else
          paintResult(gaussian.chirp_rate, power(gaussian), gaussianIcon, &plot);
      }

      QPixmap pulseIcon = loader.loadIcon(PulseIcon, KIcon::User);
      if(data->state.best_pulse.score > 0.0)
      {
        pulse_summary pulse = monitor->toSummary(data->state.best_pulse);

        paintResult(pulse.chirp_rate, power(pulse), pulseIcon, &plot);
      }
      for(uint i = 0; i < data->output.pulses.count(); i++)
      {
        pulse_summary pulse = data->output.pulses[i];

        paintResult(pulse.chirp_rate, power(pulse), pulseIcon, &plot);
      }

      QPixmap tripletIcon = loader.loadIcon(TripletIcon, KIcon::User);
      if(data->state.best_triplet.score > 0.0)
      {
        triplet_summary triplet = monitor->toSummary(data->state.best_triplet);

        paintResult(triplet.chirp_rate, power(triplet), tripletIcon, &plot);
      }
      for(uint i = 0; i < data->output.triplets.count(); i++)
      {
        triplet_summary triplet = data->output.triplets[i];

        paintResult(triplet.chirp_rate, power(triplet), tripletIcon, &plot);
      }

      QPixmap driftIcon(1, 2*plot.height());
      driftIcon.fill(red);
      paintResult(data->state.cr, 0.5, driftIcon, &plot);
    }
  }
  painter.drawPixmap(2*lineHeight + ResultPlotMargin, lineHeight + 3, plot);
}

QSize ResultsPlotView::sizeHint() const
{
  const int lineHeight = this->fontMetrics().lineSpacing();

  return(QSize(ResultPlotMinSize.width() + 2*lineHeight + 2*ResultPlotMargin,
               ResultPlotMinSize.height() + 3*lineHeight));
}

QSizePolicy ResultsPlotView::sizePolicy() const
{
  return(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum));
}

double ResultsPlotView::power(spike_summary spike) const
{
  return(2.877 * log10(SetiClientMonitor::signalRatio(spike)) - 3.3692);
}

double ResultsPlotView::power(gaussian_summary gaussian) const
{
  return(0.6325 * log10(SetiClientMonitor::signalRatio(gaussian)) + 0.176);
}

double ResultsPlotView::power(pulse_summary pulse) const
{
  return(1.683 * log10(SetiClientMonitor::score(pulse)) + 0.5);
}

double ResultsPlotView::power(triplet_summary triplet) const
{
  return(0.72488 * log10(SetiClientMonitor::score(triplet)) - 0.151);
}

void ResultsPlotView::paintResult(double chirp_rate, double power, const QPixmap& target, QPixmap *plot)
{
  chirp_rate = (chirp_rate > 50.0) ? 50.0 : chirp_rate;
  chirp_rate = (chirp_rate < -50.0) ? -50.0 : chirp_rate;

  const double d1 = log10((chirp_rate > 0) ? chirp_rate + 1.0 : 1.0 - chirp_rate);
  const double d2 =  (0.03131 * d1 + 0.23888) * d1;

  const int xpos = (chirp_rate > 0) ? int(plot->width() * (0.5 + d2)) : int(plot->width() * (0.5 - d2));

  power = (power > 1.0) ? 1.0 : power;
  power = (power < 0.0) ? 0.0 : power;

  const int ypos = int(plot->height() * (1.0 - power)) + 1;

  QPainter(plot).drawPixmap(xpos - target.width()/2, ypos - target.height()/2, target);
}

#include "resultsplotview.moc"
