/***************************************************************************
                          seticlientmonitor.cpp  -  description
                             -------------------
    begin                : Tue Jul 10 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qxml.h>

#include <kapplication.h>
#include <klocale.h>
#include <kio/netaccess.h>
#include <kprocess.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdeversion.h>

#include "csvlogmonitor.h"
#include "ksetispydoc.h"
#include "setispylogmonitor.h"
#include "signalplot.h"

#include "seticlientmonitor.h"

const QString pid_file = "pid.sah";

const QString CalibrationXML = "ksetispy/xml/calibration.xml";

QValueList<calibrationDataStruct> SetiClientMonitor::presets;

SetiClientMonitor::SetiClientMonitor(const KURL& setiURL,
                                     const KURL& sysInfoURL,
                                     const KURL& logURL, bool usesSETISpyLog,
                                     QObject *parent, const char *name)
                  : SetiDataMonitor(setiURL, parent, name)
{
  addFile(pid_file, pid_index);

  system = new SysInfoMonitor(sysInfoURL, this);
  cpu = 0;

  realAddress = data.user.email;
  defaultAddress = QString::null;

  calibrationData.cpu = QString::null;
  calibrationData.low.clear();
  calibrationData.medium.clear();
  calibrationData.high.clear();

  backup.wu.name = QString::null;

  log = usesSETISpyLog ? (LogMonitor *) new SETISpyLogMonitor(logURL) : (LogMonitor *) new CSVLogMonitor(logURL);
  connect(log, SIGNAL(updated()), this, SLOT(handleLogUpdates()));

  writesSETISpy = writesCSV = false;

  const imageLogStruct defaultImageLog = { imageLogStruct::None, 2.0, imageLogStruct::PNG, imageLogStruct::Default, QString::null };
  bestGaussian = returnedGaussians = defaultImageLog;

  cmd[Regular] = cmd[VLAR] = args = QString::null;

  liveness = false;

  persistence = false;
  bootstrapped = false;
  cached = false;

  local = isRunning();
  if(local && state == Idle && !isOK(result_index))
    state = Running;
}

SetiClientMonitor::~SetiClientMonitor()
{
}

void SetiClientMonitor::setInterval(int secs)
{
  if(log != NULL)
    log->setInterval(secs);

  SetiDataMonitor::setInterval(secs);
}

const sys_info *SetiClientMonitor::sysInfo()
{
  return(system->sysInfo());
}

KURL SetiClientMonitor::sysInfoURL() const
{
  return(system->url());
}

int SetiClientMonitor::currentCPU()
{
  return((system->sysInfo() != NULL) ? cpu : -1);
}


QString SetiClientMonitor::defaultEMail() const
{
  KSetiSpyDoc *kdoc = (KSetiSpyDoc *) this->parent();
  return(cached ? kdoc->getCacheManager()->getEMail() : defaultAddress);
}

calibrationDataStruct SetiClientMonitor::calibration() const
{
  return(calibrationData);
}

void SetiClientMonitor::loadCalibrationPresets()
{
  CalibrationDataReader handler(presets);
  QFile xmlFile(kapp->dirs()->findResource("data", CalibrationXML));
  QXmlInputSource source(xmlFile);
  QXmlSimpleReader reader;
  reader.setContentHandler(&handler);
  reader.parse(&source);
}

const QValueList<calibrationDataStruct>& SetiClientMonitor::calibrationPresets()
{
  return(presets);
}

const QValueList<log_data> SetiClientMonitor::logData()
{
  return(log->logData());
}

bool SetiClientMonitor::usesSETISpyLog()
{
  return(log->isA("SETISpyLogMonitor"));
}

bool SetiClientMonitor::writesSETISpyLog()
{
  return(writesSETISpy);
}

bool SetiClientMonitor::writesCSVLog()
{
  return(writesCSV);
}

imageLogStruct SetiClientMonitor::bestGaussianLog() const
{
  return(bestGaussian);
}

imageLogStruct SetiClientMonitor::returnedGaussiansLog() const
{
  return(returnedGaussians);
}

KURL SetiClientMonitor::logURL() const
{
  return(log->url());
}

KURL SetiClientMonitor::command(ClientType type) const
{
  if(cmd[type].isEmpty() && cached)
  {
    KSetiSpyDoc *kdoc = (KSetiSpyDoc *) parent();
    return(kdoc->getCacheManager()->getClientURL());
  }
  
  return(cmd[type]);
}

QString SetiClientMonitor::arguments() const
{
  return(args);
}

QString SetiClientMonitor::filteredArguments()
{
  if(!cached) return(args);

  QString result = args;
  result.replace(QRegExp("-stop_after_process"), "");
  result.replace(QRegExp("-stop_after_xfer"), "");
  result += " -stop_after_process";

  return(result);
}

bool SetiClientMonitor::isRunning()
{
  if(!isOK(pid_index) || pid <= 0) return(false);

  bool read_ok;
  QStringList lines = readFile(QString("/proc/%1/cmdline").arg(pid), read_ok);

  if(!read_ok || lines.count() == 0) return(false);

  return(lines[0].contains(command().fileName()));
}

int SetiClientMonitor::getPriority()
{
  return(isRunning() ? getpriority(PRIO_PROCESS, pid) : 0);
}

bool SetiClientMonitor::launchOnStartup() const
{
  return(bootstrapped);
}

bool SetiClientMonitor::killOnExit() const
{
  return(!persistence);
}

bool SetiClientMonitor::keepAlive() const
{
  return(liveness);
}

bool SetiClientMonitor::usesCache() const
{
  return(cached);
}

void SetiClientMonitor::updateSysInfo()
{
  system->checkFiles();
  emit updated(this);
}

void SetiClientMonitor::setCurrentCPU(int index)
{
  if(index >= 0 && system->sysInfo() != NULL && index < int(system->sysInfo()->cpus.count()))
  {
    cpu = index;
    emit updated(this);
  }
}

void SetiClientMonitor::setDefaultEMail(const QString& address)
{
  if(cached) return;

  defaultAddress = address;
  if(realAddress.isEmpty())
    data.user.email = defaultAddress;
}

void SetiClientMonitor::setCalibration(int preset)
{
  if(preset < 0 || preset >= int(presets.count())) return;
  calibrationData = presets[preset];
  data.state.progress = calibrate(data.state.progress, data.wu.angle_range);
}

void SetiClientMonitor::setCalibration(const calibrationDataStruct& custom)
{
  calibrationData = custom;
  data.state.progress = calibrate(data.state.progress, data.wu.angle_range);
}

void SetiClientMonitor::useSETISpyLog(bool use)
{
  if(use == usesSETISpyLog())
    return;

  KURL url = logURL();

  if(log != NULL)
  {
    log->setInterval(0);
    disconnect(log, 0, 0, 0);
    delete log;
  }

  log = use ? (LogMonitor *) new SETISpyLogMonitor(url) : (LogMonitor *) new CSVLogMonitor(url);
  connect(log, SIGNAL(updated()), this, SLOT(handleLogUpdates()));
  log->setInterval(getInterval());

  emit logUpdated(this);
}

void SetiClientMonitor::writeSETISpyLog(bool write)
{
  writesSETISpy = write;
}

void SetiClientMonitor::writeCSVLog(bool write)
{
  writesCSV = write;
}

void SetiClientMonitor::setLogURL(const KURL& url)
{
  bool isSETISpyLog = usesSETISpyLog();

  if(log != NULL)
  {
    disconnect(log, 0, 0, 0);
    delete log;
  }

  log = isSETISpyLog ? (LogMonitor *) new SETISpyLogMonitor(url)
                     : (LogMonitor *) new CSVLogMonitor(url);
  connect(log, SIGNAL(updated()), this, SLOT(handleLogUpdates()));
  log->setInterval(getInterval());
}

void SetiClientMonitor::setBestGaussianLog(const imageLogStruct& log)
{
  bestGaussian = log;
}

void SetiClientMonitor::setReturnedGaussiansLog(const imageLogStruct& log)
{
  returnedGaussians = log;
}

void SetiClientMonitor::setCommand(const KURL& command, ClientType type)
{
  cmd[type] = checkCommand(command) ? command : QString::null;
}

void SetiClientMonitor::setArguments(const QString& arguments)
{
  args = checkCommand(cmd[Regular]) ? arguments : QString::null;
}

void SetiClientMonitor::exec()
{
  if(cached && state < Idle) {
    KSetiSpyDoc *kdoc = (KSetiSpyDoc *) parent();
    if(kdoc->getCacheManager()->fetchWU(url().path(+1))) {
      updateData();
      QTimer::singleShot(2000, this, SLOT(checkState()));
    }
  }

  KURL executable = command();
                    
  if(executable.isEmpty() || isRunning())
    return;

  if(data.wu.angle_range < 0.1 && !command(VLAR).isEmpty())
    executable = command(VLAR);

  if(url().isLocalFile())
    QDir::setCurrent(url().path(+1));
  else
    // this is useful to start/stop clients remotely via ssh and such
    QDir::setCurrent(executable.directory());

  KShellProcess process;
  process << executable.path(-1);

  QString args = filteredArguments();
  if(!args.isEmpty()) process << args;

  process.start(KProcess::DontCare);

  QTimer::singleShot(1000, this, SLOT(updateData()));
}

void SetiClientMonitor::kill()
{
  if(isRunning())
  {
    ::kill(pid_t(pid), SIGKILL);

    local = true;

    QTimer::singleShot(1000, this, SLOT(updateData()));
  }
}

void SetiClientMonitor::setPriority(int priority)
{
  if(!isRunning()) return;

  if(priority > 20) priority = 20;
  if(priority < -20) priority = -20;
  setpriority(PRIO_PROCESS, pid, priority);
}

void SetiClientMonitor::setLaunchOnStartup(bool set)
{
  if(set && !url().isLocalFile()) return;

  bootstrapped = set;
}

void SetiClientMonitor::setKillOnExit(bool set)
{
  if(set && !url().isLocalFile()) return;

  persistence = !set;
}

void SetiClientMonitor::setKeepAlive(bool set)
{
  if(set && !url().isLocalFile()) return;

  liveness = set;
}

void SetiClientMonitor::useCache(bool use)
{
  if(cached == use) return;

  if(use && !url().isLocalFile()) return;

  cached = use;
  if(cached) {
    KSetiSpyDoc *kdoc = (KSetiSpyDoc *) parent();
    defaultAddress = kdoc->getCacheManager()->getEMail();
  }
  if(isRunning()) {
    kill();
    QTimer::singleShot(1000, this, SLOT(exec()));
  }
}

bool SetiClientMonitor::parseFile(int index, const QString& fileName)
{
  if(index == pid_index)
    return(parsePIDFile(fileName));
  else
  {
    bool ok = SetiDataMonitor::parseFile(index, fileName);

    if(ok && index == user_info_index)
    {
      realAddress = data.user.email;
      if(realAddress.isEmpty())
        data.user.email = defaultEMail();
    }
    else if(ok && index == state_index && isOK(work_unit_index))
      data.state.progress = calibrate(data.state.progress, data.wu.angle_range);

    return(ok);
  }
}

void SetiClientMonitor::updateData()
{
  bool running = isRunning();

  if(running)
  {
    if(!isOK(result_index))
      lastModified = QDateTime::currentDateTime();
  }
  else if(local)
  {
    const int secs = lastModified.secsTo(QDateTime::currentDateTime()) - clientTimeOut();
    if(secs < 0) lastModified = lastModified.addSecs(secs);
  }

  local = running;

  SetiDataMonitor::updateData();

  if(state >= Idle)
  {
    bool done = isOK(result_index);

    if(backup.wu.name.isEmpty())
    {
      if(!done)
      {
        backup = data;
        if(data.state.best_gaussian.score > 0) {
          logGaussian(toSummary(data.state.best_gaussian), bestGaussian, data.wu.name + "." + i18n("best"));
          if(isInteresting(data.state.best_gaussian))
            emit notify(this, InterestingGaussian);
        }
      }
    }
    else if(backup.wu.name != data.wu.name)
    {
      logWorkUnit(backup);
      emit notify(this, WUCompleted);
      backup = data;
      if(data.state.best_gaussian.score > 0) {
        logGaussian(toSummary(data.state.best_gaussian), bestGaussian, data.wu.name + "." + i18n("best"));
        if(isInteresting(data.state.best_gaussian))
          emit notify(this, InterestingGaussian);
      }
    }
    else
    {
      if(done)
      {
        backup.wu.name = QString::null;
        logWorkUnit(data);
        emit notify(this, WUCompleted);
      }
      else
      {
        if(backup.state.best_gaussian.score < data.state.best_gaussian.score)
	{
          logGaussian(toSummary(data.state.best_gaussian), bestGaussian, data.wu.name + "." + i18n("best"));
          if(isInteresting(data.state.best_gaussian))
            emit notify(this, InterestingGaussian);
        }
        backup = data;
      }
    }

    if(cached)
    {
      KSetiSpyDoc *kdoc = (KSetiSpyDoc *) parent();

      if(done)
      {
        kill(); // in case it didn't die of natural causes
        if(kdoc->getCacheManager()->storeWU(url().path(+1))) {
          checkFiles();
          QTimer::singleShot(1000, this, SLOT(exec()));
        }
      }
      else if(state < Idle)
        QTimer::singleShot(1000, this, SLOT(exec()));
    }
  }
  if(liveness && !running)
    QTimer::singleShot(1000, this, SLOT(exec()));
}

void SetiClientMonitor::checkState()
{
  if(local) {
    if(state == Running
       && lastModified.secsTo(QDateTime::currentDateTime()) >= clientTimeOut()
       && !isRunning())
    {
      updateData();
      emit updated(this);
    }
  } else
    SetiDataMonitor::checkState();
}

bool SetiClientMonitor::parsePIDFile(const QString& fileName)
{
  bool read_ok;

  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  pid = (lines.count() > 0) ? lines[0].toInt() : 0;
  return(true);
}

bool SetiClientMonitor::checkCommand(const KURL& cmd)
{
  if(!url().isLocalFile())
    return(false);

  if(!cmd.isValid() || !cmd.isLocalFile())
    return(false);

  QFileInfo fileInfo(cmd.path(-1));
  return(fileInfo.exists() && fileInfo.isFile() && fileInfo.isExecutable());
}

double SetiClientMonitor::calibrate(double progress, double angle_range)
{
  if(state < Idle) return(-1.0);

  calibrationItemStruct min = {0.0, 0.0};
  calibrationItemStruct max = {1.0, 1.0};
  QValueList<calibrationItemStruct> list = (angle_range < 0.2255) ? calibrationData.low
                                         : (angle_range > 1.1274) ? calibrationData.high
                                                                  : calibrationData.medium;

  for(uint i = 0; i < list.count(); i++)
  {
    calibrationItemStruct item = list[i];

    if(progress >= list[i].reported && list[i].reported > min.reported)
      min = list[i];
    if(progress < list[i].reported && list[i].reported < max.reported)
      max = list[i];
  }

  if(max.reported == min.reported) return(0.0);
  return(min.actual + (max.actual - min.actual)/(max.reported - min.reported) * (progress - min.reported));
}

void SetiClientMonitor::logWorkUnit(const seti_data& data)
{
  // write to the WU log(s)

  if(writesSETISpy && log != NULL)
    SETISpyLogMonitor::writeSetiData(log->url(), data);
  if(writesCSV && log != NULL)
    CSVLogMonitor::writeSetiData(log->url(), data);

  // log all returned gaussians

  for(uint i = 0; i < data.output.gaussians.count(); i++)
    logGaussian(data.output.gaussians[i], returnedGaussians, data.wu.name + "." + i18n("returned"));
}

void SetiClientMonitor::logGaussian(const gaussian_summary& gaussian, const imageLogStruct& log, const QString& prefix)
{
  // first make sure it satisfies the filter
  const double snr = gaussian.peak / gaussian.mean;

  switch(log.filter) {
    case imageLogStruct::None:
      return;
    case imageLogStruct::Interesting:
      if(!isInteresting(gaussian)) return;
      break;
    case imageLogStruct::SNRAbove:
      if(snr <= log.threshold) return;
      break;
    default:
      break;
  }

  // then find a suitable name

  QString pattern = prefix + "%1.";
  QString fileType;

  switch(log.format) {
    case imageLogStruct::Bitmap:
      pattern += "bmp";
      fileType = "BMP";
      break;
    case imageLogStruct::PNG:
      pattern += "png";
      fileType = "PNG";
      break;
    default:
      pattern += "jpg";
      fileType = "JPEG";
  }

  int index = 0;
  KURL fileURL;

  while(true) {
    fileURL = log.url;
    fileURL.addPath(pattern.arg(index++));
    
    #if KDE_IS_VERSION(3,2,0)
    if(!KIO::NetAccess::exists(fileURL, false, qApp->mainWidget()));
    #else
    if(!KIO::NetAccess::exists(fileURL, false));
    #endif
      break;
  }

  // create a pixmap of the signal with the right size and save it

  SignalPlot signal;

  signal.hide();
  if(log.size == imageLogStruct::Default) signal.resize(DefaultPlotSize);
  else signal.resize(SetiPlotSize);
  signal.setData(gaussian);

  QPixmap pixmap = QPixmap::grabWidget(&signal);

  KTempFile fileTemp;
  fileTemp.setAutoDelete(true);
  
  QString fileName = fileURL.isLocalFile() ? fileURL.path(-1) : fileTemp.name();
  
  pixmap.save(fileName, fileType);
  if(!fileURL.isLocalFile())
  #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::upload(fileName, fileURL, qApp->mainWidget());
  #else
    KIO::NetAccess::upload(fileName, fileURL);
  #endif
}

void SetiClientMonitor::handleLogUpdates()
{
  emit logUpdated(this);
}

#include "seticlientmonitor.moc"

