/***************************************************************************
                          setidatamonitor.cpp  -  description
                             -------------------
    begin                : Sun Jun 17 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>

#include <qfileinfo.h>
#include <qregexp.h>

#include <klocale.h>
#include <krfcdate.h>

#include "setidatamonitor.h"

const double WorkUnitBandwidth = 9765.625;
const double MaxWUSize = 366720.0;

SetiDataMonitor::SetiDataMonitor(const KURL& setiURL, QObject *parent, const char *name)
                : DataMonitor(setiURL, parent, name)
{
  timeout = 120;

  timer = new QTimer(this);
  connect(timer, SIGNAL(timeout()), this, SLOT(checkState()));
  timer->start(timeout * 1000);

  addFile(version_file, version_index);
  addFile(user_info_file, user_info_index);
  addFile(work_unit_file, work_unit_index);
  addFile(state_file, state_index);
  addFile(output_file, output_index);
  addFile(result_file, result_index);
  addFile(wtemp_file, wtemp_index);

  lastModified = exists(state_index) ? timestamp(state_index)
                                     : QDateTime::currentDateTime();

  if(isOK(version_index) && isOK(user_info_index))
    if(isOK(work_unit_index))
      if(lastModified.secsTo(QDateTime::currentDateTime()) >= timeout)
        state = Idle;
      else
        state = Running;
    else if(isOK(result_index))
      state = Idle;
    else if(!exists(result_index) && isOK(wtemp_index))
      state = Loading;
    else
      state = No_WU;
  else
    state = No_Data;
}

SetiDataMonitor::~SetiDataMonitor()
{
}

int SetiDataMonitor::clientTimeOut() const
{
  return(timeout);
}

SetiDataMonitor::State SetiDataMonitor::currentState() const
{
  return(state);
}

const seti_data *SetiDataMonitor::setiData()
{
  return((state != No_Data) ? &data : NULL);
}

double SetiDataMonitor::percentLoaded() const
{
  return((state < Loading) ? 0.0 : (state > Loading) ? 100.0 : 100.0 * loaded);
}

double SetiDataMonitor::teraFLOPs(const seti_data& data)
{
  if(data.wu.angle_range < 0.2255)
    return(3.54 * exp(data.wu.angle_range * 0.0327));
  if(data.wu.angle_range > 1.1274)
    return(3.37 * pow(data.wu.angle_range, -0.0065));
  return(3.74 * pow(data.wu.angle_range, -0.1075));
}

QString SetiDataMonitor::timeToString(double secs)
{
  const int h = int(secs / 3600.0);
  const int m = int((secs - h * 3600.0) / 60.0);
  const int s = int(secs - h * 3600.0 - m * 60.0);

  return(QString().sprintf("%d:%.2d:%.2d", h, m, s));
}

QString SetiDataMonitor::raToString(double ra)
{
  const int h = int(ra) % 24;
  ra -= int(ra); ra *= 60;
  const int m = int(ra);
  ra -= m; ra *= 60;
  const int s = int(ra);
  
  return(i18n("%1 h %2' %3\"").arg(h).arg(m).arg(s));
}

QString SetiDataMonitor::decToString(double dec, bool sign)
{
  const QString p = (dec < 0.0) ? QString("-") : sign ? QString("+") : QString::null;
  dec = (dec < 0.0) ? -dec : dec;
  const int d = int(dec);
  dec -= d; dec *= 60;
  const int m = int(dec);
  dec -= m; dec *= 60;
  const int s = int(dec);

  return(QString("%1%2\xb0 %3' %4\"").arg(p).arg(d).arg(m).arg(s));
}

void SetiDataMonitor::setClientTimeOut(int secs)
{
  timeout = (secs > 0) ? secs : timeout;
  timer->changeInterval(timeout * 1000);
}

bool SetiDataMonitor::parseFile(int index, const QString& fileName)
{
  if(index == version_index)
    return(parseVersionFile(fileName));
  else if(index == user_info_index)
    return(parseUserInfoFile(fileName));
  else if(index == work_unit_index)
    return(parseWorkUnitFile(fileName));
  else if(index == state_index)
    return(parseStateFile(fileName));
  else if(index == output_index)
    return(parseOutputFile(fileName));
  else if(index == wtemp_index)
    return(parseWUTempFile(fileName));
  else if(index == result_index) {
    const bool ok = parseWorkUnitFile(fileName);
    if(ok) data.state.progress = 1.0;
    return(ok);
  } else
    return(DataMonitor::parseFile(index, fileName));
}

void SetiDataMonitor::updateData()
{
  State old_state = state;

  if(isOK(version_index) && isOK(user_info_index))
    if(isOK(work_unit_index))
      if(lastModified.secsTo(QDateTime::currentDateTime()) >= timeout)
        state = Idle;
      else
        state = Running;
    else if(isOK(result_index))
      state = Idle;
    else if(!exists(result_index) && isOK(wtemp_index))
      state = Loading;
    else
      state = No_WU;
  else
    state = No_Data;

  lastModified = QDateTime::currentDateTime();

  if(old_state != state || state == Loading || state == Running)
    emit updated(this);
}

void SetiDataMonitor::checkState()
{
  if(state == Running
     && lastModified.secsTo(QDateTime::currentDateTime()) >= timeout)
  {
    state = Idle;
    emit updated(this);
  }
}
bool SetiDataMonitor::parseVersionFile(const QString& fileName)
{
  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  return(parseVersionLines(this, lines, data.version));
}

bool SetiDataMonitor::parseVersionLines(DataMonitor *monitor, QStringList& lines, version_info& result)
{
  const QString sep("=");
  bool parse_ok = true;

  if(!monitor->findIntEntry("major_version", lines, sep, result.major))
    parse_ok = false;

  if(!monitor->findIntEntry("minor_version", lines, sep, result.minor))
    parse_ok = false;

  return(parse_ok);
}

bool SetiDataMonitor::parseUserInfoFile(const QString& fileName)
{
  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  return(parseUserInfoLines(this, lines, data.user));
}

bool SetiDataMonitor::parseUserInfoLines(DataMonitor *monitor, QStringList& lines, user_info& result)
{
  const QString sep("=");
  bool parse_ok = true;

  if(!monitor->findIntEntry("id", lines, sep, result.id))
    parse_ok = false;

  if(!monitor->findIntEntry("key", lines, sep, result.key))
    parse_ok = false;

  if(!monitor->findStringEntry("name", lines, sep, result.name))
    parse_ok = false;

  if(!monitor->findStringEntry("email_addr", lines, sep, result.email))
    parse_ok = false;

  if(!monitor->findStringEntry("url", lines, sep, result.url))
    parse_ok = false;

  if(!monitor->findStringEntry("country", lines, sep, result.country))
    parse_ok = false;

  if(!monitor->findIntEntry("postal_code", lines, sep, result.postal_code))
    parse_ok = false;

  QString boolean_value;

  if(!monitor->findStringEntry("show_name", lines, sep, boolean_value))
    parse_ok = false;
  result.show_name = (boolean_value == "yes");

  if(!monitor->findStringEntry("show_email", lines, sep, boolean_value))
    parse_ok = false;
  result.show_email = (boolean_value == "yes");

  if(!monitor->findIntEntry("venue", lines, sep, result.venue))
    parse_ok = false;

  QString date_value;

  if(!monitor->findStringEntry("register_time", lines, sep, date_value))
    parse_ok = false;
  result.registered.string = date_value;
  result.registered.time = date_value.isEmpty() ? QDateTime() : convertToDate(date_value);

  if(!monitor->findStringEntry("last_wu_time", lines, sep, date_value))
    parse_ok = false;
  result.last_wu.string = date_value;
  result.last_wu.time = date_value.isEmpty() ? QDateTime() : convertToDate(date_value);

  if(!monitor->findStringEntry("last_result_time", lines, sep, date_value))
    parse_ok = false;
  result.last_result.string = date_value;
  result.last_result.time = date_value.isEmpty() ? QDateTime() : convertToDate(date_value);

  if(!monitor->findIntEntry("nwus", lines, sep, result.n_wus))
    parse_ok = false;

  if(!monitor->findIntEntry("nresults", lines, sep, result.n_results))
    parse_ok = false;

  if(!monitor->findDoubleEntry("total_cpu", lines, sep, result.total_cpu))
    parse_ok = false;

  if(!monitor->findIntEntry("params_index", lines, sep, result.params_index))
    parse_ok = false;

  return(parse_ok);
}

bool SetiDataMonitor::parseWorkUnitFile(const QString& fileName)
{
  bool read_ok;
  QStringList lines = readFile(fileName, read_ok, "end_seti_header");
  if(!read_ok) return(false);

  return(parseWorkUnitLines(this, lines, data.wu));
}

bool SetiDataMonitor::parseWorkUnitLines(DataMonitor *monitor, QStringList& lines, wu_info& result)
{
  const QString sep("=");
  bool parse_ok = true;

  if(!monitor->findStringEntry("task", lines, sep, result.task))
    parse_ok = false;

  if(!monitor->findIntEntry("version", lines, sep, result.version))
    parse_ok = false;

  if(!monitor->findStringEntry("name", lines, sep, result.name))
    parse_ok = false;

  if(!monitor->findStringEntry("data_type", lines, sep, result.data_type))
    parse_ok = false;

  if(!monitor->findIntEntry("data_class", lines, sep, result.data_class))
    parse_ok = false;

  if(!monitor->findStringEntry("splitter_version", lines, sep, result.splitter_version))
    parse_ok = false;

  if(!monitor->findDoubleEntry("start_ra", lines, sep, result.start.ra))
    parse_ok = false;

  if(!monitor->findDoubleEntry("start_dec", lines, sep, result.start.dec))
    parse_ok = false;

  if(!monitor->findDoubleEntry("end_ra", lines, sep, result.end.ra))
    parse_ok = false;

  if(!monitor->findDoubleEntry("end_dec", lines, sep, result.end.dec))
    parse_ok = false;

  if(!monitor->findDoubleEntry("angle_range", lines, sep, result.angle_range))
    parse_ok = false;

  QString date_value;

  if(!monitor->findStringEntry("time_recorded", lines, sep, date_value))
    parse_ok = false;
  result.recorded.string = date_value;
  result.recorded.time = date_value.isEmpty() ? QDateTime() : convertToDate(date_value);

  if(!monitor->findDoubleEntry("subband_center", lines, sep, result.subband.center))
    parse_ok = false;

  if(!monitor->findDoubleEntry("subband_base", lines, sep, result.subband.base))
    parse_ok = false;

  if(!monitor->findDoubleEntry("subband_sample_rate", lines, sep, result.subband.sample_rate))
    parse_ok = false;

  if(!monitor->findIntEntry("subband_number", lines, sep, result.subband.number))
    parse_ok = false;

  if(!monitor->findIntEntry("fft_len", lines, sep, result.fft_length))
    parse_ok = false;

  if(!monitor->findIntEntry("ifft_len", lines, sep, result.ifft_length))
    parse_ok = false;

  if(!monitor->findStringEntry("receiver", lines, sep, result.receiver))
    parse_ok = false;

  if(!monitor->findIntEntry("nsamples", lines, sep, result.n_samples))
    parse_ok = false;

  QString tape_version;

  if(monitor->findStringEntry("tape_version", lines, sep, tape_version))
  {
    QStringList items = QStringList::split(".", tape_version);
    parse_ok = parse_ok && (items.count() == 2);
    result.tape_version.major = (items.count() > 0) ? items[0].stripWhiteSpace().toInt() : 0;
    result.tape_version.minor = (items.count() > 1) ? items[1].stripWhiteSpace().toInt() : 0;
  }
  else
  {
    parse_ok = false;
    result.tape_version.major = result.tape_version.minor = 0;
  }

  int num_positions;

  if(!monitor->findIntEntry("num_positions", lines, sep, num_positions))
    parse_ok = false;

  result.coordinates.clear();
  for(int i = 0; i < num_positions; i++)
  {
    QString coord;

    if(!monitor->findStringEntry(QString("coord%1").arg(i), lines, sep, coord))
      continue;

    QStringList list_aux = QStringList::split(" ", coord);
    if(list_aux.count() != 3) continue;

    telescope_coord coordinate;
    coordinate.time = convertToDate(list_aux[0].toDouble());
    coordinate.ra = list_aux[1].toDouble();
    coordinate.dec = list_aux[2].toDouble();

    result.coordinates += coordinate;
  }

  return(parse_ok);
}

bool SetiDataMonitor::parseStateFile(const QString& fileName)
{
  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  return(parseStateLines(this, lines, data.state));
}

bool SetiDataMonitor::parseStateLines(DataMonitor *monitor, QStringList& lines, state_info& result)
{
  const QString sep("=");
  bool parse_ok = true;

  if(!monitor->findIntEntry("ncfft", lines, sep, result.ncfft))
    parse_ok = false;

  if(!monitor->findDoubleEntry("cr", lines, sep, result.cr))
    parse_ok = false;

  if(!monitor->findIntEntry("fl", lines, sep, result.fl))
    parse_ok = false;

  if(!monitor->findDoubleEntry("cpu", lines, sep, result.cpu))
    parse_ok = false;

  if(!monitor->findDoubleEntry("prog", lines, sep, result.progress))
    parse_ok = false;

  if(!monitor->findIntEntry("potfreq", lines, sep, result.pot_frequency))
    parse_ok = false;

  if(!monitor->findIntEntry("potactivity", lines, sep, result.pot_activity))
    parse_ok = false;

  if(!monitor->findIntEntry("outfilepos", lines, sep, result.outfile_pos))
    parse_ok = false;

  {
    if(!monitor->findDoubleEntry("bs_power", lines, sep, result.best_spike.power))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bs_score", lines, sep, result.best_spike.score))
      parse_ok = false;

    if(!monitor->findIntEntry("bs_bin", lines, sep, result.best_spike.bin))
      parse_ok = false;

    if(!monitor->findIntEntry("bs_fft_ind", lines, sep, result.best_spike.fft_index))
      parse_ok = false;

    if(!monitor->findIntEntry("bs_fft_len", lines, sep, result.best_spike.fft_length))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bs_chirp_rate", lines, sep, result.best_spike.chirp_rate))
      parse_ok = false;
  }

  {
    if(!monitor->findDoubleEntry("bg_power", lines, sep, result.best_gaussian.power))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bg_score", lines, sep, result.best_gaussian.score))
      parse_ok = false;

    if(!monitor->findIntEntry("bg_bin", lines, sep, result.best_gaussian.bin))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bg_sigma", lines, sep, result.best_gaussian.sigma))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bg_true_mean", lines, sep, result.best_gaussian.true_mean))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bg_chisq", lines, sep, result.best_gaussian.chi_squared))
      parse_ok = false;

    if(!monitor->findIntEntry("bg_fft_ind", lines, sep, result.best_gaussian.fft_index))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bg_chirp_rate", lines, sep, result.best_gaussian.chirp_rate))
      parse_ok = false;

    if(!monitor->findIntEntry("bg_fft_len", lines, sep, result.best_gaussian.fft_length))
      parse_ok = false;

    result.best_gaussian.pot.clear();
    for(uint i = 0; i < 64; i++)
    {
      double value;

      if(!monitor->findDoubleEntry(QString("bg_pot %1").arg(i), lines, sep, value))
        parse_ok = false;
      result.best_gaussian.pot += value;
    }
  }

  {
    if(!monitor->findDoubleEntry("bp_power", lines, sep, result.best_pulse.power))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bp_score", lines, sep, result.best_pulse.score))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bp_mean", lines, sep, result.best_pulse.mean))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bp_period", lines, sep, result.best_pulse.period))
      parse_ok = false;

    if(!monitor->findIntEntry("bp_freq_bin", lines, sep, result.best_pulse.frequency_bin))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bp_time_bin", lines, sep, result.best_pulse.time_bin))
      parse_ok = false;

    if(!monitor->findIntEntry("bp_fft_len", lines, sep, result.best_pulse.fft_length))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bp_chirp_rate", lines, sep, result.best_pulse.chirp_rate))
      parse_ok = false;

    QString bp_pot;

    result.best_pulse.pot.clear();
    if(monitor->findStringEntry("bp_pot", lines, sep, bp_pot))
      for(uint i = 0; i < bp_pot.length()/2; i++)
      {
        bool ok;

        result.best_pulse.pot += bp_pot.mid(2*i, 2).toShort(&ok, 16);
        parse_ok = parse_ok && ok;
      }
    else
      parse_ok = false;
  }

  {
    if(!monitor->findDoubleEntry("bt_power", lines, sep, result.best_triplet.power))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bt_score", lines, sep, result.best_triplet.score))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bt_mean", lines, sep, result.best_triplet.mean))
      parse_ok  = false;

    if(!monitor->findDoubleEntry("bt_period", lines, sep, result.best_triplet.period))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bt_bperiod", lines, sep, result.best_triplet.bperiod))
      parse_ok = false;

    for(uint i = 0; i < 3; i++)
      for(uint j = 0; j < 2; j++)
        if(!monitor->findIntEntry(QString("bt_tpotind%1_%2").arg(i).arg(j), lines, sep, result.best_triplet.tpotind[i][j]))
          parse_ok = false;

    if(!monitor->findIntEntry("bt_freq_bin", lines, sep, result.best_triplet.frequency_bin))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bt_time_bin", lines, sep, result.best_triplet.time_bin))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bt_scale", lines, sep, result.best_triplet.scale))
      parse_ok = false;

    if(!monitor->findIntEntry("bt_fft_len", lines, sep, result.best_triplet.fft_length))
      parse_ok = false;

    if(!monitor->findDoubleEntry("bt_chirp_rate", lines, sep, result.best_triplet.chirp_rate))
      parse_ok = false;

    QString bt_pot;

    result.best_triplet.pot.clear();
    if(monitor->findStringEntry("bt_pot", lines, sep, bt_pot))
      for(uint i = 0; i < bt_pot.length(); i++)
      {
        bool ok;
        result.best_triplet.pot += bt_pot.mid(i, 1).toShort(&ok, 16);
        parse_ok = parse_ok && ok;
      }
    else
      parse_ok = false;
  }

  return(parse_ok);
}

bool SetiDataMonitor::parseOutputFile(const QString& fileName)
{
  bool read_ok;

  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  const QString sep(":");

  double wu_date = 0.0;
  convertToDate(data.wu.recorded.string, &wu_date);

  QString summary_value;

  data.output.spikes.clear();
  while(findStringEntry("spike", lines, sep, summary_value))
  {
    const QString sep_aux("=");

    summary_value.replace(QRegExp(sep_aux + "\\s+"), sep_aux);
    QStringList lines_aux = QStringList::split(" ", summary_value);

    spike_summary spike;

    findDoubleEntry("ra", lines_aux, sep_aux, spike.index.ra);

    findDoubleEntry("dec", lines_aux, sep_aux, spike.index.dec);

    double date;
    findDoubleEntry("time", lines_aux, sep_aux, date);
    spike.index.time = QTime(0, 0).addMSecs(int((date - wu_date) * 24 * 60 * 60 * 1e3));

    findDoubleEntry("power", lines_aux, sep_aux, spike.power);

    findDoubleEntry("freq", lines_aux, sep_aux, spike.frequency);

    findIntEntry("fft_len", lines_aux, sep_aux, spike.fft_length);

    findDoubleEntry("chirp_rate", lines_aux, sep_aux, spike.chirp_rate);

    data.output.spikes += spike;
  }

  data.output.gaussians.clear();
  while(findStringEntry("gaussian", lines, sep, summary_value))
  {
    const QString sep_aux("=");

    summary_value.replace(QRegExp(sep_aux + "\\s+"), sep_aux);
    QStringList lines_aux = QStringList::split(" ", summary_value);

    gaussian_summary gaussian;

    findDoubleEntry("ra", lines_aux, sep_aux, gaussian.index.ra);

    findDoubleEntry("dec", lines_aux, sep_aux, gaussian.index.dec);

    double date;
    findDoubleEntry("time", lines_aux, sep_aux, date);
    gaussian.index.time = QTime(0, 0).addMSecs(int((date - wu_date) * 24 * 60 * 60 * 1e3));

    findDoubleEntry("peak", lines_aux, sep_aux, gaussian.peak);

    findDoubleEntry("mean", lines_aux, sep_aux, gaussian.mean);

    findDoubleEntry("freq", lines_aux, sep_aux, gaussian.frequency);

    findDoubleEntry("sigma", lines_aux, sep_aux, gaussian.sigma);

    findDoubleEntry("chisqr", lines_aux, sep_aux, gaussian.chi_squared);

    findIntEntry("fft_len", lines_aux, sep_aux, gaussian.fft_length);

    findDoubleEntry("chirprate", lines_aux, sep_aux, gaussian.chirp_rate);

    findDoubleEntry("maxpow", lines_aux, sep_aux, gaussian.max_power);

    QString pot_value;

    findStringEntry("pot", lines_aux, sep_aux, pot_value);

    gaussian.pot.clear();
    if(!pot_value.isEmpty())
      for(uint j = 0; j < pot_value.length()/2; j++)
        gaussian.pot += pot_value.mid(j*2, 2).toInt(0, 16);

    data.output.gaussians += gaussian;
  }

  data.output.pulses.clear();
  while(findStringEntry("pulse", lines, sep, summary_value))
  {
    const QString sep_aux("=");

    summary_value.replace(QRegExp(sep_aux + "\\s+"), sep_aux);
    QStringList lines_aux = QStringList::split(" ", summary_value);

    pulse_summary pulse;

    findDoubleEntry("ra", lines_aux, sep_aux, pulse.index.ra);

    findDoubleEntry("dec", lines_aux, sep_aux, pulse.index.dec);

    double date;
    findDoubleEntry("time", lines_aux, sep_aux, date);
    pulse.index.time = QTime(0, 0).addMSecs(int((date - wu_date) * 24 * 60 * 60 * 1e3));

    findDoubleEntry("power", lines_aux, sep_aux, pulse.power);

    findDoubleEntry("mean", lines_aux, sep_aux, pulse.mean);

    findDoubleEntry("period", lines_aux, sep_aux, pulse.period);

    findDoubleEntry("freq", lines_aux, sep_aux, pulse.frequency);

    findIntEntry("fft_len", lines_aux, sep_aux, pulse.fft_length);

    findDoubleEntry("chirp_rate", lines_aux, sep_aux, pulse.chirp_rate);

    findDoubleEntry("snr", lines_aux, sep_aux, pulse.snr);

    findDoubleEntry("thresh", lines_aux, sep_aux, pulse.threshold);

    int n_items;
    findIntEntry("len_prof", lines_aux, sep_aux, n_items);

    QString prof_value;

    pulse.prof.clear();
    if(findStringEntry("prof", lines_aux, sep_aux, prof_value)
       && int(prof_value.length()) >= 2*n_items)
      for(int j = 0; j < n_items; j++)
        pulse.prof += prof_value.mid(j*2, 2).toInt(0, 16);

    data.output.pulses += pulse;
  }

  data.output.triplets.clear();
  while(findStringEntry("triplet", lines, sep, summary_value))
  {
    const QString sep_aux("=");

    summary_value.replace(QRegExp(sep_aux + "\\s+"), sep_aux);
    QStringList lines_aux = QStringList::split(" ", summary_value);

    triplet_summary triplet;

    findDoubleEntry("ra", lines_aux, sep_aux, triplet.index.ra);

    findDoubleEntry("dec", lines_aux, sep_aux, triplet.index.dec);

    double date;
    findDoubleEntry("time", lines_aux, sep_aux, date);
    triplet.index.time = QTime(0, 0).addMSecs(int((date - wu_date) * 24 * 60 * 60 * 1e3));

    findDoubleEntry("power", lines_aux, sep_aux, triplet.power);

    findDoubleEntry("mean", lines_aux, sep_aux, triplet.mean);

    findDoubleEntry("period", lines_aux, sep_aux, triplet.period);

    findDoubleEntry("freq", lines_aux, sep_aux, triplet.frequency);

    findIntEntry("fft_len", lines_aux, sep_aux, triplet.fft_length);

    findDoubleEntry("chirp_rate", lines_aux, sep_aux, triplet.chirp_rate);

    data.output.triplets += triplet;
  }

  return(true);
}

bool SetiDataMonitor::parseWUTempFile(const QString &fileName)
{
  loaded = size(fileName) / MaxWUSize;

  return(true);
}

QDateTime SetiDataMonitor::convertToDate(double n)
{
  const double epoch = 58574102.0;
  QDateTime out;

  out.setTime_t(uint((n * 24 - epoch) * 60 * 60));
  out = out.addSecs(KRFCDate::localUTCOffset() * 60);

  return(out);
}

QDateTime SetiDataMonitor::convertToDate(const QString &str, double *date)
{
  const int pos = str.find(" (");
  double out;

  out = (pos != -1) ? str.left(pos).toDouble() : 0.0;
  if(date != NULL) *date = out;

  return(convertToDate(out));
}

double SetiDataMonitor::resolution(int fft_length)
{
  return((fft_length > 0) ? WorkUnitBandwidth/fft_length : 0.0);
}

double SetiDataMonitor::score(const spike_summary& summary)
{
  return((summary.power > 0) ? log10(0.025 * summary.power) : 0.0);
}

double SetiDataMonitor::score(const gaussian_summary& summary)
{
  return((summary.chi_squared > 0) ? summary.peak/summary.chi_squared : 0.0);
}

double SetiDataMonitor::score(const pulse_summary& summary)
{
  return((summary.threshold > 0) ? summary.snr/summary.threshold : 0.0);
}

double SetiDataMonitor::score(const triplet_summary& summary)
{
  return(summary.power);
}

double SetiDataMonitor::signalRatio(const spike_summary& summary)
{
  return((summary.fft_length > 0) ? 16 * 1024 * summary.power/summary.fft_length : 0.0);
}

double SetiDataMonitor::signalRatio(const gaussian_summary& summary)
{
  return((summary.mean > 0) ? summary.peak/summary.mean : 0.0);
}

spike_summary SetiDataMonitor::toSummary(const spike_detail& detail) const
{
  const double res = resolution(detail.fft_length);

  spike_summary summary;

  summary.index.ra = summary.index.dec = 0.0;
  summary.index.time = (res > 0) ? QTime(0, 0).addMSecs(int((detail.fft_index + 0.5)/res * 10e2)) : QTime();

  summary.power = detail.power;
  summary.frequency = data.wu.subband.base + detail.bin * res;
  summary.fft_length = detail.fft_length;
  summary.chirp_rate = detail.chirp_rate;

  return(summary);
}

gaussian_summary SetiDataMonitor::toSummary(const gaussian_detail& detail) const
{
  const double res = resolution(detail.fft_length);

  gaussian_summary summary;

  summary.index.ra = summary.index.dec = 0.0;
  summary.index.time = (res > 0) ? QTime(0, 0).addMSecs(int(detail.fft_index/res * 10e2)) : QTime();

  summary.peak = detail.power;
  summary.mean = detail.true_mean;
  summary.frequency = data.wu.subband.base + detail.bin * res;
  summary.sigma = detail.sigma;
  summary.chi_squared = detail.chi_squared;
  summary.fft_length = detail.fft_length;
  summary.chirp_rate = detail.chirp_rate;

  summary.max_power = 0;
  for(QValueList<double>::ConstIterator it = detail.pot.begin(); it != detail.pot.end(); it++)
    if(*it > summary.max_power) summary.max_power = *it;

  summary.pot.clear();
  for(QValueList<double>::ConstIterator it = detail.pot.begin(); it != detail.pot.end(); it++)
    summary.pot += int(255 * *it / summary.max_power);

  return(summary);
}

pulse_summary SetiDataMonitor::toSummary(const pulse_detail& detail) const
{
  const double res = resolution(detail.fft_length);

  pulse_summary summary;

  summary.index.ra = summary.index.dec = 0.0;
  summary.index.time = (res > 0) ? QTime(0, 0).addMSecs(int(detail.time_bin/res * 10e2)) : QTime();

  summary.power = detail.power;
  summary.mean = detail.mean;
  summary.period = detail.period;
  summary.frequency = data.wu.subband.base + detail.frequency_bin * res;
  summary.fft_length = detail.fft_length;
  summary.chirp_rate = detail.chirp_rate;
  summary.snr = detail.score;
  summary.threshold = 1.0;

  summary.prof = detail.pot;

  return(summary);
}

triplet_summary SetiDataMonitor::toSummary(const triplet_detail& detail) const
{
  const double res = resolution(detail.fft_length);

  triplet_summary summary;

  summary.index.ra = summary.index.dec = 0.0;
  summary.index.time = (res > 0) ? QTime(0, 0).addMSecs(int(detail.time_bin/res * 10e2)) : QTime();

  summary.power = detail.power;
  summary.mean = detail.mean;
  summary.period = detail.period;
  summary.frequency = data.wu.subband.base + detail.frequency_bin * res;
  summary.fft_length = detail.fft_length;
  summary.chirp_rate = detail.chirp_rate;

  return(summary);
}

bool SetiDataMonitor::isInteresting(const gaussian_detail& gaussian)
{
  const double peak = gaussian.power;
  const double chi_squared = gaussian.chi_squared;
  const double snr = gaussian.true_mean > 0.0 ? peak / gaussian.true_mean : 0.0;

  const bool interesting = peak > 0.0
                           && chi_squared > 0.0
                           && chi_squared < 2.0 + 1.6 * peak && chi_squared < 10.0;
  const bool returned = chi_squared > 0.0 && chi_squared < 8.8 && snr > 3.2;

  return(interesting && returned);
}

bool SetiDataMonitor::isInteresting(const gaussian_summary& gaussian)
{
  const double peak = gaussian.peak;
  const double chi_squared = gaussian.chi_squared;
  const double snr = gaussian.mean > 0.0 ? peak / gaussian.mean : 0.0;

  const bool interesting = peak > 0.0
                           && chi_squared > 0.0
                           && chi_squared < 2.0 + 1.6 * peak && chi_squared < 10.0;
  const bool returned = chi_squared > 0.0 && chi_squared < 8.8 && snr > 3.2;

  return(interesting && returned);
}

#include "setidatamonitor.moc"

