/***************************************************************************
                          setispylogmonitor.cpp  -  description
                             -------------------
    begin                : Mon Jul 30 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qapplication.h>
#include <qdatetime.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qtextstream.h>

#include <kio/netaccess.h>
#include <klocale.h>
#include <ktempfile.h>
#include <kdeversion.h>

#include "setidatamonitor.h"

#include "setispylogmonitor.h"

const QString SETISpy_log_file = "setispy.log";

const QString SETISpy_log_headers[3] =
{
  QString("----- Caution: New Log File format from here! -----"),
  QString("    Date       Time              Work Unit Name              Start   Start  Angle  Tera-  Process") +
  QString("  Percent  Returned   Best  Returned   Best  Returned   Best  Returned   Best"),
  QString("    Done       Done                                           RA      Dec   Range  FLOPs  Time-hr") +
  QString("    Done    Spikes   Spike  Gauss'ns Gauss'n  Pulses   Pulse  Triplets Triplet")
};

SETISpyLogMonitor::SETISpyLogMonitor(const KURL& logURL, QObject *parent, const char *name)
                  : LogMonitor(logURL, parent, name)
{
  addFile(SETISpy_log_file, SETISpy_log_index);
}

SETISpyLogMonitor::~SETISpyLogMonitor()
{
}

const QValueList<log_data> SETISpyLogMonitor::logData()
{
  return(isOK(SETISpy_log_index) ? data : LogMonitor::logData());
}

void SETISpyLogMonitor::writeSetiData(const KURL& logURL, const seti_data& data)
{
  KURL fileURL = logURL;
  fileURL.addPath(SETISpy_log_file);

  KTempFile tempFile;
  tempFile.close();

  QString fileName;
  if(!fileURL.isLocalFile())
  {
    fileName = tempFile.name();
    // try to download the log file if present
    #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::download(fileURL, fileName, qApp->mainWidget());
    #else
    KIO::NetAccess::download(fileURL, fileName);
    #endif
    
  }
  else
    fileName = fileURL.path(-1);

  QFile file(fileName);

  if(file.open(IO_ReadWrite))
  {
    writeHeader(&file);
    writeRow(&file, data);
    file.close();
  }

  if(!fileURL.isLocalFile())
  #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::upload(fileName, fileURL, qApp->mainWidget());
  #else
    KIO::NetAccess::upload(fileName, fileURL);
  #endif

  tempFile.unlink();
}

bool SETISpyLogMonitor::parseFile(int index, const QString& fileName)
{
  if(index == SETISpy_log_index)
    return(parseLogFile(fileName));
  else
    return(DataMonitor::parseFile(index, fileName));
}

bool SETISpyLogMonitor::parseLogFile(const QString& fileName)
{
  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  QStringList::Iterator line = lines.begin();
  while(line != lines.end())
  {
    QStringList::Iterator lineNew = lines.find(line, SETISpy_log_headers[0]);

    if(lineNew == lines.end()) break;
    else line = ++lineNew;
  }

  data.clear();

  if(line++ == lines.end()) return(true);

  log_data item;
  while(line != lines.end())
    if(parseLine(*(line++), item)) data += item;

  return(true);
}

bool SETISpyLogMonitor::parseLine(const QString& line, log_data& item)
{
  QStringList fields = QStringList::split(QRegExp("\\s"), line);
  if(fields.count() < 18) return(false);

  QStringList::Iterator field;

  field = fields.begin();
  {
    QStringList subfields = QStringList::split("/", *field);
    if(subfields.count() == 3)
    {
      const int month = subfields[0].toInt();
      const int day = subfields[1].toInt();
      int year = subfields[2].toInt();
      if(year < 100)
        if(year > 50) year+= 1900;
        else year+= 2000;
      item.done.setDate(QDate(year, month, day));
    }
    else return(false);
  }

  {
    QStringList subfields = QStringList::split(":", *(++field));
    if(subfields.count() == 3)
    {
      const int hour = subfields[0].toInt();
      const int minute = subfields[1].toInt();
      const int second = subfields[2].toInt();
      item.done.setTime(QTime(hour, minute, second));
    }
    else return(false);
  }

  if(*(++field) == "PM")
    item.done = item.done.addSecs(3600 * 12);
  if(item.done.time().hour() % 12 == 0)
    item.done = item.done.addSecs(-3600 * 12);

  if(!item.done.isValid()) return(false);

  item.name = *(++field);

  item.start.ra = (*(++field)).toDouble();
  item.start.dec = (*(++field)).toDouble();

  item.angle_range = (*(++field)).toDouble();
  item.teraFLOPs = (*(++field)).toDouble();

  item.cpu = (*(++field)).toDouble() * 3600;
  {
    QString progress = *(++field);
    progress.remove('%');
    item.progress = progress.toDouble() * 1e-2;
  }
    
  item.spikes.returned = (*(++field)).toInt();
  item.spikes.best_score = (*(++field)).toDouble();

  item.gaussians.returned = (*(++field)).toInt();
  {
    QString best_score = *(++field);
    item.interesting = best_score.contains('*') > 0;
    best_score.remove('*');
    item.gaussians.best_score = best_score.toDouble();
  }

  item.pulses.returned = (*(++field)).toInt();
  item.pulses.best_score = (*(++field)).toDouble();

  item.triplets.returned = (*(++field)).toInt();
  item.triplets.best_score = (*(++field)).toDouble();

  return(true);
}

void SETISpyLogMonitor::writeHeader(QFile *file)
{
  file->at(0);

  QTextStream stream(file);
  QString line;

  while(true)
  {
    line = stream.readLine();

    if(line.isNull())
      break;
    else if(line == SETISpy_log_headers[1])
    {
      line = stream.readLine();
      if(line.isNull() || line == SETISpy_log_headers[2])
        break;
    }
  }

  if(line.isNull())
  {
    file->at(file->size());

    if(file->size() > 0)
      stream << SETISpy_log_headers[0] << "\r\n";
    stream << SETISpy_log_headers[1] << "\r\n";
    stream << SETISpy_log_headers[2] << "\r\n";

    file->flush();
  }
}

void SETISpyLogMonitor::writeRow(QFile *file, const seti_data& data)
{
  file->at(file->size());

  QTextStream stream(file);
  QDateTime now = QDateTime::currentDateTime();
  QString field;

  const QDate now_date = now.date();
  field.sprintf("%d/%d/%d", now_date.month(), now_date.day(), now_date.year());
  stream << field.rightJustify(10);

  const QTime now_time = now.time();
  if(now.time().hour() == 0)
    field.sprintf("12:%.2d:%.2d AM", now_time.minute(), now_time.second());
  else if(now.time().hour() == 12)
    field.sprintf("12:%.2d:%.2d PM", now_time.minute(), now_time.second());
  else if(now.time().hour() < 12)
    field.sprintf("%d:%.2d:%.2d AM", now_time.hour(), now_time.minute(), now_time.second());
  else
    field.sprintf("%d:%.2d:%.2d PM", now_time.hour()-12, now_time.minute(), now_time.second());
  stream << field.rightJustify(13);

  field = "  " + data.wu.name;
  stream << field.leftJustify(35);

  field = formatDouble(data.wu.start.ra);
  stream << field.rightJustify(8);

  field = formatDouble(data.wu.start.dec);
  stream << field.rightJustify(8);

  field = formatDouble(data.wu.angle_range);
  stream << field.rightJustify(7);

  field = formatDouble(SetiDataMonitor::teraFLOPs(data));
  stream << field.rightJustify(7);

  field = formatDouble(data.state.cpu/3600);
  stream << field.rightJustify(9);

  field = formatDouble(data.state.progress * 1e2, 2) + "%";
  stream << field.rightJustify(9);

  field = QString::number(data.output.spikes.count());
  stream << field.rightJustify((9 + field.length())/2 + 2).leftJustify(9);

  field = formatDouble(data.state.best_spike.score);
  stream << field.rightJustify(8);

  field = QString::number(data.output.gaussians.count());
  stream << field.rightJustify((9 + field.length())/2 + 2).leftJustify(9);

  field = formatDouble(data.state.best_gaussian.score);
  stream << field.rightJustify((9 + field.length())/2 + 1).leftJustify(9);

  field = QString::number(data.output.pulses.count());
  stream << field.rightJustify((9 + field.length())/2 + 1).leftJustify(9);

  field = formatDouble(data.state.best_pulse.score);
  stream << field.rightJustify(7);

  field = QString::number(data.output.triplets.count());
  stream << field.rightJustify((9 + field.length())/2 + 2).leftJustify(9);

  field = formatDouble(data.state.best_triplet.score);
  stream << field.rightJustify(8);

  stream << "\r\n";

  file->flush();
}

#include "setispylogmonitor.moc"

