/***************************************************************************
                          signalplot.cpp  -  description
                             -------------------
    begin                : Sat Jun 2 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>

#include <qpainter.h>

#include <kglobal.h>
#include <klocale.h>

#include "seticlientmonitor.h"

#include "signalplot.h"

bool operator==(const summary_coord& coord1, const summary_coord& coord2)
{
  if(coord1.ra != coord2.ra)
    return(false);
  if(coord1.dec != coord2.dec)
    return(false);
  if(coord1.time != coord2.time)
    return(false);
  return(true);
}

bool operator==(const gaussian_summary& gaussian1, const gaussian_summary& gaussian2)
{
  return(gaussian1.index == gaussian2.index);
}

bool operator==(const pulse_summary& pulse1, const pulse_summary& pulse2)
{
  return(pulse1.index == pulse2.index);
}

bool operator==(const wu_info& wu1, const wu_info& wu2)
{
  return(wu1.name == wu2.name);
}

SignalPlot::SignalPlot(QWidget *parent, const char *name, WFlags f) : QWidget(parent, name, f)
{
  setBackgroundMode(PaletteBase);
  setPalette(black);

  type = NoData;
}

SignalPlot::~SignalPlot()
{
}

void SignalPlot::clearData()
{
  if(type == NoData)
    return;
  
  type = NoData;
  repaint();
}

void SignalPlot::setData(const gaussian_summary& gaussian)
{
  if(SetiClientMonitor::score(gaussian) > 0.0)
  {
    if(type == Gaussian && data.gaussian == gaussian)
      return;

    type = Gaussian;
    data.gaussian = gaussian;
  }
  else
  {
    if(type == NoData)
      return;

    type = NoData;
  }

  repaint();
}

void SignalPlot::setData(const pulse_summary& pulse)
{
  if(SetiClientMonitor::score(pulse) > 0.0)
  {
    if(type == Pulse && data.pulse == pulse)
      return;

    type = Pulse;
    data.pulse = pulse;
  }
  else
  {
    if(type == NoData)
      return;

    type = NoData;
  }

  repaint();
}

void SignalPlot::setData(const wu_info& wu)
{
  if(type == TelescopePath && data.wu == wu)
      return;

  type = TelescopePath;
  data.wu = wu;
  repaint();
}

void SignalPlot::paintEvent(QPaintEvent *)
{
  const int margin = 5;
  const QString noDataMsg = (type == NoData) ? i18n("No data")
                          : (type == Gaussian) ? i18n("No gaussian data")
                          : (type == Pulse) ? i18n("No pulse data")
                          : /* type == TelescopePath */ i18n("No telescope path data");

  QPainter painter(this);

  if(type != NoData)
  {
    QRect headerRect = QRect(margin, margin, width() - 2*margin, 3*fontMetrics().lineSpacing());
    QRect plotRect = QRect(margin, headerRect.bottom() + margin,
                           width() - 2*margin, height() - headerRect.bottom() - 2*margin);

    KLocale *locale = KGlobal::locale();

    QString header;

    switch(type) {
      case Gaussian:
        {
          const double score = SetiClientMonitor::score(data.gaussian);
          const double resolution = SetiClientMonitor::resolution(data.gaussian.fft_length);;
          const double signal_ratio = data.gaussian.peak/data.gaussian.mean;
          const double secs = QTime(0, 0).msecsTo(data.gaussian.index.time)/1e3;
          header = i18n("Peak power %1, fit %2, score %3").arg(locale->formatNumber(data.gaussian.peak, 3))
                                                          .arg(locale->formatNumber(data.gaussian.chi_squared, 3))
                                                          .arg(locale->formatNumber(score, 3)) + "\n" +
                   i18n("Resolution %1 Hz, signal ratio %2").arg(locale->formatNumber(resolution, 3))
                                                            .arg(locale->formatNumber(signal_ratio, 3)) + "\n" +
                   i18n("(%1 Hz at %2 s, %3 Hz/s drift)").arg(locale->formatNumber(data.gaussian.frequency, 2))
                                                         .arg(locale->formatNumber(secs, 2))
                                                         .arg(locale->formatNumber(data.gaussian.chirp_rate, 4));
        }
        break;
      case Pulse:
        {
          const double score = SetiClientMonitor::score(data.pulse);
          const double resolution = SetiClientMonitor::resolution(data.pulse.fft_length);
          const double secs = QTime(0, 0).msecsTo(data.pulse.index.time)/1e3;
          header = i18n("Power %1, score %2").arg(locale->formatNumber(data.pulse.power, 3))
                                             .arg(locale->formatNumber(score, 3)) + "\n" +
                   i18n("Resolution %1 Hz, period %2").arg(locale->formatNumber(resolution, 3))
                                                      .arg(locale->formatNumber(data.pulse.period, 3)) + "\n" +
                   i18n("(%1 Hz at %2 s, %3 Hz/s drift)").arg(locale->formatNumber(data.pulse.frequency, 2))
                                                         .arg(locale->formatNumber(secs, 2))
                                                         .arg(locale->formatNumber(data.pulse.chirp_rate, 4));
        }
        break;
      default:
        {
          double min_ra = (data.wu.start.ra > data.wu.end.ra) ? data.wu.end.ra : data.wu.start.ra;
          double max_ra = (data.wu.start.ra > data.wu.end.ra) ? data.wu.start.ra : data.wu.end.ra;
          double ra_diff = max_ra - min_ra; if(ra_diff > 12.0) ra_diff = 24.0 - ra_diff;

          double min_dec = (data.wu.start.dec > data.wu.end.dec) ? data.wu.end.dec : data.wu.start.dec;
          double max_dec = (data.wu.start.dec > data.wu.end.dec) ? data.wu.start.dec : data.wu.end.dec;
          double dec_diff = max_dec - min_dec;

          header = i18n("RA range: %1 to %2 (%3)").arg(SetiClientMonitor::raToString(min_ra))
                                                  .arg(SetiClientMonitor::raToString(max_ra))
                                                  .arg(SetiClientMonitor::raToString(ra_diff)) + "\n" +
                   i18n("Dec range: %1 to %2 (%3)").arg(SetiClientMonitor::decToString(min_dec))
                                                   .arg(SetiClientMonitor::decToString(max_dec))
                                                   .arg(SetiClientMonitor::decToString(dec_diff, false)) + "\n" +
                   i18n("Angle range: %1").arg(locale->formatNumber(data.wu.angle_range, 3) + "");
        }
        break;
    }
    painter.drawText(headerRect, AlignLeft, header);
    if(type != TelescopePath)
    {
      QValueList<int> pot = (type == Gaussian) ? data.gaussian.pot : data.pulse.prof;

      if(pot.count() > 0)
      {
        int min, max;

        min = max = pot[0];
        for(uint i = 1; i < pot.count(); i++)
        {
          min = (pot[i] < min) ? pot[i] : min;
          max = (pot[i] > max) ? pot[i] : max;
        }

        const double unit = plotRect.width() / double(pot.count());
        const double scale = plotRect.height() / double(max - min + 1);

        painter.translate(plotRect.x(), plotRect.bottom());
        painter.setPen(red);
        painter.moveTo(0, 0);

        for(uint i = 0; i < pot.count(); i++)
        {
          const int y = - int(scale * (pot[i] - min) + 1);

          painter.lineTo(int(unit * i), y);
          painter.lineTo(int(unit * (i+1)), y);
        }

        if(type == Gaussian)
        {
          const double resolution = SetiClientMonitor::resolution(data.gaussian.fft_length);;
          const double secs = QTime(0, 0).msecsTo(data.gaussian.index.time)/1e3;

          const int base = int(255 * data.gaussian.mean/data.gaussian.max_power);
          const double offset_scale = 255 * data.gaussian.peak/data.gaussian.max_power;

          const double mu = secs * resolution;
          const double sigma = data.gaussian.sigma;

          painter.setPen(white);

          painter.moveTo(0, -int(scale * (base - min) + 1));
          for(int i = 0; i < plotRect.width()/3; i++)
          {
            const int offset = int(offset_scale * exp(-0.5 * pow((3*i/unit - mu)/sigma, 2)));
            const int y = -int(scale * (base + offset - min) + 1);

            if(i == 0) painter.moveTo(3*i, y);
            else painter.lineTo(3*i, y);
          }
        }
      } else {
        painter.setPen(gray);
        painter.drawText(plotRect, AlignCenter, noDataMsg);
      }
    } else {
      uint n_points = data.wu.coordinates.count();
      double minX, maxX, minY, maxY;

      if(n_points > 1)
      {
        minX = maxX = data.wu.coordinates[0].ra;
        minY = maxY = data.wu.coordinates[0].dec;
        for(uint i = 1; i < n_points; i++)
        {
          const double x = data.wu.coordinates[i].ra;
          const double y = data.wu.coordinates[i].dec;

          minX = (x < minX) ? x : minX;
          maxX = (x > maxX) ? x : maxX;
          minY = (y < minY) ? y : minY;
          maxY = (y > maxY) ? y : maxY;
        }
      } else
        minX = maxX = minY = maxY = 0.0;

      if(minX != maxX || minY != maxY) {
        painter.translate(plotRect.x(), plotRect.y());
        painter.setPen(red);

        int oldX, oldY;
        oldX = oldY = 0;
        for(uint i = 0; i < n_points; i++)
        {
          double normX = (minX != maxX) ? (data.wu.coordinates[i].ra - minX)/(maxX - minX) : 0.5;
          double normY = (minY != maxY) ? (data.wu.coordinates[i].dec - minY)/(maxY - minY) : 0.5;

          int newX = (data.wu.start.ra < data.wu.end.ra) ? int(plotRect.width() * (1.0 - normX))
                                                         : int(plotRect.width() * normX);
          int newY = (data.wu.start.dec < data.wu.end.dec) ? int(plotRect.height() * normY)
                                                           : int(plotRect.height() * (1.0 - normY));

          if(newX == oldX && newY == oldY) continue;

          if(i > 0) arrow(&painter, oldX, oldY, newX, newY);

          oldX = newX;
          oldY = newY;
        }
      } else {
        painter.setPen(gray);
        painter.drawText(plotRect, AlignCenter, noDataMsg);
      }
    }
  } else {
    painter.setPen(gray);
    painter.drawText(rect(), AlignCenter, noDataMsg);
  }
}

void SignalPlot::arrow(QPainter *painter, int x1, int y1, int x2, int y2)
{
  painter->moveTo(x1, y1);
  painter->lineTo(x2, y2);

  double ang = (x1 == x2) ? ((y2 > y1) ? -90 : 90)
                          : (180 * atan((y2 - y1)/double(x2 - x1)) / M_PI);
  int x0 = (x2 > x1) ? -3 : 3;

  painter->save();
  painter->translate(x2, y2);
  painter->rotate(ang);
  painter->moveTo(0, 0);
  painter->lineTo(x0, 3);
  painter->moveTo(0, 0);
  painter->lineTo(x0, -3);
  painter->restore();
}

#include "signalplot.moc"

