/***************************************************************************
                          sysinfomonitor.cpp  -  description
                             -------------------
    begin                : Sun Jun 17 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qregexp.h>

#include <kglobal.h>
#include <klocale.h>

#include "sysinfomonitor.h"

const QString cpu_info_file        = "cpuinfo";
const QString memory_info_file    = "meminfo";
const QString kernel_info_file    = "version";

SysInfoMonitor::SysInfoMonitor(const KURL& procURL, QObject *parent, const char *name)
               : DataMonitor(procURL, parent, name)
{
  addFile(cpu_info_file, cpu_info_index);
  addFile(memory_info_file, memory_info_index);
  addFile(kernel_info_file, kernel_info_index);

  ok = isOK(cpu_info_index) && isOK(memory_info_index) && isOK(kernel_info_index);
}

SysInfoMonitor::~SysInfoMonitor()
{
}

const sys_info *SysInfoMonitor::sysInfo()
{
  return(ok ? &data : NULL);
}

QString SysInfoMonitor::kbToString(int kb)
{
  QString suffix = " KB";

  if(kb >= 1024 && kb % 1024 == 0)
  {
    kb /= 1024;
    suffix = " MB";
  }
  if(kb >= 1024 && kb % 1024 == 0)
  {
    kb /= 1024;
    suffix = " GB";
  }

  return(KGlobal::locale()->formatNumber(kb, 0) + suffix);
}

int SysInfoMonitor::stringToKB(QString size)
{
  if(size.contains(QRegExp(" [kK][bB]$")))
  {
    size.truncate(size.length()-3);
    return(size.toInt());
  }
  else if(size.contains(QRegExp(" [mM][bB]$")))
  {
    size.truncate(size.length()-3);
    return(size.toInt() * 1024);
  }
  else if(size.contains(QRegExp(" [gG][bB]$")))
  {
    size.truncate(size.length()-3);
    return(size.toInt() * 1024 * 1024);
  }
  else return(-1);
}

bool SysInfoMonitor::parseFile(int index, const QString& fileName)
{
  if(index == cpu_info_index)
    return(parseCPUFile(fileName));
  else if(index == memory_info_index)
    return(parseMemoryFile(fileName));
  else if(index == kernel_info_index)
    return(parseVersionFile(fileName));
  else
    return(DataMonitor::parseFile(index, fileName));
}

void SysInfoMonitor::updateData()
{
  ok = isOK(cpu_info_index) && isOK(memory_info_index) && isOK(kernel_info_index);

  emit updatedSysInfo(this, ok ? &data : NULL);
}

bool SysInfoMonitor::parseCPUFile(const QString& fileName)
{
  const QString sep(":");

  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  QString processor;

  data.cpus.clear();
  while(findStringEntry("processor", lines, sep, processor))
  {
    cpu_info entry;

    findStringEntry("vendor_id", lines, sep, entry.vendorID);

    if(!findIntEntry("cpu family", lines, sep, entry.cpuFamily))
      entry.cpuFamily = -1;

    if(!findIntEntry("model", lines, sep, entry.model.id))
      entry.model.id = -1;

    findStringEntry("model name", lines, sep, entry.model.name);

    if(!findIntEntry("stepping", lines, sep, entry.stepping))
      entry.stepping = -1;

    findDoubleEntry("cpu MHz", lines, sep, entry.MHz);

    QString size_value;

    findStringEntry("cache size", lines, sep, size_value);
    entry.cacheSize = size_value.isEmpty() ? -1L : stringToKB(size_value);

    findDoubleEntry("bogomips", lines, sep, entry.bogoMIPS);

    data.cpus += entry;
  }

  return(true);
}

bool SysInfoMonitor::parseMemoryFile(const QString& fileName)
{
  const QString sep(":");

  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  QString size_value;

  findStringEntry("MemTotal", lines, sep, size_value);
  data.memory.overall.total = size_value.isEmpty() ? -1L : stringToKB(size_value);

  findStringEntry("MemFree", lines, sep, size_value);
  data.memory.overall.free = size_value.isEmpty() ? -1L : stringToKB(size_value);

  findStringEntry("MemShared", lines, sep, size_value);
  data.memory.shared = size_value.isEmpty() ? -1L : stringToKB(size_value);

  findStringEntry("Buffers", lines, sep, size_value);
  data.memory.buffers = size_value.isEmpty() ? -1L : stringToKB(size_value);

  findStringEntry("SwapTotal", lines, sep, size_value);
  data.memory.swap.total = size_value.isEmpty() ? -1L : stringToKB(size_value);

  findStringEntry("SwapFree", lines, sep, size_value);
  data.memory.swap.free = size_value.isEmpty() ? -1L : stringToKB(size_value);

  return(true);
}

bool SysInfoMonitor::parseVersionFile(const QString& fileName)
{
  bool read_ok;
  QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  if(lines.count() == 0) return(false);

  QStringList split = QStringList::split(QRegExp("[()]"), lines.first());

  if(split.count() < 2) return(false);

  data.kernel.name = split[0].stripWhiteSpace();
  data.kernel.builder = split[1].stripWhiteSpace();

  return(true);
}

#include "sysinfomonitor.moc"

