/***************************************************************************
                          userstatsview.cpp  -  description
                             -------------------
    begin                : Wed May 30 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qlabel.h>
#include <qlayout.h>
#include <qwidgetstack.h>

#include <kapplication.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpushbutton.h>

#include "anniversarydialog.h"
#include "wucalendardialog.h"

#include "userstatsview.h"

const QString RegistrationClassURL = "http://%1/classpages/days/%2.html";
const QString VenueURL             = "http://%1/stats/venue_%2.html";

const int N_COUNTRIES = 226;

const struct {
  const char *us_name;
  QString name;
  int id;
} Countries[N_COUNTRIES] =
{
  {"Afghanistan", I18N_NOOP("Afghanistan"), 924},
  {"Albania", I18N_NOOP("Albania"), 2151},
  {"Algeria", I18N_NOOP("Algeria"), 6790},
  {"American Samoa", I18N_NOOP("American Samoa"), 2887},
  {"Andorra", I18N_NOOP("Andorra"), 2744},
  {"Angola", I18N_NOOP("Angola"), 7901},
  {"Anguilla", I18N_NOOP("Anguilla"), 59260},
  {"Antarctica", I18N_NOOP("Antarctica"), 194925},
  {"Antigua and Barbuda", I18N_NOOP("Antigua and Barbuda"), 448},
  {"Argentina", I18N_NOOP("Argentina"), 200},
  {"Armenia", I18N_NOOP("Armenia"), 1790},
  {"Aruba", I18N_NOOP("Aruba"), 1607},
  {"Australia", I18N_NOOP("Australia"), 169},
  {"Austria", I18N_NOOP("Austria"), 348},
  {"Azerbaijan", I18N_NOOP("Azerbaijan"), 3506},
  {"Bahamas", I18N_NOOP("Bahamas"), 290},
  {"Bahrain", I18N_NOOP("Bahrain"), 58483},
  {"Bangladesh", I18N_NOOP("Bangladesh"), 1974},
  {"Barbados", I18N_NOOP("Barbados"), 2986},
  {"Belarus", I18N_NOOP("Belarus"), 3626},
  {"Belgium", I18N_NOOP("Belgium"), 70},
  {"Belize", I18N_NOOP("Belize"), 5233},
  {"Benin", I18N_NOOP("Benin"), 25593},
  {"Bermuda", I18N_NOOP("Bermuda"), 213},
  {"Bhutan", I18N_NOOP("Bhutan"), 56365},
  {"Bolivia", I18N_NOOP("Bolivia"), 2165},
  {"Bosnia and Herzegovina", I18N_NOOP("Bosnia and Herzegovina"), 1314},
  {"Botswana", I18N_NOOP("Botswana"), 4447},
  {"Brazil", I18N_NOOP("Brazil"), 235},
  {"British Virgin Islands", I18N_NOOP("British Virgin Islands"), 9541},
  {"Brunei Darussalam", I18N_NOOP("Brunei Darussalam"), 917},
  {"Bulgaria", I18N_NOOP("Bulgaria"), 890},
  {"Burkina Faso", I18N_NOOP("Burkina Faso"), 3272},
  {"Burundi", I18N_NOOP("Burundi"), 25072},
  {"Cambodia", I18N_NOOP("Cambodia"), 2212},
  {"Cameroon", I18N_NOOP("Cameroon"), 1917},
  {"Canada", I18N_NOOP("Canada"), 50},
  {"Cape Verde", I18N_NOOP("Cape Verde"), 3427},
  {"Cayman Islands", I18N_NOOP("Cayman Islands"), 2453},
  {"Central African Republic", I18N_NOOP("Central African Republic"), 66347},
  {"Chad", I18N_NOOP("Chad"), 9212},
  {"Channel Islands", I18N_NOOP("Channel Islands"), 3849},
  {"Chile", I18N_NOOP("Chile"), 373},
  {"China", I18N_NOOP("China"), 279},
  {"Colombia", I18N_NOOP("Colombia"), 459},
  {"Comoros", I18N_NOOP("Comoros"), 7030},
  {"Congo", I18N_NOOP("Congo"), 5660},
  {"Cook Islands", I18N_NOOP("Cook Islands"), 6096},
  {"Costa Rica", I18N_NOOP("Costa Rica"), 1154},
  {"Cote d'Ivoire", I18N_NOOP("Cote d'Ivoire"), 3253},
  {"Croatia", I18N_NOOP("Croatia"), 311},
  {"Cuba", I18N_NOOP("Cuba"), 1516},
  {"Cyprus", I18N_NOOP("Cyprus"), 720},
  {"Czech Republic", I18N_NOOP("Czech Republic"), 140},
  {"Dem. People's Rep. of Korea", I18N_NOOP("Dem. People's Rep. of Korea"), 2124},
  {"Dem. Republic of the Congo", I18N_NOOP("Dem. Republic of the Congo"), 8922},
  {"Denmark", I18N_NOOP("Denmark"), 131},
  {"Djibouti", I18N_NOOP("Djibouti"), 4036},
  {"Dominica", I18N_NOOP("Dominica"), 5237},
  {"Dominican Republic", I18N_NOOP("Dominican Republic"), 800},
  {"East Timor", I18N_NOOP("East Timor"), 4489},
  {"Ecuador", I18N_NOOP("Ecuador"), 665},
  {"Egypt", I18N_NOOP("Egypt"), 2787},
  {"El Salvador", I18N_NOOP("El Salvador"), 1867},
  {"Equatorial Guinea", I18N_NOOP("Equatorial Guinea"), 153840},
  {"Eritrea", I18N_NOOP("Eritrea"), 122803},
  {"Estonia", I18N_NOOP("Estonia"), 294},
  {"Ethiopia", I18N_NOOP("Ethiopia"), 5458},
  {"Faeroe Islands", I18N_NOOP("Faeroe Islands"), 1492},
  {"Falkland Islands", I18N_NOOP("Falkland Islands"), 5663},
  {"Fiji", I18N_NOOP("Fiji"), 2838},
  {"Finland", I18N_NOOP("Finland"), 18},
  {"Former Yugoslav Republic of Macedonia", I18N_NOOP("Former Yugoslav Republic of Macedonia"), 174559},
  {"France", I18N_NOOP("France"), 98},
  {"French Guiana", I18N_NOOP("French Guiana"), 2823},
  {"French Polynesia", I18N_NOOP("French Polynesia"), 2179},
  {"Gabon", I18N_NOOP("Gabon"), 2607},
  {"Gambia", I18N_NOOP("Gambia"), 9237},
  {"Gaza Strip", I18N_NOOP("Gaza Strip"), 4674},
  {"Georgia", I18N_NOOP("Georgia"), 2096},
  {"Germany", I18N_NOOP("Germany"), 58},
  {"Ghana", I18N_NOOP("Ghana"), 4185},
  {"Gibraltar", I18N_NOOP("Gibraltar"), 1891},
  {"Greece", I18N_NOOP("Greece"), 208},
  {"Greenland", I18N_NOOP("Greenland"), 1128},
  {"Grenada", I18N_NOOP("Grenada"), 4292},
  {"Guadeloupe", I18N_NOOP("Guadeloupe"), 476},
  {"Guam", I18N_NOOP("Guam"), 839},
  {"Guatemala", I18N_NOOP("Guatemala"), 1098},
  {"Guinea", I18N_NOOP("Guinea"), 115474},
  {"Guinea-Bissau", I18N_NOOP("Guinea-Bissau"), 6450},
  {"Guyana", I18N_NOOP("Guyana"), 7630},
  {"Haiti", I18N_NOOP("Haiti"), 5373},
  {"Honduras", I18N_NOOP("Honduras"), 704},
  {"Hungary", I18N_NOOP("Hungary"), 403},
  {"Iceland", I18N_NOOP("Iceland"), 1172},
  {"India", I18N_NOOP("India"), 732},
  {"Indonesia", I18N_NOOP("Indonesia"), 2320},
  {"Iran", I18N_NOOP("Iran"), 8744},
  {"Iraq", I18N_NOOP("Iraq"), 131801},
  {"Ireland", I18N_NOOP("Ireland"), 103},
  {"Isle of Man", I18N_NOOP("Isle of Man"), 3073},
  {"Israel", I18N_NOOP("Israel"), 295},
  {"Italy", I18N_NOOP("Italy"), 243},
  {"Jamaica", I18N_NOOP("Jamaica"), 591},
  {"Japan", I18N_NOOP("Japan"), 226},
  {"Jordan", I18N_NOOP("Jordan"), 1711},
  {"Kazakstan", I18N_NOOP("Kazakstan"), 3608},
  {"Kenya", I18N_NOOP("Kenya"), 1661},
  {"Kiribati", I18N_NOOP("Kiribati"), 2680},
  {"Kuwait", I18N_NOOP("Kuwait"), 930},
  {"Kyrgyzstan", I18N_NOOP("Kyrgyzstan"), 119387},
  {"Laos", I18N_NOOP("Laos"), 73943},
  {"Latvia", I18N_NOOP("Latvia"), 2618},
  {"Lebanon", I18N_NOOP("Lebanon"), 5072},
  {"Lesotho", I18N_NOOP("Lesotho"), 176286},
  {"Liberia", I18N_NOOP("Liberia"), 121841},
  {"Libya", I18N_NOOP("Libya"), 23636},
  {"Liechtenstein", I18N_NOOP("Liechtenstein"), 7242},
  {"Lithuania", I18N_NOOP("Lithuania"), 269},
  {"Luxembourg", I18N_NOOP("Luxembourg"), 1170},
  {"Macau", I18N_NOOP("Macau"), 416},
  {"Madagascar", I18N_NOOP("Madagascar"), 9305},
  {"Malawi", I18N_NOOP("Malawi"), 112652},
  {"Malaysia", I18N_NOOP("Malaysia"), 149},
  {"Maldives", I18N_NOOP("Maldives"), 120415},
  {"Mali", I18N_NOOP("Mali"), 186418},
  {"Malta", I18N_NOOP("Malta"), 2636},
  {"Marshall Islands", I18N_NOOP("Marshall Islands"), 3848},
  {"Martinique", I18N_NOOP("Martinique"), 2639},
  {"Mauritania", I18N_NOOP("Mauritania"), 9373},
  {"Mexico", I18N_NOOP("Mexico"), 469},
  {"Micronesia", I18N_NOOP("Micronesia"), 68967},
  {"Monaco", I18N_NOOP("Monaco"), 6014},
  {"Mongolia", I18N_NOOP("Mongolia"), 4237},
  {"Montserrat", I18N_NOOP("Montserrat"), 154836},
  {"Morocco", I18N_NOOP("Morocco"), 4312},
  {"Mozambique", I18N_NOOP("Mozambique"), 5665},
  {"Myanmar", I18N_NOOP("Myanmar"), 112103},
  {"Namibia", I18N_NOOP("Namibia"), 791},
  {"Nauru", I18N_NOOP("Nauru"), 155789},
  {"Nepal", I18N_NOOP("Nepal"), 825},
  {"Netherlands", I18N_NOOP("Netherlands"), 120},
  {"Netherlands Antilles", I18N_NOOP("Netherlands Antilles"), 2304},
  {"New Caledonia", I18N_NOOP("New Caledonia"), 1216},
  {"New Zealand", I18N_NOOP("New Zealand"), 113},
  {"Nicaragua", I18N_NOOP("Nicaragua"), 2749},
  {"Niger", I18N_NOOP("Niger"), 118726},
  {"Nigeria", I18N_NOOP("Nigeria"), 8524},
  {"Niue", I18N_NOOP("Niue"), 167143},
  {"Northern Mariana Islands", I18N_NOOP("Northern Mariana Islands"), 304174},
  {"Norway", I18N_NOOP("Norway"), 46},
  {"Oman", I18N_NOOP("Oman"), 3247},
  {"Pakistan", I18N_NOOP("Pakistan"), 2533},
  {"Palau", I18N_NOOP("Palau"), 6894},
  {"Panama", I18N_NOOP("Panama"), 671},
  {"Papua New Guinea", I18N_NOOP("Papua New Guinea"), 2634},
  {"Paraguay", I18N_NOOP("Paraguay"), 2233},
  {"Peru", I18N_NOOP("Peru"), 1841},
  {"Philippines", I18N_NOOP("Philippines"), 221},
  {"Pitcairn Islands", I18N_NOOP("Pitcairn Islands"), 305568},
  {"Poland", I18N_NOOP("Poland"), 134},
  {"Portugal", I18N_NOOP("Portugal"), 146},
  {"Puerto Rico", I18N_NOOP("Puerto Rico"), 1116},
  {"Qatar", I18N_NOOP("Qatar"), 6524},
  {"Republic of Korea", I18N_NOOP("Republic of Korea"), 444},
  {"Republic of Moldova", I18N_NOOP("Republic of Moldova"), 5506},
  {"Reunion", I18N_NOOP("Reunion"), 3991},
  {"Romania", I18N_NOOP("Romania"), 1313},
  {"Russian Federation", I18N_NOOP("Russian Federation"), 390},
  {"Rwanda", I18N_NOOP("Rwanda"), 13104},
  {"Saint Kitts and Nevis", I18N_NOOP("Saint Kitts and Nevis"), 3102},
  {"Saint Lucia", I18N_NOOP("Saint Lucia"), 6503},
  {"Saint Pierre and Miquelon", I18N_NOOP("Saint Pierre and Miquelon"), 3565},
  {"Saint Vincent and Grenadines", I18N_NOOP("Saint Vincent and Grenadines"), 7561},
  {"Samoa", I18N_NOOP("Samoa"), 8150},
  {"San Marino", I18N_NOOP("San Marino"), 7892},
  {"Sao Tome and Principe", I18N_NOOP("Sao Tome and Principe"), 6523},
  {"Saudi Arabia", I18N_NOOP("Saudi Arabia"), 3214},
  {"Senegal", I18N_NOOP("Senegal"), 165622},
  {"Seychelles", I18N_NOOP("Seychelles"), 4662},
  {"Sierra Leone", I18N_NOOP("Sierra Leone"), 5443},
  {"Singapore", I18N_NOOP("Singapore"), 380},
  {"Slovakia", I18N_NOOP("Slovakia"), 811},
  {"Slovenia", I18N_NOOP("Slovenia"), 264},
  {"Solomon Islands", I18N_NOOP("Solomon Islands"), 8337},
  {"Somalia", I18N_NOOP("Somalia"), 7029},
  {"South Africa", I18N_NOOP("South Africa"), 882},
  {"Spain", I18N_NOOP("Spain"), 8},
  {"Sri Lanka", I18N_NOOP("Sri Lanka"), 2847},
  {"Sudan", I18N_NOOP("Sudan"), 63826},
  {"Suriname", I18N_NOOP("Suriname"), 10675},
  {"Swaziland", I18N_NOOP("Swaziland"), 5017},
  {"Sweden", I18N_NOOP("Sweden"), 43},
  {"Switzerland", I18N_NOOP("Switzerland"), 75},
  {"Syrian Arab Republic", I18N_NOOP("Syrian Arab Republic"), 50973},
  {"Taiwan", I18N_NOOP("Taiwan"), 443},
  {"Tajikistan", I18N_NOOP("Tajikistan"), 7256},
  {"Thailand", I18N_NOOP("Thailand"), 1068},
  {"Togo", I18N_NOOP("Togo"), 113781},
  {"Tokelau", I18N_NOOP("Tokelau"), 142079},
  {"Tonga", I18N_NOOP("Tonga"), 6816},
  {"Trinidad and Tobago", I18N_NOOP("Trinidad and Tobago"), 4067},
  {"Tunisia", I18N_NOOP("Tunisia"), 8477},
  {"Turkey", I18N_NOOP("Turkey"), 339},
  {"Turkmenistan", I18N_NOOP("Turkmenistan"), 4801},
  {"Turks and Caicos Islands", I18N_NOOP("Turks and Caicos Islands"), 1918},
  {"Tuvalu", I18N_NOOP("Tuvalu"), 145637},
  {"Uganda", I18N_NOOP("Uganda"), 4871},
  {"Ukraine", I18N_NOOP("Ukraine"), 687},
  {"United Arab Emirates", I18N_NOOP("United Arab Emirates"), 2659},
  {"United Kingdom", I18N_NOOP("United Kingdom"), 47},
  {"United Rep. of Tanzania", I18N_NOOP("United Rep. of Tanzania"), 4144},
  {"United States", I18N_NOOP("United States"), 7},
  {"United States Virgin Islands", I18N_NOOP("United States Virgin Islands"), 115},
  {"Uruguay", I18N_NOOP("Uruguay"), 1380},
  {"Uzbekistan", I18N_NOOP("Uzbekistan"), 8638},
  {"Vanuatu", I18N_NOOP("Vanuatu"), 3373},
  {"Venezuela", I18N_NOOP("Venezuela"), 128},
  {"Viet Nam", I18N_NOOP("Viet Nam"), 919},
  {"Wallis and Futuna Islands", I18N_NOOP("Wallis and Futuna Islands"), 6948},
  {"Western Sahara", I18N_NOOP("Western Sahara"), 5280},
  {"Yemen", I18N_NOOP("Yemen"), 5822},
  {"Yugoslavia", I18N_NOOP("Yugoslavia"), 1347},
  {"Zambia", I18N_NOOP("Zambia"), 3542},
  {"Zimbabwe", I18N_NOOP("Zimbabwe"), 1074}
};

UserStatsView::UserStatsView(QWidget *parent, const char *name)
              : KSetiSpyView(Text, parent, name)
{
  QBoxLayout *layout = new QVBoxLayout(this);

  QWidgetStack *stack = new QWidgetStack(this, "UserStatsView::stack");
  layout->addWidget(stack);

  QLabel *label = new QLabel(stack, "UserStatsView::message");
  label->setAlignment(AlignCenter);
  stack->addWidget(label, 0);

  QWidget *w = new QWidget(stack, "UserStatsView::main");
  stack->addWidget(w, 1);

  QGridLayout *grid = new QGridLayout(w, 8, 2);
  grid->setSpacing(8);

  grid->setColStretch(0, 2);
  grid->setColStretch(1, 3);

  grid->addMultiCellWidget(addField(w, "UserStatsView::user"), 0, 0, 0, 1);
  setFieldName("UserStatsView::user", i18n("User:"));

  grid->addWidget(addField(w, "UserStatsView::registered"), 1, 0);
  setFieldName("UserStatsView::registered", i18n("Registered:"));

  grid->addWidget(addField(w, "UserStatsView::total_results"), 1, 1);
  setFieldName("UserStatsView::total_results", i18n("Results returned:"));

  grid->addWidget(addField(w, "UserStatsView::total_time"), 2, 0);
  setFieldName("UserStatsView::total_time", i18n("Total CPU time:"));

  grid->addWidget(addField(w, "UserStatsView::interval"), 2, 1);
  setFieldName("UserStatsView::interval", i18n("Result interval:"));

  grid->addWidget(addField(w, "UserStatsView::avg_time"), 3, 0);
  setFieldName("UserStatsView::avg_time", i18n("Avg. CPU time:"));

  grid->addWidget(addField(w, "UserStatsView::cpus"), 3, 1);
  setFieldName("UserStatsView::cpus", i18n("CPU dedication:"));

  grid->addWidget(addField(w, "UserStatsView::venue"), 4, 0);
  setFieldName("UserStatsView::venue", i18n("Environment:"));

  grid->addWidget(addField(w, "UserStatsView::country"), 4, 1);

  grid->addMultiCellWidget(addField(w, "UserStatsView::rank"), 5, 5, 0, 1);

  grid->setRowStretch(6, 1);

  QBoxLayout *hbox = new QHBoxLayout(8);
  grid->addMultiCellLayout(hbox, 7, 7, 0, 1);

  hbox->addStretch(1);

  calendar = new WUCalendarDialog(-1, 0, "WU Calendar Dialog");

  KPushButton *button;

  button = new KPushButton(i18n("&Calendar"), w, "UserStatsView::calendar");
  button->installEventFilter(this);
  hbox->addWidget(button);
  connect(button, SIGNAL(clicked()), calendar, SLOT(show()));
  connect(kdoc, SIGNAL(updatedLog()), this, SLOT(updateLog()));

  button = new KPushButton(i18n("&Update"), w, "UserStatsView::update");
  button->installEventFilter(this);
  hbox->addWidget(button);
  connect(button, SIGNAL(clicked()), this, SLOT(updateUserStats()));
  connect(kdoc, SIGNAL(updatedUserStats(const user_stats *)),
          this, SLOT(handleUserStatsUpdates(const user_stats *)));

  stats = NULL;
  anniversary = 99;

  updateContent(false);
  updateLog();

  showMain();
}

UserStatsView::~UserStatsView()
{
  delete calendar;
}

void UserStatsView::readConfig(bool readGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("User Stats");

  anniversary = config->readNumEntry("Anniversary", 0);

  config->setGroup("Work Unit Calendar Dialog");

  if(readGeometry)
  {
    const QRect rect = geometry();
    calendar->setGeometry(config->readRectEntry("Geometry", &rect));
  }

  calendar->setWUIncrement(config->readNumEntry("Work units per day", 0));
}

void UserStatsView::saveConfig(bool saveGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("User Stats");

  config->writeEntry("Anniversary", anniversary);

  config->setGroup("Work Unit Calendar Dialog");

  if(saveGeometry)
    config->writeEntry("Geometry", calendar->geometry());

  config->writeEntry("Work units per day", calendar->wuIncrement());
}

void UserStatsView::updateContent(bool force)
{
  // optimization: do not update if this is not the visible widget
  const QWidgetStack *views = (QWidgetStack *) this->parent();
  if(views->visibleWidget() != this && !force) return;

  const SetiClientMonitor::State state = kdoc->setiMonitorState();
  const KLocale *locale = KGlobal::locale();

  const QString server = kdoc->userStatsServer();

  if(state >= SetiClientMonitor::No_WU)
  {
    SetiClientMonitor *monitor = kdoc->setiMonitor();
    const seti_data *data = monitor->setiData();

    const QString email = data->user.email;

    if(stats != NULL && stats->email != email)
    {
      delete stats;
      stats = NULL;
    }

    const QString userStatsURL = kdoc->userStatsURL(email);
    if(!userStatsURL.isEmpty())
      setFieldContentURL("UserStatsView::user", data->user.name, userStatsURL,
                         i18n("Click to visit your SETI@home personal stats web page"),
                         this, SLOT(handleURL(const QString&)));
    else
      setFieldContentText("UserStatsView::user", data->user.name);

    KPushButton *button = (KPushButton *) child("UserStatsView::update", "KPushButton");
    button->setEnabled(!email.isEmpty());

    const QDateTime register_time = data->user.registered.time;
    double register_day;
    monitor->convertToDate(data->user.registered.string, &register_day);
    if(!server.isEmpty())
      setFieldContentURL("UserStatsView::registered", locale->formatDate(register_time.date(), true),
                         QString(RegistrationClassURL).arg(server).arg(int(register_day + 0.5)),
                         i18n("Click to visit your SETI@home registration class web page"),
                         this, SLOT(handleURL(const QString&)));
    else
      setFieldContentText("UserStatsView::registered", locale->formatDate(register_time.date(), true));

    checkAnniversary(register_time.date());

    const int n_results = (stats != NULL) ? stats->n_results : data->user.n_results;
    const QString certificateURL = kdoc->certificateURL(email, n_results);
    if(!certificateURL.isEmpty())
      setFieldContentURL("UserStatsView::total_results", locale->formatNumber(n_results, 0),
                         certificateURL,
                         i18n("Click to view your latest SETI@home certificate"),
                         this, SLOT(handleURL(const QString&)));
    else
      setFieldContentText("UserStatsView::total_results", locale->formatNumber(n_results, 0));

    const double total_cpu_secs = (stats != NULL) ? stats->total_cpu : data->user.total_cpu;
    const double total_cpu_years = total_cpu_secs / (60 * 60 * 24 * 365);
    setFieldContentText("UserStatsView::total_time",
                        i18n("%1 years").arg(locale->formatNumber(total_cpu_years, 2)));

    const QDateTime last_result_time = (stats != NULL) ? stats->last_result_time
                                                       : data->user.last_result.time;
    const double interval = register_time.secsTo(last_result_time)/double(n_results);
    setFieldContentText("UserStatsView::interval", SetiClientMonitor::timeToString(int(interval)));
    setFieldToolTip("UserStatsView::interval",
                    i18n("Last result returned: %1")
                      .arg(locale->formatDateTime(last_result_time, true)));

    const double avg = (n_results > 0) ? total_cpu_secs/n_results : 0;
    setFieldContentText("UserStatsView::avg_time", SetiClientMonitor::timeToString(int(avg)));

    const double cpu_dedication = (interval > 0.0) ? avg/interval : 0.0;
    setFieldContentText("UserStatsView::cpus", locale->formatNumber(cpu_dedication, 3));

    const int venue = data->user.venue;
    const QString venueStr = formatVenue(venue);
    if(!venueStr.isEmpty())
      if(!server.isEmpty())
        setFieldContentURL("UserStatsView::venue", venueStr,
                           QString(VenueURL).arg(server).arg(venue, 0),
                           i18n("Click to see the top scores for your environment"),
                           this, SLOT(handleURL(const QString&)));
      else
        setFieldContentText("UserStatsView::venue", venueStr);
    else
      setFieldContentText("UserStatsView::venue", i18n(unknownContent));
  }
  else
  {
    setFieldContentText("UserStatsView::user", i18n(unknownContent));

    setFieldContentText("UserStatsView::registered", i18n(unknownContent));

    setFieldContentText("UserStatsView::total_results", i18n(unknownContent));
    setFieldToolTip("UserStatsView::total_results", QString::null);
    setFieldContentText("UserStatsView::total_time",  i18n(unknownContent));

    setFieldContentText("UserStatsView::interval", i18n(unknownContent));
    setFieldContentText("UserStatsView::avg_time", i18n(unknownContent));
    setFieldContentText("UserStatsView::cpus", i18n(unknownContent));

    setFieldContentText("UserStatsView::venue", i18n(unknownContent));
    
    KPushButton *button = (KPushButton *) child("UserStatsView::update", "KPushButton");
    button->setEnabled(false);
  }

  // Extra fields
  if(stats != NULL)
  {
    const double rank_percent =
      (stats->rank.total > 0) ? 1e2 * (1.0 - double(stats->rank.pos + stats->n_rank - 1)/(stats->rank.total))
                              : 0.0;
    setFieldName("UserStatsView::rank", i18n("Rank:"));
    setFieldContentText("UserStatsView::rank",
                        i18n("%1 of %2 (%3%)").arg(locale->formatNumber(stats->rank.pos, 0))
                                              .arg(locale->formatNumber(stats->rank.total, 0))
                                              .arg(locale->formatNumber(rank_percent, 3)));
    setFieldToolTip("UserStatsView::rank",
                    i18n("Users with this rank: %1").arg(locale->formatNumber(stats->n_rank, 0)));

    const int country_idx = findCountry(stats->country);
    setFieldName("UserStatsView::country", i18n("Country:"));
    if(country_idx >= 0 && !server.isEmpty())
      setFieldContentURL("UserStatsView::country", i18n(Countries[country_idx].name),
                         QString("http://%1/stats/country_%2.html").arg(server)
                                                                   .arg(Countries[country_idx].id),
                         i18n("Click to visit your SETI@home country stats web page"),
                         this, SLOT(handleURL(const QString&)));
    else if(stats->country.isEmpty())
      setFieldContentText("UserStatsView::country", i18n(unknownContent));
    else
      setFieldContentText("UserStatsView::country", stats->country);
  }
  else
  {
    setFieldName("UserStatsView::rank", QString::null);
    setFieldContentText("UserStatsView::rank", QString::null);
    setFieldToolTip("UserStatsView::rank", QString::null);

    setFieldName("UserStatsView::country", QString::null);
    setFieldContentText("UserStatsView::country", QString::null);
  }
}

void UserStatsView::updateLog()
{
  SetiClientMonitor *monitor = kdoc->setiMonitor();

  if(monitor != NULL)
    calendar->setLogData(monitor->logData());
  else {
    const QValueList<log_data> empty;
    calendar->setLogData(empty);
  }

  const seti_data *data = (monitor != NULL) ? monitor->setiData() : NULL;

  if(data != NULL) {
    const int n_results = (stats != NULL) ? stats->n_results : data->user.n_results;
    calendar->setStartWUs(n_results);
  }
}

bool UserStatsView::messageVisible()
{
  QWidgetStack *stack = (QWidgetStack *) child("UserStatsView::stack", "QWidgetStack");
  return(stack->id(stack->visibleWidget()) == 0);
}

void UserStatsView::showMessage(const QString& msg)
{
  QLabel *label = (QLabel *) child ("UserStatsView::message", "QLabel");
  label->setText(msg);

  QWidgetStack *stack = (QWidgetStack *) child("UserStatsView::stack", "QWidgetStack");
  stack->raiseWidget(0);
}

void UserStatsView::showMain()
{
  QWidgetStack *stack = (QWidgetStack *) child("UserStatsView::stack", "QWidgetStack");
  stack->raiseWidget(1);
}

void UserStatsView::checkAnniversary(const QDate& registered)
{
  QDate today = QDate::currentDate();

  if(today.day() != registered.day() || today.month() != registered.month())
    return;

  int years = today.year() - registered.year();

  if(years <= anniversary)
    return;

  anniversary = years;

  AnniversaryDialog dialog(registered, this);
  dialog.exec();
}

int UserStatsView::findCountry(const QString& country) const
{
  int start = 0;
  int range = N_COUNTRIES;

  while(range > 0) {
    const int i = start + range/2;
    const int result = country.compare(Countries[i].us_name);

    if(result == 0)
      return(i);
    else if(result > 0) {
      range = (start+range-1) - (i+1) + 1;
      start = i+1;
    } else
      range = (i-1) - start + 1;
  }

  return(-1);
}

QString UserStatsView::formatVenue(int venue) const
{
  switch(venue) {
    case 1:
      return(i18n("School"));
    case 2:
      return(i18n("Work"));
    case 3:
      return(i18n("Home"));
    default:
      ;
  }

  return(QString::null);
}

void UserStatsView::updateUserStats()
{
  showMessage(i18n("Updating from the SETI@home web site..."));
  kdoc->updateUserStats();
}

void UserStatsView::handleURL(const QString& url)
{
  kapp->invokeBrowser(url);
}

void UserStatsView::handleUserStatsUpdates(const user_stats *current)
{
  if(current == NULL)
  {
    if(messageVisible())
    {
      showMessage(i18n("Update failed. Resetting to local values..."));
      QTimer::singleShot(4*1000, this, SLOT(showMain()));
    }
    return;
  }

  QString email;
  if(stats != NULL)
    email = stats->email;
  else if(kdoc->setiMonitorState() > SetiClientMonitor::No_Data)
    email = kdoc->setiMonitor()->setiData()->user.email;
  else
    email = QString::null;

  if(current->email == email)
  {
    if(stats == NULL) stats = new user_stats;
    *stats = *current;
    updateContent(false);
    calendar->setStartWUs(current->n_results);
  }

  showMain();
}

#include "userstatsview.moc"

