/***************************************************************************
                          workunitview.cpp  -  description
                             -------------------
    begin                : Wed May 30 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>

#include <qlayout.h>
#include <qwidgetstack.h>

#include <kapplication.h>
#include <kdialog.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpushbutton.h>

#include "workunitview.h"

const QString AreciboURL = I18N_NOOP("http://www.naic.edu/");

WorkUnitView::WorkUnitView(QWidget *parent, const char *name)
             : KSetiSpyView(Text, parent, name)
{
  QBoxLayout *layout = new QBoxLayout(this, QBoxLayout::TopToBottom);
  layout->setSpacing(8);

  layout->addWidget(addField(this, "WorkUnitView::name"));
  setFieldName("WorkUnitView::name", i18n("Work unit name:"));

  layout->addWidget(addField(this, "WorkUnitView::recorded"));
  setFieldName("WorkUnitView::recorded", i18n("Recorded:"));

  layout->addWidget(addField(this, "WorkUnitView::position"));
  setFieldName("WorkUnitView::position", i18n("Position:"));

  layout->addWidget(addField(this, "WorkUnitView::constellation"));
  setFieldName("WorkUnitView::constellation", i18n("Nearest constellation:"));

  layout->addWidget(addField(this, "WorkUnitView::source"));
  setFieldName("WorkUnitView::source", i18n("Source:"));

  layout->addWidget(addField(this, "WorkUnitView::frequency"));
  setFieldName("WorkUnitView::frequency", i18n("Base frequency:"));

  layout->addStretch(1);

  QBoxLayout *hbox = new QBoxLayout(layout, QBoxLayout::RightToLeft);

  KPushButton *button = new KPushButton(i18n("&Sky Map"), this);
  button->installEventFilter(this);
  hbox->addWidget(button);
  hbox->addStretch(1);

  constellations = QDict<QString>(89, false);
  for(int i = 1; i < N_CONSTELLATIONS; i++)
    constellations.insert(Constellations[i].abbrev, &(Constellations[i].name));

  skyMapWindow = new SkyMapWindow("WorkUnitView::skymap_window");
  connect(button, SIGNAL(clicked()), skyMapWindow->topLevelWidget(), SLOT(show()));
}

WorkUnitView::~WorkUnitView()
{
  delete(skyMapWindow);
}

void WorkUnitView::readConfig(bool readGeometry)
{
  skyMapWindow->readConfig(readGeometry);
}

void WorkUnitView::saveConfig(bool saveGeometry)
{
  skyMapWindow->saveConfig(saveGeometry);
}

void WorkUnitView::updateContent(bool force)
{
  // optimization: do not update if this is not the visible widget
  const QWidgetStack *views = (QWidgetStack *) this->parent();
  if(views->visibleWidget() != this && !force) return;

  const SetiClientMonitor::State state = kdoc->setiMonitorState();

  if(state >= SetiClientMonitor::Idle)
  {
    const seti_data *data = kdoc->setiMonitor()->setiData();

    KLocale *locale = KGlobal::locale();

    setFieldContentText("WorkUnitView::name", data->wu.name);
    setFieldContentText("WorkUnitView::recorded",
                        locale->formatDateTime(data->wu.recorded.time, false, true));

    const QString ra = SetiClientMonitor::raToString(data->wu.start.ra);
    const QString dec = SetiClientMonitor::decToString(data->wu.start.dec);
    const QString ar = QString("%1").arg(locale->formatNumber(data->wu.angle_range, 3));
    setFieldContentText("WorkUnitView::position", i18n("%1 RA, %2 Dec, %3 AR").arg(ra).arg(dec).arg(ar));

    const int constellation = findNearestConstellation(data->wu.start.ra, data->wu.start.dec);
    setFieldContentURL("WorkUnitView::constellation", i18n(Constellations[constellation].name)
                                                    , i18n(ConstellationsSiteQueryURL).arg(i18n(Constellations[constellation].id))
                                                    , i18n("Click to visit web site")
                                                    , this, SLOT(handleURL(const QString &)));
    setFieldAux("WorkUnitView::constellation", QString("(%1)").arg(Constellations[constellation].abbrev));

    if(data->wu.receiver == "ao1420")
      setFieldContentURL("WorkUnitView::source", i18n("Arecibo Radio Observatory"), i18n(AreciboURL)
                                               , i18n("Click to visit web site")
                                               , this, SLOT(handleURL(const QString &)));
    else
      setFieldContentText("WorkUnitView::source", i18n(unknownContent));

    setFieldContentText("WorkUnitView::frequency",
                        i18n("%1 GHz").arg(locale->formatNumber(10e-10 * data->wu.subband.base, 9)));
  }
  else
  {
    setFieldContentText("WorkUnitView::name", i18n(unknownContent));
    setFieldContentText("WorkUnitView::recorded", i18n(unknownContent));
    setFieldContentText("WorkUnitView::position", i18n(unknownContent));
    setFieldContentText("WorkUnitView::constellation", i18n(unknownContent));
    setFieldContentText("WorkUnitView::source", i18n(unknownContent));
    setFieldContentText("WorkUnitView::frequency", i18n(unknownContent));
  }
}

void WorkUnitView::handleURL(const QString& url)
{
  kapp->invokeBrowser(url);
}

int WorkUnitView::findNearestConstellation(double ra, double dec) const
{
  int min_index = 0;
  double min_distance = coordDistance(ra, dec, Constellations[0].ra, Constellations[0].dec);

  for(int i = 1; i < N_CONSTELLATIONS; i++)
  {
    const double distance = coordDistance(ra, dec, Constellations[i].ra, Constellations[i].dec);

    if(distance < min_distance)
    {
      min_index = i;
      min_distance = distance;
    }
  }

  return(min_index);
}

double WorkUnitView::coordDistance(const double ra1, const double dec1, const double ra2, const double dec2)
{
  const double phi1 = M_PI/12 * ra1;
  const double theta1 = M_PI/180 * (90 - dec1);
  const double phi2 = M_PI/12 * ra2;
  const double theta2 = M_PI/180 * (90 - dec2);

  double x = cos(theta1)*cos(theta2) + sin(theta1)*sin(theta2)*cos(phi1 - phi2);
  x = (x > 1.0) ? 1.0 : (x < -1.0) ? -1.0 : x;

  return(acos(x));
}

#include "workunitview.moc"

