/***************************************************************************
                          wucalendar.cpp  -  description
                             -------------------
    begin                : Sun Aug 5 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpainter.h>

#include <kglobal.h>
#include <klocale.h>
#include <kdeversion.h>

#if KDE_IS_VERSION(3,2,0)
#include <kcalendarsystem.h>
#endif

#include "wucalendar.h"

WUCalendar::WUCalendar(int wus, QWidget *parent, const char *name)
           :KSetiSpyView(Pixmap, parent, name)
{
  date_start = QDate::currentDate();
  wus_start = wus;

  // set reasonable settings for the slot variables
  wus_inc = 0;
  date_current = firstOfMonth(date_start);

  // clear the cache
  setupCache();

  // black background
  setBackgroundMode(PaletteBase);
  setPalette(black);

  // set minimum size; can grow, but not shrink, from that
  {
    QFontMetrics metrics = this->fontMetrics();
    const int lineHeight = this->fontMetrics().lineSpacing();
    const int cellWidth = metrics.width("_________");

    setMinimumSize(QSize(2*lineHeight + 7*(cellWidth + 4 * border),
                         3*lineHeight + 6*(2*lineHeight + 4 * border)));
  }
  setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum));
}

WUCalendar::~WUCalendar()
{
}

QDate WUCalendar::currentDate() const
{
  return(date_current);
}

int WUCalendar::startWUs() const
{
  return(wus_start);
}

double WUCalendar::wuIncrement() const
{
  return(wus_inc);
}

void WUCalendar::addMonths(int months)
{
  if(months == 0)
    return;

  int months_old = date_current.year() * 12 + (date_current.month() - 1);
  int months_new = months_old + months;
  date_current = QDate(months_new / 12, months_new % 12 + 1, 1);
  setupCache();

  repaint();
}

void WUCalendar::setStartWUs(int wus)
{
  if(wus == wus_start) return;

  // setting the WU's updates the start date
  date_start = QDate::currentDate();
  wus_start = wus;

  repaint();
}

void WUCalendar::setWUIncrement(double inc)
{
  if(inc == wus_inc) return;

  wus_inc = inc;

  repaint();
}

void WUCalendar::setLogData(const QValueList<log_data>& data)
{
  if(log == data) return;

  log = data;
  setupCache();

  repaint();
}

void WUCalendar::paintEvent(QPaintEvent *)
{
  QPainter painter(this);
  KLocale *locale = KGlobal::locale();

  QFontMetrics metrics = painter.fontMetrics();
  const int lineHeight = metrics.lineSpacing();

  // draw on screen the name of the month and the year
  QRect header(QPoint(lineHeight, lineHeight), QPoint(this->width() - lineHeight, this->height() - lineHeight));
  #if KDE_IS_VERSION(3,2,0)
  const QString month = locale->calendar()->monthName(date_current, false);
  #else
  const QString month = locale->monthName(date_current.month());
  #endif

  painter.setPen(yellow);
  painter.drawText(header, AlignLeft | AlignTop, i18n("%1, %2").arg(month).arg(date_current.year()));

  // as many columns as days of the week, and 6 rows are enough to acommodate all months
  const uint columns = 7;
  const uint rows = 6;

  // day grid rectangle
  QRect grid(QPoint(lineHeight, 2*lineHeight), header.bottomRight());
  // text area inside a day grid cell
  QRect textArea(QPoint(2*border, border), QSize(grid.width()/columns - 4*border, grid.height()/rows - 2*border));

  // offset is the column offset for the first day of the month
  const int offset =  (date_current.dayOfWeek() + 7 - locale->weekStartDay()) % 7;

  // draw the day grid rectangle and its contents
  painter.setPen(white);
  painter.drawRect(grid);
  for(uint i = 0; i < rows; i++)
    for(uint j = 0; j < columns; j++)
    {
      painter.save();

      painter.translate(lineHeight + double(grid.width() * j)/columns,
                        2*lineHeight + double(grid.height() * i)/rows);

      painter.setPen(white);
      if(i > 0 && j == 0)
        painter.drawLine(0, 0, grid.width(), 0);
      if(i == 0 && j > 0)
        painter.drawLine(0, 0, 0, grid.height());

      // draw the cell interior for meaningful days
      const int day = i * 7 + j - offset + 1;
      if(day > 0 && day <= date_current.daysInMonth())
      {
        const QDate date_day = date_current.addDays(day-1);

        if(date_day > date_start)
        {
          // draw the number of future wu's in blue, provided it is meaningful (i.e. nonnegative)
          const int wus = wus_start + int(date_start.daysTo(date_day) * wus_inc);

          if(wus_start >= 0 && wus >= 0)
          {
            painter.setPen(QColor(0, 231, 231));
            painter.drawText(textArea, AlignRight | AlignBottom, QString::number(wus));
          }
        }
        else
        {
          // draw the number of wu's in the log, if it is greater than zero
          const int wus = cache[day-1];

          if(wus > 0)
          {
            // compute the rect of this cell
            const int w = int(double(grid.width() * (j+1))/columns) - int(double(grid.width() * j)/columns);
            const int h = int(double(grid.height() * (i+1))/rows) - int(double(grid.height() * i)/rows);
            const QRect cell(QPoint(1, 1), QSize(w-2, h-2));

            painter.fillRect(cell, darkRed);

            painter.setPen(yellow);
            painter.drawText(textArea, AlignRight | AlignBottom, QString::number(wus));
          }
        }

        painter.setPen(white);
        painter.drawText(textArea, AlignLeft | AlignTop, QString::number(day));
      }

      painter.restore();
    }
}

void WUCalendar::pixmapToClipboard()
{
  pixmap = QPixmap::grabWidget(this);
  KSetiSpyView::pixmapToClipboard();
}

void WUCalendar::pixmapToFile()
{
  pixmap = QPixmap::grabWidget(this);
  KSetiSpyView::pixmapToFile();
}

void WUCalendar::setupCache()
{
  if(date_current > firstOfMonth(date_start)) return;

  for(uint i = 0; i < 31; i++)
    cache[i]=0;

  for(uint i = 0; i < log.count(); i++)
    if(firstOfMonth(log[i].done.date()) == date_current)
      cache[log[i].done.date().day()-1]++;
}

QDate WUCalendar::firstOfMonth(const QDate& date)
{
  return(QDate(date.year(), date.month(), 1));
}

#include "wucalendar.moc"

