/*
	appscheduler.h - applications scheduler
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __APPSCHEDULER_H__
#define __APPSCHEDULER_H__

#include "mmainwindow.h"

#define REMOVE_DEAD_TASKS_TIMEOUT 300000

/** @short Task info. This class is used by the @ref AppScheduler. */
class AppInfo: public QObject
{
public:
	/**
	 * Constructor.
	 * @param id A task ID
	 * @param name An application name
	 * @param description A task description
	 * @param action An action associated with this task
	 */
	AppInfo(const int id, const QString &name, const QString &description,
		const int action);

	/** Destructor. */
	virtual ~AppInfo();

	/** Returns the action associated with this task. */
	inline int getAction() const { return _action; }

	/** (NOT USED) Returns the registered application ID string. */
	inline QString getAppId() const { return _appId; }

	/** Returns the registered task description text. */
	inline QString getDescription() const { return _description; }

	/** Sets the task description to @p value. */
	inline void setDescription(const QString &value) { _description = value; }

	/** Returns the registered task ID. */
	inline int getId() const { return _id; }

	/** Sets the task ID to @p value. */
	inline void setId(const int value) { _id = value; }

	/** Returns the registered application name. */
	inline QString getName() const { return _name; }
private:
	int
		_action,
		_id;
	QString
		/** Application ID string */
		_appId,
		/** Task description */
		_description,
		/** Application name */
		_name;
};

/**
 * @short Applications scheduler.
 * See Handbook (F1) for details.
 */
class AppScheduler: public QObject
{
	Q_OBJECT
public:
	/** Constructor. */
	AppScheduler();

	/** Destructor. */
	virtual ~AppScheduler();

	/** Returns @c TRUE if Scheduler is enabled. */
	bool isEnabled() const;

	/** Returns the list with registered tasks. */
	inline QValueList<AppInfo *> *list() const { return _list; }

	/**
	 * Translates DCOP API action value (ACTION_*)
	 * to the main window action (@ref MMainWindow::What).
	 */
	MMainWindow::What translateAction(const int action) const;

	// DCOP
	/** @see MMainWindow::activateAction */
	virtual bool activateAction(int id);

	/** @see MMainWindow::registerTask */
	virtual int registerTask(const QString &name, const QString &description,
		int action);

	/** @see MMainWindow::unregisterTask */
	virtual bool unregisterTask(int id);
private:
	int _newId; // Used to generate an unique task ID.
	QTimer *_removeDeadTasksTimer;
	QValueList<AppInfo *> *_list;

	/**
	 * Searches for @p id.
	 * @param id An ID to find
	 * @return An application info or @c 0 if the ID wasn't found
	 */
	QValueList<AppInfo *>::iterator findId(int id);

	/** Displays "The scheduler is disabled!" text. */
	void showStatusInfo();
private slots:
	/** Removes all "dead" tasks. */
	void removeDeadTasksSlot();
};

#endif // __APPSCHEDULER_H__
