/*
	main.cpp - KShutDown
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "../config.h"
#include "mmainwindow.h"

#include <qspinbox.h>

#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kuniqueapplication.h>

// 0.1.x, 0.3.x, etc - Unstable, Beta
// 0.2.x, 0.4.x, etc - Stable
QString version = QString("%1 (build: %2 %3)")
	.arg(VERSION)
	.arg(__DATE__)
	.arg(__TIME__);

KAboutData aboutData(
	"kshutdown", // internal name
	"KShutDown", // full name
	version,
	I18N_NOOP("A Shut Down Utility for KDE"), // description
	KAboutData::License_GPL_V2, // license
	"(C) 2003-3000 Konrad Twardowski", // copyright
	"", // text
	"http://kshutdown.sourceforge.net/", // project page
	"kdtonline@poczta.onet.pl" // bugs
);

static KCmdLineOptions options[] =
{
	{ "s", 0, 0 },
	{ "shutdown", I18N_NOOP("Turn off computer"), 0 },
	{ "h", 0, 0 },
	{ "halt", I18N_NOOP("Turn off computer"), 0 },
	{ "r", 0, 0 },
	{ "reboot", I18N_NOOP("Restart computer"), 0 },
	{ "k", 0, 0 },
	{ "lock", I18N_NOOP("Lock session"), 0 },
	{ "l", 0, 0 },
	{ "logout", I18N_NOOP("End current session"), 0 },
	{ "w", 0, 0 },
	{ "wizard", I18N_NOOP("Wizard"), 0 },
	{ "standard", I18N_NOOP("Show standard logout dialog"), 0 },
	{ "c", 0, 0 },
	{ "cancel", I18N_NOOP("Cancel an active action"), 0 },
	{ "init", I18N_NOOP("Don't show window at startup"), 0 },
	{ "test", I18N_NOOP("Enable test mode"), 0 },
	{ "default", I18N_NOOP("Disable test mode"), 0 },
	{ "+[time]", I18N_NOOP("Time; Examples: 01:30 - absolute time (HH:MM); " \
		"10 - number of minutes to wait from now"), 0 },
	{ 0, 0, 0 }
};

/** @short KShutDown application. */
class KShutDownApplication: public KUniqueApplication
{
public:
	/** Constructor. */
	KShutDownApplication()
		: KUniqueApplication(),
		isTimeArg(FALSE),
		now(FALSE),
		timeArg(),
		timeArgIsValid(FALSE)
	{
	}

	/** Initializes the main window, and checks command line arguments. */
	virtual int newInstance();
private:
	enum Mode {
		Mode_Visible = -1,
		Mode_Hidden = -2,
		Mode_Ok = 0,
		Mode_Error = 1
	};
	bool isTimeArg;
	bool now;
	QString timeArg;
	bool timeArgIsValid;
	int doExecAction(MMainWindow::What action);
	int processArgs(KCmdLineArgs *args);
};

/* public */

int KShutDownApplication::newInstance()
{
	static bool doShow = FALSE;
	bool doRaise = (mainWindow != 0);

	// create main window (only one)
	if (!mainWindow)
	{
		(void)new MMainWindow(0, "MMainWindow::mainWindow");
		kapp->setMainWidget(mainWindow);
	}

	// check command line args
	KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
	int retval = processArgs(args);
	args->clear();

	if (retval == Mode_Visible || retval == Mode_Hidden) // gui mode
	{
		if (retval == Mode_Visible && (!kapp->isRestored() || doShow))
		{
			mainWindow->show();
			if (doRaise)
				mainWindow->makeVisible();
		}
		doShow = TRUE;

		return Mode_Ok;
	}

	return retval; // command line exit code
}

/* private */

int KShutDownApplication::doExecAction(MMainWindow::What action)
{
	if (!mainWindow)
		return Mode_Error;

	// use time arg.
	if (timeArgIsValid)
	{
		mainWindow->cancel(); // cancel any running action
		if (!now)
		{
			mainWindow->setActive(
				action,
				TRUE, // activate
				FALSE // no confirmation
			);

			return Mode_Ok;
		}
	}

	bool stopTimer = action != MMainWindow::What_LockScreen;
	if (mainWindow->execAction(action, stopTimer));
		return Mode_Ok;

	return Mode_Error;
}

int KShutDownApplication::processArgs(KCmdLineArgs *args)
{
	if (!mainWindow)
		return Mode_Error;

	isTimeArg = FALSE;
	now = FALSE;
	timeArg = "";
	timeArgIsValid = FALSE;

	// read time arg.
	bool absoluteTime = FALSE;
	int minutes = 0;
	QDateTime dt = QDateTime::currentDateTime();
	if (args->count())
	{
		timeArg = args->arg(0); // first arg. is a "time" arg.
		if (!timeArg.isEmpty())
		{
			isTimeArg = TRUE;

			if (timeArg.upper() == "NOW")
			{
				now = TRUE;
				timeArgIsValid = TRUE;
			}
			else
			{
				bool ok = FALSE;
				minutes = timeArg.toInt(&ok); // MM?
				if (ok)
				{
					absoluteTime = FALSE;
				}
				else
				{
					absoluteTime = TRUE;
// TODO: 0.7: AM/PM
					QTime t = QTime::fromString(timeArg);
					if (t.isValid() && !t.isNull()) // HH:MM[:SS]?
					{
						ok = TRUE;
						dt.setTime(t);
					}
				}
				if (!ok || !dt.isValid() || dt.isNull())
				{
					QString s = QString(i18n("Invalid time: <b>%1</b>"))
						.arg(timeArg);
					KMessageBox::error(mainWindow, "<qt>" + s + "</qt>");
				}
				else
				{
					timeArgIsValid = TRUE;
				}
			}
		}
	}

	// test mode
	if (args->isSet("test"))
		mainWindow->setTestMode(TRUE);

	// normal mode
	if (args->isSet("default"))
		mainWindow->setTestMode(FALSE);

	// cancel
	if (args->isSet("cancel"))
	{
		mainWindow->cancel();

		return Mode_Ok;
	}

	// wizard
	if (args->isSet("wizard"))
	{
		if (mainWindow->runWizard())
			mainWindow->makeVisible();

		return Mode_Ok;
	}

	// standard logout dialog
	if (args->isSet("standard"))
	{
		kapp->requestShutDown(
			KApplication::ShutdownConfirmYes,
			KApplication::ShutdownTypeDefault,
			KApplication::ShutdownModeDefault
		);

		return Mode_Ok;
	}

	// do nothing ..
	if (isTimeArg && !timeArgIsValid)
		return Mode_Error;

	// setup main window
	if (timeArgIsValid)
	{
		if (!now)
		{
			if (mainWindow->isRestricted("kshutdown_tab_time"))
				return Mode_Error;

			mainWindow->cancel(); // cancel action before "setDelayType"
			if (absoluteTime)
			{
				mainWindow->setDelayType(MMainWindow::DelayType_DateTime);
				mainWindow->setDate(dt.date());
				mainWindow->setTime(dt.time());
			}
			else
			{
				mainWindow->setDelayType(MMainWindow::DelayType_Misc);
				mainWindow->setDelayMisc(MMainWindow::DelayMisc_Minutes);
				mainWindow->delay()->setValue(minutes);
			}
		}
		mainWindow->makeVisible();

		// exec action below ..
	}

	// halt
	if (args->isSet("shutdown") || args->isSet("halt"))
		return doExecAction(MMainWindow::What_ShutDown);

	// reboot
	if (args->isSet("reboot"))
		return doExecAction(MMainWindow::What_Reboot);

	// lock
	if (args->isSet("lock"))
		return doExecAction(MMainWindow::What_LockScreen);

	// logout
	if (args->isSet("logout"))
		return doExecAction(MMainWindow::What_Logout);

	if (args->isSet("init"))
		return Mode_Hidden; // init window, but don't show it

	return Mode_Visible; // gui mode
}

/* main */

int main(int argc, char **argv)
{
	aboutData.addAuthor("Konrad Twardowski", I18N_NOOP("Maintainer"),
		"kdtonline@poczta.onet.pl", "http://www.kdtonline.prv.pl/");
	aboutData.addAuthor("Caryn \"Hellchick\" Law",
		"Female Quake III Arena Announcer Voice Files",
		"hellchick ( at ) planetquake.com",
		"http://www.planetquake.com/voxfeminae/");

	// credits
	aboutData.addCredit("Arend van Beelen jr.", I18N_NOOP("Ideas"), "",
		"http://www.liacs.nl/~dvbeelen");

	aboutData.addCredit("Bram Schoenmakers", "Dutch translation",
		"bram_s ( at ) softhome.net", "http://home.wanadoo.nl/bram_s");

	aboutData.addCredit("Charles Barcza",
		"Hungarian translation, blackPanther-Linux RPM",
		"kbarcza ( at ) blackpanther.hu", "http://www.blackpanther.hu/");

	aboutData.addCredit("Elias Probst", "Gentoo ebuilds, German translation",
		"elias.probst ( at ) gmx.de");

	aboutData.addCredit("Giovanni Venturi", "Italian translation",
		"jumpyj ( at ) tiscali.it");

	aboutData.addCredit("Guido Tack", I18N_NOOP("Ideas"), "",
		"http://www.ps.uni-sb.de/~tack");

	aboutData.addCredit("Jozef Riha", "Slovak translation",
		"zefo ( at ) seznam.cz");

	aboutData.addCredit("Karol Adamczyk [rampage]", "Gentoo ebuild");

	aboutData.addCredit("Matrix", "SuSE RPM", "matrix ( at ) ehelp.pl", "http://micek.no-ip.com/~matrix/");

	aboutData.addCredit("Michael Goettsche", I18N_NOOP("Bug reports"));

	aboutData.addCredit("Philipp Weissenbacher");

	aboutData.addCredit("Piotr Budny");

	aboutData.addCredit("Quique", "Spanish translation",
		"quique ( at ) sindominio.net");

	aboutData.addCredit("Robert Kratky", "Czech translation",
		"kratky ( at ) rob.cz");

	aboutData.addCredit("Spider (ALT Linux)", "Russian translation");

	aboutData.addCredit("Zdenko Podobny", "Slovak translation, Mandrake RPM",
		"zdpo ( at ) mailbox.sk");

	aboutData.addCredit("KDE-APPS.org", "", "", "http://www.kde-apps.org/");
	aboutData.addCredit("Lock/Logout Applet Team");
	aboutData.addCredit("SourceForge.net", "", "",
		"http://www.sourceforge.net/");

	// init cmd line
	KCmdLineArgs::init(argc, argv, &aboutData);
	KCmdLineArgs::addCmdLineOptions(options);

	if (!KUniqueApplication::start())
		return 0;

	// init application
	KShutDownApplication software;

	// main loop
	return software.exec();
}
