/*
	mmainwindow.h - main window
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MMAINWINDOW_H__
#define __MMAINWINDOW_H__

#include "kshutdowniface.h"

#include <qdatetime.h>

#include <kapplication.h>
#include <kmainwindow.h>

class AppScheduler;
class Config;
class Extras;
class MMessageDialog;
class MSchedulerTab;
class MStatsTab;
class MSystemTray;

class QLabel;
class QSpinBox;
class QTabWidget;
class QTimeEdit;
class QToolButton;
class QVGroupBox;

class KDateWidget;
class KPushButton;

#define mainWindow MMainWindow::mainAppWindow()
#define kshutdownrc MMainWindow::mainAppWindow()->configuration()

/** @short Main window. */
class MMainWindow: public KMainWindow, virtual public KShutdownIface
{
	Q_OBJECT
public:
	/** Delays for @ref DelayType_Misc. Don't change these values! */
	enum DelayMisc {
		/** Delay in seconds. */
		DelayMisc_Seconds = 0,
		/** Delay in minutes. */
		DelayMisc_Minutes = 1,
		/** Delay in hours. */
		DelayMisc_Hours = 2
	};

	/** Action methods. Don't change these values! */
	enum Method {
		/** Standard KDE/KDM method (default). */
		Method_KDE = 0,
		/** Default external command (e.g. /sbin/reboot). */
		Method_DefaultCommand = 1,
		/** User command. Any command that can be executed by @c KRun::run. */
		Method_UserCommand = 2
	};

	/** Action types. */
	enum What {
		/** Nothing (null). */
		What_Nothing = 0,
		/** System shut down. */
		What_ShutDown = 1,
		/** System reboot. */
		What_Reboot = 2,
		/** Screen lock. */
		What_LockScreen = 3,
		/** End current session. */
		What_Logout = 4,
		/** Extras. Execute an item selected from the "Extras..." menu. */
		What_Extras = 5
	};

	/** Timeout options. Don't change these values! */
	enum DelayType {
		/** Immediate action (no delay). */
		DelayType_Now = 0,
		/**
		 * Delay in seconds, minutes, or hours.
		 * @see DelayMisc
		 */
		DelayType_Misc = 1,
		/** Time from now in HH:MM format. */
		DelayType_TimeFromNow = 2,
		/** Absolute date and time. */
		DelayType_DateTime = 3
	};

	/**
	 * Constructor.
	 * @param parent A parent widget
	 * @param name A widget name
	 */
	MMainWindow(QWidget *parent = 0, const char *name = 0);

	/** Destructor. */
	virtual ~MMainWindow();

	/**
	 * Returns @c TRUE if the action has been cancelled by the user.
	 * Used in @ref MMessageDialog.
	 */
	inline bool actionCancelled() const { return _cancelled; }

	/** Returns the "Application scheduler". */
	inline AppScheduler *appScheduler() const { return _appScheduler; }

	/** Returns the global KShutDown configuration. */
	inline Config *configuration() const { return _configuration; }

	/** Returns an active action. */
	inline What currentAction() const { return _what; }

	/** Returns the spin box. */
	inline QSpinBox *delay() const { return sb_delay; }

	/**
	 * Ends current session.
	 * This function is a wrapper for the @c kapp->requestShutDown().
	 * @return @c TRUE if successful; otherwise @c FALSE
	 */
	bool endSession(const KApplication::ShutdownType type, const What action);

	/**
	 * Executes an action.
	 * @param action An action to execute
	 * @param stopTimer If @c TRUE, the current active action and its timer are stopped
	 * @return @c TRUE if successful; otherwise @c FALSE
	 */
	bool execAction(const What action, const bool stopTimer = TRUE);

	/** Returns the "Extras" manager. */
	inline Extras *extras() const { return _extras; }

	/** Returns an icon of @p action. */
	QPixmap getActionIcon(const What action) const;

	/**
	 * Returns an action name as text.
	 * @param action An action
	 * @see getCurrentActionName
	 */
	QString getActionName(const What action) const;

	/**
	 * Returns current action name as text.
	 * @see getActionName
	 */
	QString getCurrentActionName() const;

	/** Returns a current delay. */
	int getCurrentDelayMisc() const;

	/** Returns a current delay type. */
	int getCurrentDelayType() const;

	/** Returns a current delay type. */
	inline DelayType getDelayType() const { return _delayType; }

	/** Sets delay type to @p value. */
	void setDelayType(const DelayType value);

	/**
	 * Reads method for action from the configuration file.
	 * @param configEntry A configuration entry
	 * @param defaultCommand A default command
	 * @retval method A method to use
	 * @retval command A command to run
	 */
	void getMethod(
		const QString &configEntry,
		Method &method,
		QString &command,
		const QString &defaultCommand);

	/** Returns the tab widget. */
	inline QTabWidget *getTabs() const { return _tabs; }

	/** Returns the time editor. */
	QTime getTime() const;

	/** Sets editor time to @c time. */
	void setTime(const QTime &time);

	/** Returns @c TRUE if action is active. */
	inline bool isActive() const { return _active; }

	/**
	 * Returns @c TRUE if @p key is restricted (KDE Kiosk).
	 * The key value is read from the configuration file
	 * (usually /usr/share/config/kshutdownrc).
	 * See Handbook (F1) for details.
	 */
	bool isRestricted(const QString &key) const;

	/** Returns @c TRUE if test mode is enabled. (--test option) */
	inline bool isTestMode() const { return _testMode; }

	/** Returns the main window widget. Accessed via @c mainWindow macro. */
	inline static MMainWindow *mainAppWindow() { return mainApplicationWindow; }

	/**
	 * Creates and shows a new modal or non-modal message dialog.
	 * Non-modal message dialog is used in @ref AppScheduler.
	 * @param modal If @c TRUE, the dialog is modal and blocks the main window.
	 * @param actionToExecute An action to execute after timeout, or after
	 * button click
	 * @param text A text to display
	 */
	bool messageDialogAccepted(const bool modal, const What actionToExecute,
		const QString &text);

	/**
	 * Visual and sound notifications.
	 * @param secs Seconds
	 */
	void notifyUser(const int secs);

	/**
	 * Launches the Wizard dialog, and executes the selected action.
	 * @return @c TRUE if successfull; otherwise @c FALSE
	 */
	bool runWizard();

	/**
	 * Activates/deactivates an action.
	 * @param action An action to activate/deactivate
	 * @param yes @c TRUE - activate, @c FALSE - deactivate
	 * @param confirmation @c FALSE - no confirmation
	 * @see isActive
	 */
	void setActive(const What action, const bool yes,
		const bool confirmation = TRUE);

	/**
	 * Notifies one or more message dialogs that all actions have been
	 * cancelled.
	 * @see actionCancelled
	 */
	inline void setCancelled(const bool yes) { _cancelled = yes; }

	/** Sets editor date to @p date. */
	void setDate(const QDate &date);

	/** Sets delay to @p value. */
	void setDelayMisc(const DelayMisc value);

	/** Returns the system tray. */
	MSystemTray *tray();

	/**
	 * Updates the "Scheduler" tab.
	 * @param focus If @c TRUE the tab is raised/focused
	 * @see MSchedulerTab::update
	 */
	void updateSchedulerTab(const bool focus = FALSE);

	// DCOP:
	// general:
	virtual void cancel();
	virtual bool shutDown();
	virtual bool shutdown();
	virtual bool reboot();
	virtual bool lockScreen();
	virtual bool logout();
	// scheduler:
	virtual bool activateAction(int id);
	virtual int registerTask(const QString &name, const QString &description,
		int action);
	virtual bool unregisterTask(int id);
	// misc:
	virtual void configure();
	virtual QString getStatusInfo();
	virtual void makeVisible();
	virtual void setTestMode(bool yes);
	virtual void wizard();
private:
	AppScheduler *_appScheduler;
	bool
		_active,
		_cancelled,
		_testMode,
		_totalExit;
	Config *_configuration;
	DelayMisc _delayMisc;
	Extras *_extras;
	int
		_lastSettingsPage,
		_oldSec;
	KDateWidget *de_date; // date editor
	KPushButton
		// main buttons
		*b_shutDown,
		*b_reboot,
		*b_lockScreen,
		*b_logout,
		*b_wizard,
		*b_extras,
		*b_cancel;

	// points to the main window widget (this)
	static MMainWindow *mainApplicationWindow;

	MMessageDialog *_modalMessageDialog;
	MSchedulerTab *_schedulerTab;
	MStatsTab *_statsTab;
	MSystemTray *_systemTray;
	QComboBox
		*cb_delayMisc,
		*cb_delayType;
	QDateTime
		dt_end, // end date/time
		dt_start; // start date/time
	QLabel
		*l_date,
		*l_statusLine,
		*l_timeFromNow,
		*l_timeHHMM;
	QSpinBox *sb_delay; // delay editor (seconds, minutes, or hours)
	QTabWidget *_tabs; // tabs
	QTimeEdit *te_time; // time editor
	QTimer *_timer; // timeout timer
	QToolButton
		// time tab
		*b_misc;
	QVGroupBox *gb_delayValues;
	What _what; // selected action
	DelayType _delayType;
	void calcSelectedTime();
	void initBottomButtons(QWidget *parent);
	void initButtons(QWidget *parent);
	QWidget *initTimeTab(QWidget *parent);

	/**
	 * Sets window and system tray captions.
	 * @param remainingTime A remaining time
	 * @param selectedTime A selected time
	 */
	void setCaptions(const QString &remainingTime = QString::null, const QString &selectedTime = QString::null);

	/** Executes the active action after timeout. */
	void totalShutDown();

	/**
	 * Updates time infos (e.g. window captions).
	 * Displays time to the final shut down, and the selected action name.
	 */
	void updateTimeInfo(const int secs);
private slots:
	/**
	 * This signal is emitted when the application is about to quit, or when the
	 * user shuts down the entire desktop session.
	 */
	void aboutToQuitSlot();

	/** Checks the time, and after timeout executes active action. */
	void checkTimeSlot();

	/**
	 * Called when a new item has been selected in the combo box.
	 * @param index A new item index
	 */
	void delayMiscChangeSlot(int index);

	/**
	 * Called when a new item has been selected in the combo box.
	 * @param index A new item index
	 */
	void delayTypeChangeSlot(int index);

	void launchHelpSlot();

	/** Resets date/time in the date or time editor. */
	void miscSlot();

	/** Displays the settings dialog. */
	void settingsSlot();

	/**
	 * Called when a new tab has been selected.
	 * Used to initialize tabs "on demand".
	 */
	void tabChangedSlot(QWidget *tab);

	/** Launches the Wizard. */
	void wizardSlot();
public slots:
	/** Cancels an active action. */
	void cancelSlot();
	/** Exists the application. */
	void closeSlot();
	/** Locks the screen. */
	void lockScreenSlot();
	/** Ends current session. */
	void logoutSlot();
	/** Reboots the machine. */
	void rebootSlot();
	/** Halts the machine. */
	void shutDownSlot();
	/**
	 * Updates "Selected time:" info.
	 * @see calcSelectedTime
	 */
	void updateTimeInfoSlot();
protected:
	/** Closes or minizes the main window. */
	virtual bool queryClose();
	/**
	 * Invoked at application exit.
	 * @return @c TRUE
	 */
	virtual bool queryExit();
};

#endif // __MMAINWINDOW_H__
