/***************************************************************************
 *   Copyright (C) 2004-2005 by Giovanni Venturi                           *
 *   gventuri73@tiscali.it                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <pcap.h>
#include <errno.h>

#include <kurl.h>
#include <kdebug.h>
#include <klocale.h>

#include "sniffer.h"
#include "packetmanager.h"
#include "packetstructures.h"

Sniffer::Sniffer() : QThread()
{
  m_bSniffing = false;   // not sniffing yet
  m_frameNumber = 1;     // set the frame number
  m_bFromFile = false;   // the default capture is live
  m_canSniff = true;     // ok, now you can sniff
}


// set all sniffing options relative to the interface
Sniffer::Sniffer(const QString& strInterface) : QThread()
{
  m_strInterface = strInterface;
  m_bSniffing = false;  // not sniffing yet
  m_frameNumber = 1;    // set the frame number
  m_bFromFile = false;  // the default capture is live
  m_canSniff = true;    // ok, now you can sniff
}


// set all sniffing options relative to the interface and to the filter
Sniffer::Sniffer(const QString& strInterface, const QString& strFilter) : QThread()
{
  m_strInterface = strInterface;
  m_strFilter = strFilter;
  m_bSniffing = false;   // not sniffing yet
  m_frameNumber = 1;     // set the frame number
  m_bFromFile = false;   // the default capture is live
  m_canSniff = true;     // ok, now you can sniff
};


// set all sniffing options
Sniffer::Sniffer(CaptureOptions *captureOption) : QThread()
{
  setCaptureOptions(captureOption);
  m_bSniffing = false;  // not sniffing yet
  m_frameNumber = 1;    // set the frame number
  m_bFromFile = false;  // the default capture is live
  m_canSniff = true;    // ok, now you can sniff
}


Sniffer::~Sniffer()
{
  m_canSniff = false;  // no need to sniff anymore
}


// reset frame number
void Sniffer::resetFrameNumber()
{
  m_frameNumber = 1;  // reset frame number
}


// set all sniffing options
void Sniffer::setCaptureOptions(CaptureOptions *captOpt)
{
  m_strInterface = captOpt->getStrInterface();
  m_strFilter = captOpt->getStrFilter();
}


// start sniffing on the selected interface
void Sniffer::startSniffing()
{
  m_frameNumber = 1;  // reset the number
  m_canSniff = true;  // ok, now you can sniff
  if (!m_bSniffing)
  {
    // if the Sniffing Thread is not active yet, active it now
    m_bSniffing = true;
    start();
  }
}


// stop sniffing on the selected interface
void Sniffer::stopSniffing()
{
  // stop sniffing: the capture cycle into run() exits at next packet arrived and
  // m_bSniffing will be false and the Sniffing Thread stops
  m_canSniff = false;
}


// check if a file is in a valid libpcap format
int Sniffer::checkFile( QString strFile )
{
  int ret;   // return condition
  char errbuf[PCAP_ERRBUF_SIZE];

  // pcap_open_offline(...) for capture from a file needs it's a local one
  pcap_t *pd;
  pd = pcap_open_offline( strFile, errbuf );

  if (pd == NULL)
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    ret = BAD_FORMAT_FILE;
  }
  else
  {
    ret = RIGHT_FORMAT_FILE;
    m_strFilename = strFile;
    pcap_close(pd);
  }

  return ret;
}


void Sniffer::run()
{
  if (m_bFromFile)
    startSniffingFromFile();
  else
    startSniffingInterface();
}


void Sniffer::startSniffingFromFile( QString strFile )
{
  m_strFilename = strFile;
  m_canSniff = true;  // ok, now you can sniff
  startSniffingFromFile();
  m_canSniff = false;
}

void Sniffer::startSniffingFromFile()
{
  m_bFromFile = true;
  m_canSniff = true;  // ok, now you can sniff

  // set all parameters to prepare sniffing structures
  struct bpf_program bpfp;
  char errbuf[PCAP_ERRBUF_SIZE];

#ifdef MUST_DO_SELECT
  int selRet = 0;
  m_pcap_fd = 0;
#endif

  kdDebug() << "Ok, sniffing from file" << endl;
  m_pcapfp = pcap_open_offline( m_strFilename, errbuf );
  if ( m_pcapfp == NULL )
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    pcap_close( m_pcapfp );
    return;
  }
#ifdef MUST_DO_SELECT
  m_pcap_fd = pcap_fileno(m_pcapfp);
#endif

  // start reading packets on the selected source
  while (m_canSniff)
  {
    struct pcap_pkthdr hdr;
    int linkhdr_len;
    struct IpHdr *ip;
    unsigned char *p;

    p = (unsigned char*) pcap_next(m_pcapfp, &hdr);

    if (p == NULL)
        break;     // file is end

    // we got a packet: raw bytes (p), the timestamp and the captured packet lenght and the real packet length (hdr)
    /*
     *  hdr is a struct pcap_pkthdr:
     *    ts - a struct timeval containing the time when the packet was captured
     *    caplen - a bpf_u_int32 giving the number of bytes of the packet that are available from the capture
     *    len - a bpf_u_int32 giving the length of the packet, in bytes (which might be more than the number
     *      of bytes available from the capture, if the length of the packet is larger than the maximum
     *      number of bytes to capture)
     */

    int frameLength, frameType;
    setHeaderData(m_pcapfp, frameType, frameLength);
    if (frameLength < 0)
    {
      kdDebug() << "unknown packet" << endl;
      continue;
    }

    // now we call PacketManager to store:
    //  1 - the raw data
    //  2 - the timestamp and the captured packet length and the real packet lenght
    //  3 - the frame type string
    //  4 - the frame length: we need this to know how to get the data from the frame
    emit gotPacket(p, hdr, m_frameNumber++, frameType, frameLength);
  }

  kdDebug() << "Thread stopped" << endl;

  // release libpcap resource
  pcap_close(m_pcapfp);

  // now the thread has stopped and the boolean variable could be "resetted"
  m_bSniffing = false;

  // next time can be sniffed from interface
  m_bFromFile = false;
}


void Sniffer::startSniffingInterface()
{
  // set all parameters to prepare sniffing structures
  struct bpf_program bpfp;
  char errbuf[PCAP_ERRBUF_SIZE];

#ifdef MUST_DO_SELECT
  int selRet = 0;
  m_pcap_fd = 0;
#endif

  // open libpcap using ifname interface
  if (m_strInterface == "any")
      // promiscue mode not supported by "any" interface
    m_pcapfp = pcap_open_live(m_strInterface.ascii(), PKTMAX, 0, -1, errbuf);
  else
    m_pcapfp = pcap_open_live(m_strInterface.ascii(), PKTMAX, 1, -1, errbuf);

  if (m_pcapfp == NULL)
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    return;
  }
  else
    m_pdump = pcap_dump_open( m_pcapfp, TEMPFILE );

#ifdef MUST_DO_SELECT
  m_pcap_fd = pcap_fileno(m_pcapfp);
#endif

  // start reading packets on the selected source
  while (m_canSniff)
  {
    struct pcap_pkthdr hdr;
    int linkhdr_len;
    struct IpHdr *ip;
    unsigned char *p;

    // get next tail packet calling pcap_next()
#ifdef MUST_DO_SELECT
    /*
     * snipped from ethereal code.
     */
     FD_ZERO(&m_fdset);
     FD_SET(m_pcap_fd, &m_fdset);
     m_fdtimeout.tv_sec = 0;
     m_fdtimeout.tv_usec = CAP_READ_TIMEOUT*1000;
     selRet = select(m_pcap_fd+1, &m_fdset, NULL, NULL, &m_fdtimeout);
     if (selRet > 0) {
       /*
        * "select()" says we can read from it without blocking; go for
        * it.
        */
       p = (unsigned char*) pcap_next(m_pcapfp, &hdr);
     } else {
        p = NULL;
        if (selRet < 0 && errno != EINTR) {

          kdDebug() << "Unexpected error from select:" << 
              strerror(errno) << endl;
          break;
        }
     }
#else
     p = (unsigned char*) pcap_next(m_pcapfp, &hdr);
#endif /* MUST_DO_SELECT */

     if (p == NULL)
       continue;  // we didn't get any packet
     else
       pcap_dump((u_char *)m_pdump, &hdr, p);

     // we got a packet: raw bytes (p), the timestamp and the captured packet length and the real packet length (hdr)
     /*
      *  hdr is a struct pcap_pkthdr:
      *    ts - a struct timeval containing the time when the packet was captured
      *    caplen - a bpf_u_int32 giving the number of bytes of the packet that are available from the capture
      *    len - a bpf_u_int32 giving the length of the packet, in bytes (which might be more than the number
      *      of bytes available from the capture, if the length of the packet is larger than the maximum
      *      number of bytes to capture)
      */

     int frameLength, frameType;
     setHeaderData(m_pcapfp, frameType, frameLength);
     if (frameLength < 0)
     {
       kdDebug() << "unknown packet" << endl;
       continue;
     }

     // now we call PacketManager to store:
     //  1 - the raw data
     //  2 - the timestamp and the captured packet length and the real packet lenght
     //  3 - the frame type string
     //  4 - the frame length: we need this to know how to get the data from the frame
     emit gotPacket(p, hdr, m_frameNumber++, frameType, frameLength);
   }

  kdDebug() << "Thread stopped" << endl;

  // release libpcap resource
  pcap_close(m_pcapfp);

  // now can close dumped file
  pcap_dump_close(m_pdump);

  // now the thread has stopped and the boolean variable could be "resetted"
  m_bSniffing = false;

  // next time can be sniffed from interface
  m_bFromFile = false;
}


// parameters:
//   1 - pcap_t pHandler: packet handler
//   2 - int frameType: frame type (input/output data)
//   3 - int hdrLen: frame header length (input/output data)
// we need the header length to extract the data from the frame
// we got -1 for an unknown Link Layer
void Sniffer::setHeaderData(pcap_t *pHandler, int &frameType, int &hdrLen)
{
  frameType = pcap_datalink(pHandler);

  switch(frameType)
  {
    case DLT_EN10MB:
      hdrLen = 14;
      break;
    case DLT_LINUX_SLL:
      hdrLen = 16;
      break;
    case DLT_LOOP:
      hdrLen = 4;
      break;
    case DLT_NULL:
      hdrLen = 4;
      break;
    case DLT_RAW:
      hdrLen = 0;
      break;
    default:
      hdrLen = -1;
      break;
  }
}

#include "sniffer.moc"
