/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <qcombobox.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kmessagebox.h>

#include "capturedialog.h"
#include "options/ksnifferoptiondialog.h"

CaptureDialog::CaptureDialog(QWidget *parent)
 : KDialogBase( Plain, i18n("New Capture"), Default|Ok|Cancel, Cancel, parent, "newcapture",
                /*modal*/true, /*separator*/false, KGuiItem (i18n("&Start"), "ok") )
{
  KIconLoader* il = KGlobal::iconLoader();

  QVBoxLayout* mainLay = new QVBoxLayout( plainPage() );
  mainLay->activate();

  // the groupbox to select the interface to sniff
  QGroupBox* groupBox = new QGroupBox( plainPage(), "groupBox" );
  groupBox->setTitle( i18n( "Select Interface" ) );
  mainLay->addWidget( groupBox );

  mainLay->addItem( new QSpacerItem( 10, 10, QSizePolicy::Minimum, QSizePolicy::Minimum ) );

  // redefine the "Default" button
  KGuiItem defaultButton( i18n("Advanced Options", "&Advanced..."), "ksniffer" );
  setButtonWhatsThis( Default, i18n( "<qt><p>Opens the Configure Dialog where you can "
                                     "choose the settings about the capture.</p>"
                                     "<p>In case you don't request all the options, the "
                                     "capture will start using the default "
                                     "settings.</p></qt>" ) );
  setButtonTip(Default, i18n( "Show advanced options" ) );
  setButtonGuiItem( Default, defaultButton );

  QVBoxLayout* groupLay = new QVBoxLayout( groupBox, 15, 6 );
  groupLay->addItem( new QSpacerItem( 10, 10, QSizePolicy::Minimum, QSizePolicy::Minimum ) );

  QHBoxLayout* ifaceSelectLay = new QHBoxLayout();
  groupLay->addLayout( ifaceSelectLay );

  QLabel* textLabel = new QLabel( groupBox, "textLabel" );
  textLabel->setText( i18n( "I&nterface:" ) );

  m_interface = new QComboBox( false, groupBox, "m_interface" );
  textLabel->setBuddy( m_interface );
  QWhatsThis::add( m_interface,
                   i18n( "<p>Contains the list of network interfaces "
                         "detected on your system.<br><br>"
                         "The special interface <b>any</b> captures network "
                         "packets from all network interfaces of your "
                         "system.</p>" ) );

  KPushButton* m_redetect = new KPushButton( groupBox, "m_rescan" );
  m_redetect->setText( i18n( "Re&detect Interfaces" ) );
  m_redetect->setIconSet( il->loadIconSet( "reload", KIcon::Small ) );
  QWhatsThis::add( m_redetect, i18n( "Updates the Interface combobox scanning your system again about its network interfaces." ) );

  ifaceSelectLay->addWidget( textLabel );
  ifaceSelectLay->addItem( new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Minimum ) );
  ifaceSelectLay->addWidget( m_interface );
  ifaceSelectLay->addItem( new QSpacerItem( 10, 10, QSizePolicy::Minimum, QSizePolicy::Minimum ) );
  ifaceSelectLay->addWidget( m_redetect );

  QVBoxLayout* vertSpace = new QVBoxLayout();
  groupLay->addLayout( vertSpace );
  vertSpace->addItem( new QSpacerItem( 5, 5, QSizePolicy::Fixed, QSizePolicy::Fixed ) );

  QGridLayout* detailLay = new QGridLayout( 3, 4, 5 );
  groupLay->addLayout( detailLay );

//   QLabel* label = new QLabel( groupBox, "label" );
//   QPixmap icon( DesktopIcon( "hwinfo" ) );
//   label->setPixmap( icon );
//   detailLay->addWidget( label, 1, 0, Qt::AlignHCenter );

  QLabel* label = new QLabel( groupBox, "label" );
  label->setText( i18n( "IP address:" ) );
  detailLay->addWidget( label, 0, 1 );

  m_ipaddress = new QLabel( groupBox, "m_ipaddress" );
  QToolTip::add( m_ipaddress, i18n( "Shows the IP address of the selected interface, if it's available." ) );
  detailLay->addWidget( m_ipaddress, 0, 2 );

  label = new QLabel( groupBox, "label" );
  label->setText( i18n( "Netmask:" ) );
  detailLay->addWidget( label, 1, 1 );

  m_netmask = new QLabel( groupBox, "m_netmask" );
  QToolTip::add( m_netmask, i18n( "Shows the netmask of the selected interface, if it's available." ) );
  detailLay->addWidget( m_netmask, 1, 2 );

  label = new QLabel( groupBox, "label" );
  label->setText( i18n( "MAC address:" ) );
  detailLay->addWidget( label, 2, 1 );

  m_macAddress = new QLabel( groupBox, "m_macAddress" );
  QToolTip::add( m_macAddress, i18n( "Shows the MAC address of the selected interface, if it's available." ) );
  detailLay->addWidget( m_macAddress, 2, 2 );

  groupLay->addItem( new QSpacerItem( 10, 50, QSizePolicy::Minimum, QSizePolicy::Expanding ) );

  setButtonWhatsThis( Ok, i18n( "Start a new network packet-capture session." ) );

  // fill the Interface ComboBox
  m_interface->insertStringList( m_netDevice.interfaceList() );
  setData( m_interface->currentText() );

  resize( QSize( 350, 160 ) );

  connect( m_redetect, SIGNAL( clicked() ), this, SLOT( slotRedetect() ) );
  connect( m_interface, SIGNAL( activated(const QString&) ), this, SLOT( setData(const QString&) ) );

  m_strUsedInterface = QString::null;  // no selection at startup
}


CaptureDialog::~CaptureDialog()
{
  m_interface->clear();
}


void CaptureDialog::slotOk()
{
  m_strUsedInterface = m_interface->currentText();  // set the selected interface just when click Start button

  accept();
}


void CaptureDialog::slotDefault()
{
    KSnifferOptionDialog dialogSettings(this, "SettingsDialog", true);

    // get the user option to start capture
    dialogSettings.exec();
}


void CaptureDialog::slotRedetect()
{
  m_netDevice.detect();
  m_interface->clear();
  m_interface->insertStringList(m_netDevice.interfaceList());
  setData( m_interface->currentText() );
}


void CaptureDialog::setData( const QString & strInterface )
{
  DeviceData data = m_netDevice.interfaceData(strInterface);

  // updating IP address
  QString str = data.getAddress();
  if ( str.isEmpty() )
    str = i18n("not available");
  m_ipaddress->setText( str );

  // updating netmask
  str = data.getNetmask();
  if ( str.isEmpty() )
    str = i18n("not available");
  m_netmask->setText( str );

  // updating MAC address
  str = data.getHWaddress();
  if ( str.isEmpty() || (strInterface == "lo"))
    str = i18n("not available");
  m_macAddress->setText( str );
}

#include "capturedialog.moc"
