/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#ifndef PACKETMANAGER_H
#define PACKETMANAGER_H

#include <pcap.h>

#include <qobject.h>

#include "packet.h"

/**
 * This class manages the set of the sniffed packets
 *
 * @author Giovanni Venturi
 */

class PacketManager : public QObject
{
  Q_OBJECT

public:
  PacketManager( QObject *parent, const char *name );
  ~PacketManager();

  /**
   * Set the frame number reference to get all needed information
   */
  void setFrameNumber( const long& num );
  long getFrameNumber() const;

  void setFilePath( const QString& tmp );
  void setFilename( const QString& tmpFile );
  QString tmpFilePath() const;

  bpf_u_int32 frameCapturedLength();
  bpf_u_int32 frameLength();
  QString strTimeStamp() const;
  QString strAbsoluteTimeStamp() const;
  QString strRelativeTimeStamp() const;
  QString strDiffTimePrevious() const;
  QString strDiffTimeFirst() const;
  QString strDetailTimeStamp() const;

  QString strSourceAddress();
  QString strDestinationAddress();
  QString strProtocol();
  QString strInfo();
  int getFrameHType();
  int getFrameHLength() const;
  ptrPacketType getPacket();

  /**
   * Clean the packet vector.
   */
  void clearList();

  /**
   * Set the policy to store or not packets.
   */
  void acceptPacket( bool acceptCond );

  /**
   * @return true is the current policy is: store packets.
   */
  bool acceptingPacket();

  /**
   * Did we get any packets?
   * @return true if there are any packets into the packet vector.
   */
  bool packetAvailable();

  /**
   * @return number of available packets.
   */
  long numberOfPackets();

protected slots:
  /**
   * Use this slot to save a new packet into the packet vector.
   */
  void savePacket( ptrPacketType p, struct pcap_pkthdr packetHeader, long ord, int frameType, int frameLen );

signals:
  /**
   * A packet was saved
   */
  void savedPacket( long ord, PacketManager *packetManager );

private:
  Packet m_packet, m_lastPacket, m_firstPacket;
  bool m_acceptingPacket;  // true  = captured packet have to be stored
                           // false = no storing captured packets

  long m_frameNumber;

  QString m_strFilePath;
  bool m_bPacketAvailable;

private:
  void setHeaderData(pcap_t *pHandler, int &frameType, int &hdrLen);
};

#endif  // PACKETMANAGER_H
