/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <netinet/in.h>

#include <qstringlist.h>

#include <klocale.h>

#include "icmp-protocol.h"
#include "ip-protocol.h"
#include "../packet.h"
#include "../packetstructures.h"

IcmpProtocol::IcmpProtocol(Packet *packet)
{
  m_packet = packet;
}


u_int8_t IcmpProtocol::type() const
{
  struct IcmpHdr *icmp = (struct IcmpHdr *) m_packet;

  return icmp->type;
}


QString IcmpProtocol::strType() const
{
  /**
   * icmp-codes [RFC 792]
   *
   * Summary of Message Types
   *   0  Echo Reply
   *   3  Destination Unreachable
   *   4  Source Quench
   *   5  Redirect
   *   8  Echo
   *  11  Time Exceeded
   *  12  Parameter Problem
   *  13  Timestamp
   *  14  Timestamp Reply
   *  15  Information Request
   *  16  Information Reply
   */

  QString stringType;
  switch (type())
  {
    case 0:
      stringType = i18n("ICMP type", "Echo Reply");
      break;
    case 3:
      stringType = i18n("ICMP type", "Destination Unreachable");
      break;
    case 4:
      stringType = i18n("ICMP type", "Source Quench");
      break;
    case 5:
      stringType = i18n("ICMP type", "Redirect");
      break;
    case 8:
      stringType = i18n("ICMP type", "Echo");
      break;
    case 11:
      stringType = i18n("ICMP type", "Time Exceeded");
      break;
    case 12:
      stringType = i18n("ICMP type", "Parameter Problem");
      break;
    case 13:
      stringType = i18n("ICMP type", "Timestamp");
      break;
    case 14:
      stringType = i18n("ICMP type", "Timestamp Reply");
      break;
    case 15:
      stringType = i18n("ICMP type", "Information Request");
      break;
    case 16:
      stringType = i18n("ICMP type", "Information Reply");
      break;
    default:
      stringType = i18n("Unknown type for ICMP packet");
  }

  return stringType;
}


u_int8_t IcmpProtocol::code() const
{
  struct IcmpHdr *icmp = (struct IcmpHdr *) m_packet;

  return icmp->code;
}


u_int16_t IcmpProtocol::checksum() const
{
  struct IcmpHdr *icmp = (struct IcmpHdr *) m_packet;

  return ntohs(icmp->checksum);
}


u_int16_t IcmpProtocol::id() const
{
  struct IcmpHdr *icmp = (struct IcmpHdr *) m_packet;

  return icmp->un.echo.id;
}


u_int16_t IcmpProtocol::sequence() const
{
  struct IcmpHdr *icmp = (struct IcmpHdr *) m_packet;

  return icmp->un.echo.sequence;
}


QStringList IcmpProtocol::headerLines() const
{
  QStringList list, ipList;
  IpProtocol ip((Packet *) m_packet, sizeof(struct IcmpHdr));

  list << i18n("ICMP protocol field", "Type: %1 (%2)").arg( type() ).arg( strType() );
  list << i18n("ICMP protocol field", "Code: %1").arg( code() );
  list << i18n("ICMP protocol field", "Checksum: 0x%1").arg( checksum(), 0, 16 );
  if (type() == 3 || type() == 4 || type() == 11 || type() == 12)
  {
    list << "*open";
    list << i18n("IP information:");
    ipList = ip.headerLines();
    for ( QStringList::Iterator it = ipList.begin(); it != ipList.end(); ++it )
    {
      list << *it;
    }
    list << "*close";
  }
  //list << i18n("ICMP protocol field", "Identificator: %1").arg( id() );
  //list << i18n("ICMP protocol field", "Sequence: %1").arg( sequence() );

  return list;
}
