/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <qstring.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kdebug.h>
#include <klocale.h>

#include "protocolnamelist.h"

ProtocolNameList::ProtocolNameList()
{
  m_protocolName = QString::null;
  parseServicesFile();
}


ProtocolNameList::~ProtocolNameList()
{
}


QString ProtocolNameList::detectProtocol( uint16_t const& proto )
{
  QString result;
  int index = 0;

  if (!m_portData.isEmpty())
  {
    // just if I filled the names vector I can look for the related protocol name
    PortName *it;
    it = m_portData.first();
    while ((it->number() < proto) && (m_portData.last() != it) )
    {
      it = m_portData.at(++index);
    }
    if (it->number() == proto)
      result = it->name();
  }

  return result;
}


void ProtocolNameList::parseServicesFile()
{
  QFile file( "/etc/services" );
  PortName *pn;

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream stream( &file );
    QString line;

    while ( !stream.atEnd() )
    {
      line = stream.readLine(); // line of text excluding '\n'
      int j = 0;
      while ( line.at( j ) == ' ' )
        j++;
      if ( (line.at( j ) != '#') && (line.length() > 0))
      {
        QString token, port;
        short tcp_udp = 0;

        while ( (line.at( j ) != ' ') && (line.at( j ).unicode() != 9) )
          // 9 is unicode for TAB character
          token.append( line.at( j++ ) );
        while ( !line.at( j ).isDigit() )
          j++;
        while ( line.at( j ).isDigit() )
          port.append( line.at( j++ ) );
        j++;
        if ( line.mid( j, 3 ) == "tcp" )
          tcp_udp = TCP_NUM;
        else if ( line.mid( j, 3 ) == "udp" )
          tcp_udp = UDP_NUM;
        if ( line.mid( j, 3 ) == "ddp" )
          tcp_udp = DDP_NUM;
        if ( !m_portData.isEmpty() )
        {
          // the second time and all the other times the protocol could be in the protocols list
          if ( m_portData.at(m_portData.count() - 1 )->number() == port.toInt() )
          {
            // the item is present so you have to add the second protocol (UDP or TCP)
            m_portData.last()->addTcpUdpState( tcp_udp );
          }
          else
          {
            // the protocol was not in the list
            pn = new PortName ( token /* service name, for example FTP, SMTP, ...*/,
              port.toInt(), tcp_udp);
            m_portData.append(pn);
          }
        }
        else
        {
          // first time the list is empty, no protocol in it
          pn = new PortName ( token /* service name, for example FTP, SMTP, ...*/,
            port.toInt(), tcp_udp);
          m_portData.append(pn);
        }
      }
    }
    file.close();
  }
  if (m_portData.isEmpty())
    kdDebug() << "Empty list" << endl;
  for ( uint i = 0; i < m_portData.count(); ++i )
    if ( m_portData.at(i) )
    {
      if (isTCP(m_portData.at( i )->number()))
        kdDebug() << "port: " << m_portData.at( i )->number() << " name: " << m_portData.at( i )->name() << " is TCP: " << m_portData.at( i )->tcpUdpState() << endl;

      if (isUDP(m_portData.at( i )->number()))
        kdDebug() << "port: " << m_portData.at( i )->number() << " name: " << m_portData.at( i )->name() << " is UDP: " << m_portData.at( i )->tcpUdpState() << endl;

      if (isDDP(m_portData.at( i )->number()))
        kdDebug() << "port: " << m_portData.at( i )->number() << " name: " << m_portData.at( i )->name() << " is DDP: " << m_portData.at( i )->tcpUdpState() << endl;
    }
}


bool ProtocolNameList::isTCP( uint16_t const& proto )
{
  PortName *index = m_portData.first();
  int i = 0;

  while ( (index != m_portData.last()) && (index->number() != proto) )
    index = m_portData.at(++i);

  if (index->number() != proto)
    // port not found
    return false;
  else
    return ( (index->tcpUdpState() & TCP_NUM) == TCP_NUM );
}


bool ProtocolNameList::isUDP( uint16_t const& proto )
{
  PortName *index = m_portData.first();
  int i = 0;

  while ( (index != m_portData.last()) && (index->number() != proto) )
    index = m_portData.at(++i);

  if (index->number() != proto)
    // port not found
    return false;
  else
    return ( (index->tcpUdpState() & UDP_NUM) == UDP_NUM );
}


bool ProtocolNameList::isDDP( uint16_t const& proto )
{
  PortName *index = m_portData.first();
  int i = 0;

  while ( (index != m_portData.last()) && (index->number() != proto) )
    index = m_portData.at(++i);

  if (index->number() != proto)
    // port not found
    return false;
  else
    return ( (index->tcpUdpState() & DDP_NUM) == DDP_NUM );
}
