/***************************************************************************
 *   Copyright (C) 2005 by Raul Fernandes                                  *
 *   rgfbr@yahoo.com.br                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "dictzip.h"

#include <qfile.h>
#include <qcstring.h>

#include <kdebug.h>

#include <zlib.h>

#define CHUNK 0xffffL

DictZip::DictZip( const QString &arq )
{
  // Index file
  idxFile = new QFile( arq );
  // Definition file (.dict)
  // it should be on the same directory that index file
  QString defarq = arq;
  defarq.replace( ".index", ".dict" );
  // Checks if the definition file is compressed or not
  if( QFile::exists( defarq ) )
  {
    file = new QFile( defarq );
    isCompressed = false;
  }else{
    // The definition file is compressed (.dict.dz)
    // Read the header
    file = new QFile( defarq + ".dz" );
    isCompressed = true;
    file->open( IO_ReadOnly );
    ID1 = file->getch(); //ID1 = 31 (0x1f, \037)
    ID2 = file->getch(); //ID2 = 139 (0x8b, \213)
    COMPRESSION = file->getch();
    FLAGS = file->getch();
    FTEXT = FLAGS & 1;
    FHCRC = FLAGS & 2;
    m_extraField = FLAGS & 4;
    m_hasName = FLAGS & 8;
    FCOMMENT = FLAGS & 16;
    m_mtime = (unsigned char)file->getch() + (unsigned char)file->getch() * 256 + (unsigned char)file->getch() * 256 * 256 + (unsigned char)file->getch() * 256 * 256 * 256;
    XFL = file->getch(); // Compression type
    OS = file->getch(); // Operating System
    // If has extra field, read it
    if( m_extraField )
    {
      XLEN = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
      readExtraField();
    }
    // If has name of decompressed file, read it
    if( m_hasName )
    {
      readFileName();
    }
    // If has a comment field, read it
    if( FCOMMENT )
    {
      readFileName();
    }
    // If has a CRC field, read it
    if( FHCRC )
    {
      *crc16[0] = file->getch();
      *crc16[1] = file->getch();
    }
    // Get the current position
    // This is start position of the chunks of compressed data
    offset = file->at();
    file->close();
  }

#ifndef NOPTIMIZE
  QString line, headword;
  struct entry entry;
  dic.clear();
  idxFile->open( IO_ReadOnly );
  while( !idxFile->atEnd() )
  {
    idxFile->readLine( line, 1024 );

    // Catch the position of definition in definition file
    headword = line.section( '\t', 1, 1);
    // The value is coded in base64
    entry.position = b64_decode( headword.local8Bit() );

    // Catch the size of definition in definition file
    headword = line.section( '\t', 2, 2 ).remove('\n');
    // The value is coded in base64
    entry.size = b64_decode( headword.local8Bit() );

    headword = QString::fromUtf8( line.section( '\t', 0, 0 ) );
    dic.insert( headword, entry );
  }
  idxFile->close();
#endif
}


DictZip::~DictZip()
{
  delete file;
  delete idxFile;
}




/*!
    \fn DictZip::readExtraField()
 */
void DictZip::readExtraField()
{
  //kdDebug() << "readExtraField()" << endl;
  offsets.clear();
  SI1 = file->getch();
  SI2 = file->getch();
  // length of the subfield data
  LEN = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  int size = (int)LEN - 6;
  // Version
  VER = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  // length of a "chunk" of data
  CHLEN = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  // how many chunks are preset
  CHCNT = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
  unsigned long data;
  for(int a = 0; a < size; a++)
  {
    // how long each chunk is after compression
    data = (unsigned char)file->getch() + (unsigned char)file->getch() * 256;
    a++;
    offsets.append( data );
  }
}


/*!
    \fn DictZip::readFileName()
 */
void DictZip::readFileName()
{
  //kdDebug() << "readFileName()" << endl;
  QString filename;
  char byte;
  byte = file->getch();
  while( byte != '\0' )
  {
    filename += byte;
    byte = file->getch();
  }
  m_filename = filename;
}


/*!
    \fn DictZip::readComment()
 */
void DictZip::readComment()
{
  kdDebug() << "readComment()" << endl;
  QString comment;
  char byte;
  byte = file->getch();
  while( byte != '\0' )
  {
    comment += byte;
    byte = file->getch();
  }
  m_comment = comment;
}


/*!
    \fn DictZip::search( const QString &word )
 */
QString DictZip::search( const QString &word )
{
  //kdDebug() << "DictZip::search()" << endl;
  struct entry entry;

#ifndef NOPTIMIZE
  // Find the headword in index file
  entry = dic[word];
  if( entry.size == 0 ) return QString::null;
#else
  QString line, headword;
  bool found = false;
  idxFile->open( IO_ReadOnly );

  // Find the headword in index file
  while( !idxFile->atEnd() )
  {
    idxFile->readLine( line, 1024 );
    headword = QString::fromUtf8( line.section( '\t', 0, 0 ) );

    if( headword != word ) continue;
    found = true;

    // Catch the position of definition in definition file
    headword = line.section( '\t', 1, 1);
    // The value is coded in base64
    entry.position = b64_decode( headword.local8Bit() );

    // Catch the size of definition in definition file
    headword = line.section( '\t', 2, 2 ).remove('\n');
    // The value is coded in base64
    entry.size = b64_decode( headword.local8Bit() );

    // Word found. Break the loop.
    break;
  }
  idxFile->close();

  // If not found, return a null string
  if( !found ) return QString::null;
#endif

  // Check if the definition file is compressed
  if( isCompressed )
  {
    ulong a = 0;

    // Calculate how many chunks we have to skip
    uint chunk = entry.position / CHLEN ;
    // Calculate the position of definition in chunk
    uint pos = entry.position % CHLEN;

    // Size of the chunk we are looking for
    unsigned long chunkLen = offsets[chunk];
    // If the word is in the end of chunk, we have to decompress two chunks
    if( (pos + entry.size) > CHLEN ) chunkLen += offsets[chunk + 1];

    // How many bytes we have to skip
    unsigned long skip = 0;
    for(uint a=0;a<chunk;a++) skip += offsets[a];

    // Stores the compressed data
    QByteArray data( chunkLen + 1 );
    data[chunkLen] = '\0';

    // Stores the decompressed data
    QCString result;

    // Definition file
    file->open( IO_ReadOnly );

    // Jump to chunk we are looking for
    file->at( offset + skip );

    // Get the compressed data
    for(a = 0; a < chunkLen; a++) data[a] = file->getch();
    data[a] = '\0';
    file->close();

    // Decompress the data
    result = deflate( data );
    // Returns only the definition
    return QString::fromUtf8( result.mid( pos, entry.size ) );
  }else{
    // The file is not compressed
    file->open( IO_ReadOnly );
    // Jump to position of definition
    file->at( entry.position );
    // Get the definition
    QCString result( entry.size + 1 );
    for( uint a = 0; a < entry.size; a++ ) result[a] = file->getch();
    result[entry.size] = '\0';
    file->close();
    // Return the result
    return QString::fromUtf8( result.data() );
  }
}


/*!
    \fn DictZip::b64_decode( const char *val )
 */
unsigned long DictZip::b64_decode( const char *val )
{
   int b64_index[256] = {
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,62, 99,99,99,63,
    52,53,54,55, 56,57,58,59, 60,61,99,99, 99,99,99,99,
    99, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,99, 99,99,99,99,
    99,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99,
    99,99,99,99, 99,99,99,99, 99,99,99,99, 99,99,99,99, };
   unsigned long v = 0;
   int           i;
   int           offset = 0;
   int           len = strlen( val );

   for (i = len - 1; i >= 0; i--) {
      int tmp = b64_index[ (unsigned char)val[i] ];
      v |= tmp << offset;
      offset += 6;
   }

   return v;
}


QCString DictZip::deflate( const QByteArray &data )
{
    //kdDebug()<< "Deflate()" << endl;
    int ret;
    z_stream strm;
    char out[CHUNK];
    QCString result( 65536 );

    // Inicialization of zlib
    strm.zalloc = Z_NULL;
    strm.zfree = Z_NULL;
    strm.opaque = Z_NULL;
    strm.avail_in = 0;
    strm.next_in = Z_NULL;
    ret = inflateInit2( &strm, -MAX_WBITS );
    if (ret != Z_OK)
      return "";

      // Compressed data
      strm.avail_in = data.size();
      strm.next_in = (Bytef*)data.data();

      /* run inflate() on input until output buffer not full */
      do {
        strm.avail_out = CHUNK;
        strm.next_out = (Bytef*)out;
        ret = inflate(&strm, Z_SYNC_FLUSH);
        switch (ret) {
          case Z_NEED_DICT:
            ret = Z_DATA_ERROR;     /* and fall through */
          case Z_DATA_ERROR:
          case Z_MEM_ERROR:
            (void)inflateEnd(&strm);
            return ""; // Error
        }
        result += out;
      } while (strm.avail_out == 0);

    /* clean up and return */
    ret = inflateEnd(&strm);
    return result;
}
