using System;
using System.Collections;
using Gtk;
using GtkExtra;
using Glade;

namespace Kurush
{

public class DetailsFrame : Frame
{
    public override void UpdateActions (Actions actions)
    {
        actions.EditAdd.Enabled = true;
        actions.EditProperties.Enabled = true;
        actions.FilePrint.Enabled = true;

        transferForm.Visible = actions.TransferFormActive;
        ViewSelectionChanged (null, null);
    }

    [Glade.Widget] private ScrolledWindow detailsViewContainer;
    [Glade.Widget] private Widget transferForm;
    [Glade.Widget] private Table transferTable;
    [Glade.Widget] private Entry descriptionEntry;
    [Glade.Widget] private Label debitLabel;
    [Glade.Widget] private Label creditLabel;
    [Glade.Widget] private Label balanceLabel;
    [Glade.Widget] private RadioButton increaseButton;
    [Glade.Widget] private RadioButton decreaseButton;
    [Glade.Widget] private HBox dateHBox;
    [Glade.Widget] private Button todayButton;


    private ManagedEntry amountEntryM;
    private ManagedEntry dateEntryM;
    private WidgetEntry accountEntry;
    private WidgetEntry dateEntry;

    private Data.Account account;
    private DetailsView view;
    private NodeStore store;
    private EventHandler AccountEntryChangedHandler;
    private EventHandler DateEntryChangedHandler;

    private int maxPageSize = -200;
    private int growSize = 10;

    public DetailsFrame (FrameManager owner, Data.Account account)
            : base (owner, "detailsFrame")
    {
        this.account = account;
        this.key = account;

        owner.Actions.TransferFormChanged += new TransferFormHandler (TransferFormChanged);

        amountEntryM = new ManagedEntry (gxml["amountEntry"] as Entry, 0m);

        dateEntry = new WidgetEntry (Owner.DatePopup);
        dateEntry.Width = 12;
        dateEntryM = new ManagedEntry (dateEntry.Entry, DateTime.Today);
        DateEntryChangedHandler = new EventHandler (DateEntryChanged);
        dateEntry.PopUp += new EventHandler (DateEntryPopUp);
        dateEntry.PopDown += new EventHandler (DateEntryPopDown);
        dateEntry.SizeAllocated += new SizeAllocatedHandler (DateEntrySizeAllocated);
        dateEntry.Show ();

        dateHBox.PackStart (dateEntry, true, true, 0);

        accountEntry = new WidgetEntry (Owner.AccountsPopup);
        accountEntry.AutoSize = true;
        accountEntry.Entry.Editable = false;
        AccountEntryChangedHandler = new EventHandler (AccountEntryChanged);
        accountEntry.PopUp += new EventHandler (AccountEntryPopUp);
        accountEntry.PopDown += new EventHandler (AccountEntryPopDown);
        accountEntry.Show ();
        transferTable.Attach (accountEntry, 1, 4, 1, 2,
                              AttachOptions.Expand | AttachOptions.Fill,
                              AttachOptions.Shrink,
                              0,0);

        Title = "Details for " + account.Name;

        store = new NodeStore (typeof (DetailsNode));

        int c = account.Entries.Count;
        int pageSize;

        if (maxPageSize < 0 || maxPageSize > c)
            pageSize = c;
        else
            pageSize = maxPageSize;

        IEnumerator e = account.Entries.GetEnumerator (c - pageSize, pageSize);
        while (e.MoveNext ())
        {
            Data.Entry entry = e.Current as Data.Entry;
            AddNode (entry);
        }

        view = new DetailsView (store);
        view.Show ();
        detailsViewContainer.Add (view);
        view.Selection.Changed += new EventHandler (ViewSelectionChanged);
        view.Vadjustment.ValueChanged += new EventHandler (ViewValueChanged);
        view.DoubleClicked += new EventHandler (ViewDoubleClicked);

        account.EntryAdded += new Data.EntryEvent (AccountEntryAdded);
        account.EntryRemoved += new Data.EntryEvent (AccountEntryRemoved);

        if (c > 0)
        {
            TreePath path = ScrollLast ();
            view.Selection.SelectPath (path);
        }

        UpdateBalance ();
    }

    private TreePath ScrollLast ()
    {
        TreePath path = new TreePath ((account.Entries.Count - 1).ToString ());
        view.ScrollToCell (path, null, false, 0F, 0F);
        return path;
    }

    private void AddNode (Data.Entry entry)
    {
        DetailsNode node = new DetailsNode (Owner.DataBase, account, entry);
        store.AddNode (node);
        entry.Node = node;
    }

    private void AccountEntryAdded (Data.Entry entry)
    {
        AddNode (entry);
        UpdateBalance ();
    }

    private void ViewValueChanged (object obj, EventArgs args)
    {
        if (view.Vadjustment.Value == view.Vadjustment.Lower)
        {
            //store.InsertNode (0, new DetailsNode (Owner.DataBase, account, account.Entries[0] as Data.Entry));
            Console.WriteLine ("Need Data");
        }
    }

    private void DateEntrySizeAllocated (object o, SizeAllocatedArgs args)
    {
        todayButton.WidthRequest = args.Allocation.Height;
        todayButton.HeightRequest = args.Allocation.Height;
    }

    private void AccountEntryRemoved (Data.Entry entry)
    {
        if (entry.Node != null)
        {
            store.RemoveNode (entry.Node as TreeNode);
            UpdateBalance ();
        }
    }

    private void UpdateBalance ()
    {
        debitLabel.Text = Format.Value (account.Debit);
        creditLabel.Text = Format.Value (account.Credit);
        balanceLabel.Text = Format.Value (account.Balance);
    }

    private void DateEntryChanged (object obj, EventArgs args)
    {
        dateEntryM.Value = Owner.DateView.GetDate ();
        dateEntry.Active = false;
    }

    private void DateEntryPopUp (object obj, EventArgs args)
    {
        DateTime date = (DateTime) dateEntryM.Value;
        Owner.DateView.SelectMonth ( (uint) date.Month - 1, (uint) date.Year);
        Owner.DateView.SelectDay ( (uint) date.Day);

        Owner.DateView.DaySelectedDoubleClick += DateEntryChangedHandler;
    }

    private void DateEntryPopDown (object obj, EventArgs args)
    {
        Owner.DateView.DaySelectedDoubleClick -= DateEntryChangedHandler;
    }

    private void AccountEntryChanged (object obj, EventArgs args)
    {
        SummaryNode node = Owner.AccountsView.GetSelected () as SummaryNode;

        if (node != null && node.Account != null)
        {
            accountEntry.Content = node.Account;
            accountEntry.Entry.Text = node.Account.Name;
            accountEntry.Active = false;
        }
    }

    private void AccountEntryPopUp (object obj, EventArgs args)
    {
        Owner.AccountsView.Selection.Changed += AccountEntryChangedHandler;
        accountEntry.Content = null;
    }

    private void AccountEntryPopDown (object obj, EventArgs args)
    {
        Owner.AccountsView.Selection.Changed -= AccountEntryChangedHandler;
    }

    private DetailsNode selectedNode = null;

    private void ViewSelectionChanged (object obj, EventArgs args)
    {
        if (!Content.Visible)
            return;

        DetailsNode node = view.GetSelected () as DetailsNode;

        if (node != null)
        {
            Owner.Actions.EditDetails.Enabled = true;
            Owner.Actions.EditRemove.Enabled = true;

            amountEntryM.Value = node.Entry.Amount;
            accountEntry.Entry.Text = node.ToAccount.Name;
            accountEntry.Content = node.ToAccount;

            dateEntryM.Value = node.Entry.Date;
            descriptionEntry.Text = node.Entry.Description;

            Data.TransferType type =
                Owner.DataBase.GetTransferType (account, node.Entry);

            increaseButton.Active = type == Data.TransferType.Increase;
            decreaseButton.Active = type == Data.TransferType.Decrease;
        }
        else
        {
            Owner.Actions.EditDetails.Enabled = false;
            Owner.Actions.EditRemove.Enabled = false;

            Clear ();
        }

        selectedNode = node;
    }

    private void TransferFormChanged (bool active, bool sensitive)
    {
        transferForm.Visible = active;
        transferForm.Sensitive = sensitive;
    }

    public override void Add ()
    {
        view.Selection.UnselectAll ();
        transferForm.Visible = true;
        if (amountEntryM != null)
            amountEntryM.HasFocus = true;
    }

    private void ViewDoubleClicked  (object obj, EventArgs args)
    {
        Details ();
    }

    public override void Details ()
    {
        if (selectedNode != null)
        {
            if (!Owner.Activate ( typeof(DetailsFrame),
                                  selectedNode.ToAccount) )
            {
                DetailsFrame detailsFrame =
                    new DetailsFrame (Owner, selectedNode.ToAccount);

                Owner.Go (detailsFrame);
            }
        }
    }

    public override void Properties ()
    {
        if (!Owner.Activate ( typeof(AccountPropertiesFrame),
                              account) )
        {
            AccountPropertiesFrame accountProperties =
                new AccountPropertiesFrame (Owner, account);

            Owner.Go (accountProperties);
        }
    }

    public override void Remove ()
    {
        if (selectedNode != null)
            Owner.DataBase.RemoveTransfer (account, selectedNode.Entry);
    }

    public void Clear ()
    {
        dateEntryM.Value = DateTime.Today;
        accountEntry.Text = "";
        accountEntry.Content = null;
        amountEntryM.Value = 0m;
        descriptionEntry.Text = "";
    }

    private void on_transferButton_clicked (object obj, EventArgs args)
    {

        Data.Account toAccount = accountEntry.Content as Data.Account;
        decimal amount = (decimal) amountEntryM.Value;

        if (toAccount != null && toAccount != account && amount > 0m)
        {
            Data.TransferType type;

            if (increaseButton.Active)
                type = Data.TransferType.Increase;
            else
                type = Data.TransferType.Decrease;

            DateTime date = (DateTime) dateEntryM.Value;

            string description = descriptionEntry.Text;

            Owner.DataBase.Transfer (date, type,
                                     account, toAccount, amount, description);

            if (selectedNode == null)
                Clear ();
            else
                view.Selection.UnselectAll ();

            ScrollLast ();
        }

    }

    private void on_applyButton_clicked (object obj, EventArgs args)
    {
        decimal amount = (decimal) amountEntryM.Value;
        Data.Account toAccount = accountEntry.Content as Data.Account;

        if (selectedNode != null && amount > 0m && toAccount != account)
        {
            DateTime date = (DateTime) dateEntryM.Value;
            string description = descriptionEntry.Text;

            Data.TransferType type;

            if (increaseButton.Active)
                type = Data.TransferType.Increase;
            else
                type = Data.TransferType.Decrease;

            Owner.DataBase.ChangeTransfer (account, selectedNode.Entry, type, toAccount,
                                           date, amount, description);

            if (toAccount != selectedNode.ToAccount)
                selectedNode.ToAccount = toAccount;

            selectedNode.Update ();
            UpdateBalance ();
        }
    }

    private void on_clearButton_clicked (object obj, EventArgs args)
    {
        if (selectedNode == null)
            Clear ();
        else
            view.Selection.UnselectAll ();
    }

    private void on_todayButton_clicked (object obj, EventArgs args)
    {
        dateEntryM.Value = DateTime.Today;
        dateEntryM.HasFocus = true;
    }

    public override void PrintReport ()
    {
        DateTime endDate = DateTime.Today;
        Data.Entry entry = account.LastEntry;
        if (entry != null)
        {
            if (entry.Date > endDate)
                endDate = entry.Date;
        }

        new DetailsReport (Owner.DataBase, account,
                           Owner.DataBase.ReportStart, endDate);
    }
}

}
