using System;
using System.Collections;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using System.Xml.Schema;

namespace Data
{

public enum TransferType
{
    Increase,
    Decrease
}

public class Base : ICloneable
{
    private bool modified = false;
    private Kurush Content;
    private string filename = null;

    private int Version = 1;
    private EntryComparer entryComparer = new EntryComparer ();
    private AccountComparer accountComparer = new AccountComparer ();
    private AccountSortComparer accountSortComparer = new AccountSortComparer ();

    private Group[] groups = new Group[5];

    public object Clone ()
    {
        Base obj = new Base ();
        obj.filename = (string) filename.Clone ();
        obj.Content = (Kurush) Content.Clone ();

        return obj;
    }

    private void Init ()
    {
        for (int i = 0; i < 5; i++)
            Groups [i] = new Group ((AccountType) i);
    }

    public Base ()
    {
        Content = new Kurush ();
        Content.Version = Version;
        Init ();
    }

    public Base (string filename)
    {
        XmlTextReader reader = null;

        try
        {
            XmlSerializer ser = new XmlSerializer(typeof(Kurush));
            reader = new XmlTextReader(filename);
            reader.XmlResolver = null;

            Content = (Kurush) ser.Deserialize(reader);

            Init ();
            UpdateGroups ();
            this.filename = filename;
        }
        finally
        {
            if (reader != null)
                reader.Close ();
        }
    }

    /***************************************************************************
     API   
     **************************************************************************/


    public bool Modified
    {
        get
        {
            return this.modified;
        }
        set
        {
            bool fire = this.modified != value;

            this.modified = value;

            if (fire)
                if (ModifiedChanged != null)
                    ModifiedChanged (this, null);
        }
    }

    public event EventHandler ModifiedChanged;
    public event AccountEvent AccountAdded;
    public event AccountEvent AccountRemoved;

    public string Filename
    {
        get
        {
            return filename;
        }
        set
        {
            filename = value;
        }
    }

    public DateTime ReportStart
    {
        get
        {
            return Content.ReportStart;
        }
        set
        {
            Content.ReportStart = value;
        }
    }

    public string Title
    {
        get
        {
            return Content.Title;
        }
        set
        {
            Content.Title = value;
        }
    }

    public ArrayList Accounts
    {
        get
        {
            return Content.Accounts;
        }
    }

    public Group[] Groups
    {
        get
        {
            return groups;
        }
    }

    public decimal Networth
    {
        get
        {
            return groups [(int) AccountType.Asset].Balance -
                   groups [(int) AccountType.Liability].Balance;
        }
    }

    public void CalcDatedBalances (DateTime endDate)
    {
        for (int i = 0; i < 5; i++)
        {
            Groups [i].DatedDebit = 0;
            Groups [i].DatedCredit = 0;
        }

        foreach (Account account in Content.Accounts)
        {
            account.DatedDebit = 0;
            account.DatedCredit = 0;

            Entry endEntry = GetLastEntry (account, endDate);
            if (endEntry != null)
            {
                account.DatedDebit = endEntry.DebitBalance;
                account.DatedCredit = endEntry.CreditBalance;
                groups[(int) account.Type].DatedDebit += account.DatedDebit;
                groups[(int) account.Type].DatedCredit += account.DatedCredit;
            }
        }
    }

    public void CalcDatedBalances (DateTime begDate, DateTime endDate)
    {
        for (int i = 0; i < 5; i++)
        {
            Groups [i].DatedDebit = 0;
            Groups [i].DatedCredit = 0;
        }

        foreach (Account account in Content.Accounts)
        {
            account.DatedDebit = 0;
            account.DatedCredit = 0;

            decimal begDebit = 0;
            decimal begCredit = 0;

            Entry begEntry = GetLastEntry (account, begDate.AddDays (-1));
            if (begEntry != null)
            {
                begDebit = begEntry.DebitBalance;
                begCredit = begEntry.CreditBalance;
            }

            Entry endEntry = GetLastEntry (account, endDate);
            if (endEntry != null)
            {
                account.DatedDebit = endEntry.DebitBalance - begDebit;
                account.DatedCredit = endEntry.CreditBalance - begCredit;
                groups[(int) account.Type].DatedDebit += account.DatedDebit;
                groups[(int) account.Type].DatedCredit += account.DatedCredit;
            }
        }
    }

    // Gets the first entry whose date is >= date
    // Otherwise returns null

    public Entry GetFirstEntry (Account account, DateTime date)
    {
        int c = account.Entries.Count;
        if (c == 0)
            return null;

        Entry entry = new Entry ();
        entry.Date = date;
        entry.No = int.MinValue;
        int pos = ~IndexOf (account, entry);

        if (pos < c)
        {
            entry = account.Entries [pos] as Entry;
            entry.Pos = pos;
            return entry;
        }
        else
            return null;
    }

    // Gets the last entry whose date is >= date
    // Otherwise returns null

    public Entry GetLastEntry (Account account, DateTime date)
    {
        int c = account.Entries.Count;
        if (c == 0)
            return null;

        Entry entry = new Entry ();
        entry.Date = date;
        entry.No = int.MaxValue;
        int pos = ~IndexOf (account, entry);

        if (pos > 0)
        {
            entry = account.Entries [pos - 1] as Entry;
            entry.Pos = pos - 1;
            return entry;
        }
        else
            return null;
    }

    public decimal EntryBalance (Account account, Entry entry)
    {
        if (account.Type == Data.AccountType.Asset ||
                account.Type == Data.AccountType.Expense)
            return entry.DebitBalance - entry.CreditBalance;
        else
            return entry.CreditBalance - entry.DebitBalance;

    }

    public void AddAccount (Account account)
    {
        account.No = Content.AccountCounter;
        account.SortOrder = Content.AccountCounter;
        Content.AccountCounter ++;

        int pos = Content.Accounts.BinarySearch (account, accountComparer);

        Content.Accounts.Insert (~pos, account);

        if (account.ParentAccount != null)
            account.ParentAccount.Children.Add (account);
        else
            groups[(int) account.Type].Children.Add (account);

        Modified = true;

        if (AccountAdded != null)
            AccountAdded (account);
    }

    public Account LookupAccount (int no)
    {
        Account account = new Account ();
        account.No = no;

        int pos = Content.Accounts.BinarySearch (account, accountComparer);

        //account.Finalize ();

        if (pos >= 0)
            return Content.Accounts [pos] as Account;
        else
            return null;
    }

    public bool RemoveAccount (Account account)
    {
        bool success = true;

        if (account.Entries.Count == 0 && account.Children.Count == 0)
        {
            try
            {
                Content.Accounts.Remove (account);
                groups[(int) account.Type].Children.Remove (account);
                Modified = true;

                if (AccountRemoved != null)
                    AccountRemoved (account);
            }
            catch (Exception e)
            {
                success = false;
            }
        }
        else
            success = false;

        return success;
    }

    public void Transfer (DateTime date,
                          TransferType transferType,
                          Account fromAccount,
                          Account toAccount,
                          decimal amount,
                          string description)
    {
        Entry fromEntry = new Entry (date);
        Entry toEntry = new Entry (date);

        fromEntry.No = Content.EntryCounter;
        toEntry.No = Content.EntryCounter;
        Content.EntryCounter++;

        fromEntry.Description = description;
        toEntry.Description = description;

        fromEntry.AccountRef = toAccount.No;
        toEntry.AccountRef = fromAccount.No;

        if ( fromAccount.Type == AccountType.Asset ||
                fromAccount.Type == AccountType.Expense )
        {
            if (transferType == TransferType.Increase)
            {
                fromEntry.Debit = amount;
                toEntry.Credit = amount;
            }
            else
            {
                fromEntry.Credit = amount;
                toEntry.Debit = amount;
            }
        }
        else
        {
            if (transferType == TransferType.Increase)
            {
                fromEntry.Credit = amount;
                toEntry.Debit = amount;
            }
            else
            {
                fromEntry.Debit = amount;
                toEntry.Credit = amount;
            }
        }

        AddEntry (fromAccount, fromEntry);
        AddEntry (toAccount, toEntry);

        Modified = true;
    }

    public Entry LookupEntry (Account account, Entry entry)
    {
        Entry toEntry = null;

        int pos = IndexOf (account, entry);

        if (pos >= 0)
            toEntry = account.Entries [pos] as Entry;

        return toEntry;
    }

    public void RemoveTransfer (Account account, Entry entry)
    {
        Account toAccount = LookupAccount (entry.AccountRef);
        Entry toEntry = LookupEntry (toAccount, entry);

        RemoveEntry (account, entry);
        RemoveEntry (toAccount, toEntry);

        Modified = true;
    }

    public TransferType GetTransferType (Account account, Entry entry)
    {
        TransferType type;

        if ( account.Type == AccountType.Asset ||
                account.Type == AccountType.Expense )
        {
            if (entry.Debit != 0m)
                type = TransferType.Increase;
            else
                type = TransferType.Decrease;
        }
        else
        {
            if (entry.Debit != 0m)
                type = TransferType.Decrease;
            else
                type = TransferType.Increase;
        }

        return type;
    }

    public void ChangeTransfer (Account account, Entry entry,
                                TransferType newType, Account newAccount,
                                DateTime date, decimal amount,
                                string description)
    {
        Account toAccount = LookupAccount (entry.AccountRef);
        Entry toEntry = LookupEntry (toAccount, entry);
        TransferType type = GetTransferType (account, entry);

        bool recalculate = false;

        if (entry.Description != description)
        {
            entry.Description = description;
            toEntry.Description = description;
            Modified = true;
        }

        if (toAccount != newAccount)
        {
            RemoveEntry (toAccount, toEntry);
            toEntry = new Entry (entry.Date);
            toEntry.No = entry.No;
            toEntry.Debit = entry.Credit;
            toEntry.Credit = entry.Debit;
            toEntry.AccountRef = account.No;
            toEntry.Description = entry.Description;
            AddEntry (newAccount, toEntry);

            toAccount = newAccount;
            entry.AccountRef = toAccount.No;
            Modified = true;
        }

        if (type != newType)
        {
            entry.ToggleType ();
            toEntry.ToggleType ();
            recalculate = true;
            Modified = true;
        }

        if (amount != entry.Amount)
        {
            entry.Amount = amount;
            toEntry.Amount = amount;
            recalculate = true;
            Modified = true;
        }

        if (entry.Date != date)
        {
            UpdateDate (account, entry, date);
            UpdateDate (toAccount, toEntry, date);

            recalculate = false;
            Modified = true;
        }

        if (recalculate)
        {
            Recalculate (account, entry);
            Recalculate (toAccount, toEntry);
        }

    }

    public void Save (string tempFilename)
    {
        XmlTextWriter writer = null;

        try
        {
            XmlSerializer ser = new XmlSerializer (typeof(Kurush));

            writer = new XmlTextWriter (tempFilename,
                                        System.Text.Encoding.UTF8);

            writer.Formatting = Formatting.Indented;

            ser.Serialize (writer, Content);
        }
        finally
        {
            if (writer != null)
                writer.Close ();
        }
    }


    /***************************************************************************
     PRIVATE   
     **************************************************************************/

    private void UpdateGroups ()
    {
        ArrayList sortedAccounts = (ArrayList) Content.Accounts.Clone ();
        sortedAccounts.Sort (accountSortComparer);

        int i = 0;
        foreach (Account account in sortedAccounts)
        {
            account.SortOrder = i;
            i++;

            if (account.ParentRef < 0)
                groups[(int) account.Type].Children.Add (account);
            else
            {
                account.parentAccount = LookupAccount (account.ParentRef);
                account.parentAccount.Children.Add (account);
            }
        }
    }

    private void Recalculate (Account account, int no)
    {
        account.Recalculate (no);
    }

    private void Recalculate (Account account, Entry entry)
    {
        Recalculate (account, IndexOf (account, entry));
    }

    private int IndexOf (Account account, Entry entry)
    {
        return account.Entries.BinarySearch (entry, entryComparer);
    }

    private int AddEntry (Account account, Entry entry)
    {
        int pos = IndexOf (account, entry);

        account.Entries.Insert (~pos, entry);
        Recalculate (account, ~pos);
        account.DoEntryAdded (entry);

        return ~pos;
    }

    private int RemoveEntry (Account account, Entry entry)
    {
        entry.Debit = 0;
        entry.Credit = 0;
        entry.DebitBalance = 0;
        entry.CreditBalance = 0;

        int entryPos = IndexOf (account, entry);
        Recalculate (account, entryPos);

        account.Entries.RemoveAt (entryPos);
        account.DoEntryRemoved (entry);
        return ~entryPos;
    }

    private void UpdateDate (Account account, Entry entry, DateTime date)
    {
        int oldPos = IndexOf (account, entry);
        account.Entries.RemoveAt (oldPos);

        entry.Date = date;

        int newPos = ~IndexOf (account, entry);
        account.Entries.Insert (newPos, entry);

        int calcPos;

        if (newPos < oldPos)
            calcPos = newPos;
        else
            calcPos = oldPos;

        Recalculate (account, calcPos);
    }


}


}
