using System;
using Gtk;
using Glade;
using GtkExtra;

namespace Kurush
{

public class MainWindow : TopLevel
{
    [Glade.Widget] private Window mainWindow;
    [Glade.Widget] private Menu goMenu_menu;
    [Glade.Widget] private VBox mainVBox;
    [Glade.Widget] private Statusbar mainStatusbar;
    [Glade.Widget] private HandleBox mainHandleBox;
    [Glade.Widget] private CheckMenuItem viewToolbar;
    [Glade.Widget] private CheckMenuItem viewStatusbar;
    [Glade.Widget] private CheckMenuItem viewTransferForm;

    private Actions actions = new Actions ();

    [Glade.Widget] private Widget fileSave;
    [Glade.Widget] private Widget saveButton;
    [Glade.Widget] private Widget fileSaveAs;
    [Glade.Widget] private Widget fileProperties;
    [Glade.Widget] private Widget homeButton;
    [Glade.Widget] private Widget goHome;
    [Glade.Widget] private Widget goIncomeStatement;
    [Glade.Widget] private Widget goBalanceSheet;
    [Glade.Widget] private Widget goTrialBalance;
    [Glade.Widget] private Widget fileClose;

    [Glade.Widget] private Widget editMoveUp;
    [Glade.Widget] private Widget editMoveDown;
    [Glade.Widget] private Widget filePrint;
    [Glade.Widget] private Widget printButton;
    [Glade.Widget] private Widget editAdd;
    [Glade.Widget] private Widget addButton;
    [Glade.Widget] private Widget editRemove;
    [Glade.Widget] private Widget removeButton;
    [Glade.Widget] private Widget editDetails;
    [Glade.Widget] private Widget detailsButton;
    [Glade.Widget] private Widget editProperties;
    [Glade.Widget] private Widget propertiesButton;
    [Glade.Widget] private Widget editUndo;
    [Glade.Widget] private Widget undoButton;
    [Glade.Widget] private Widget editRedo;
    [Glade.Widget] private Widget redoButton;
    [Glade.Widget] private Widget viewExpandAll;
    [Glade.Widget] private Widget viewCollapseAll;
    [Glade.Widget] private Widget goBack;
    [Glade.Widget] private Widget goForward;
    [Glade.Widget] private Widget backButton;
    [Glade.Widget] private Widget forwardButton;
    [Glade.Widget] private MenuItem fileOpenRecent;

    [Glade.Widget] private Menu editContextMenu;

    private Data.Base dataBase;
    private FrameManager frameManager;
    private SummaryFrame summaryFrame;
    private UndoManager undoManager;
    private ResponseHandler open_project_dialog_response;
    private ResponseHandler save_project_dialog_response;


    private TrialBalanceFrame trialBalanceFrame;
    private IncomeStatementFrame incomeStatementFrame;
    private BalanceSheetFrame balanceSheetFrame;

    private LoaderThread loader;
    private SaverThread saver;

    private const int idSaving = 1;
    private const int idLoading = 2;

    MessageDialog cancelSaveDialog;

    public MainWindow (string filename) : base ("mainWindow", null)
    {
        Init ();

        mainStatusbar.Push (idLoading, "Loading " + filename);

        loader = new LoaderThread (filename, new LoaderHandler (LoaderFinished));
    }

    private void GenericResponse (object obj, ResponseArgs args)
    {
        (obj as Dialog).Destroy ();
    }

    private void LoaderFinished (string filename, Data.Base dataBase)
    {
        Gdk.Threads.Enter ();

        mainStatusbar.Pop (idLoading);

        if (dataBase != null)
        {
            this.dataBase = dataBase;
            AddContent ();
        }
        else
        {
            Kurush.OpenRecentManager.Remove (filename);
            this.dataBase = new Data.Base ();
            AddContent ();
            MessageDialog md =
                new MessageDialog (mainWindow,
                                   DialogFlags.DestroyWithParent,
                                   MessageType.Error,
                                   ButtonsType.Close,
                                   "Error loading file " + filename);

            md.Response += new ResponseHandler (GenericResponse);
            md.Show ();
        }

        Gdk.Threads.Leave ();
    }

    public MainWindow () : base ("mainWindow", null)
    {
        Init ();

        this.dataBase = new Data.Base ();

        AddContent ();
    }

    private void Init ()
    {
        Kurush.OpenRecentManager.AddClient (fileOpenRecent);


        open_project_dialog_response =
            new ResponseHandler (OpenProjectDialogResponse);

        save_project_dialog_response =
            new ResponseHandler (SaveProjectDialogResponse);

        actions.LoadContext = new Action (fileSave, saveButton, fileSaveAs,
                                          fileProperties, homeButton, goHome,
                                          goIncomeStatement, goBalanceSheet,
                                          goTrialBalance);

        actions.FilePrint = new Action (filePrint, printButton);
        actions.EditDetails = new Action (editDetails, detailsButton);
        actions.EditProperties = new Action (editProperties, propertiesButton);
        actions.EditAdd = new Action (editAdd, addButton);
        actions.EditRemove = new Action (editRemove, removeButton);
        actions.EditMove = new Action (editMoveUp, editMoveDown);
        actions.ViewExpandCollapse = new Action (viewExpandAll, viewCollapseAll);

        actions.EditUndo = new Action (editUndo, undoButton);
        actions.EditRedo = new Action (editRedo, redoButton);
        actions.GoBack = new Action (goBack, backButton);
        actions.GoForward = new Action (goForward, forwardButton);

        mainWindow.SetDefaultSize (Kurush.Keys.Width, Kurush.Keys.Height);

        if (Kurush.Keys.Maximized)
            mainWindow.Maximize ();

        viewToolbar.Active = Kurush.Keys.Toolbar;
        viewStatusbar.Active = Kurush.Keys.Statusbar;
        viewTransferForm.Active = Kurush.Keys.TransferForm;

        mainWindow.WindowStateEvent +=
            new WindowStateEventHandler (on_mainWindow_state_event);

        mainWindow.Present ();
    }


    private void AddContent ()
    {
        frameManager = new FrameManager (mainWindow, dataBase, actions, goMenu_menu);
        frameManager.Show ();
        mainVBox.PackStart (frameManager);

        undoManager = new UndoManager (dataBase, actions.EditUndo, actions.EditRedo);

        summaryFrame = new SummaryFrame (frameManager);
        summaryFrame.Title = "Summary";
        summaryFrame.TitleVisible = false;

        frameManager.Go (summaryFrame);

        SetTitle (this.dataBase.Filename);

        saver = new SaverThread (new SaverHandler (SaverFinished),
                                 new SaverHandler (SaverAborted));

        actions.LoadContext.Enabled = true;

        dataBase.ModifiedChanged += new EventHandler (ModifiedChanged);
        ModifiedChanged (null, null);
    }

    private void ModifiedChanged (object obj, EventArgs args)
    {
        fileSave.Sensitive = dataBase.Modified;
        saveButton.Sensitive = dataBase.Modified;
    }

    public void SetTitle (string title)
    {
        if (title != null)
            mainWindow.Title = title + " - Kurush";
        else
            mainWindow.Title = "Untitled - Kurush";
    }

    public bool CanClose
    {
        get
        {
            if (loader != null)
                if (loader.Stop ())
                    return true;

            if (saver.Active)
            {
                CancelSave ();
                return false;
            }

            if (dataBase.Modified)
            {
                mainWindow.Present ();
                bool result = true;

                SaveWarningDialog dialog =
                    new SaveWarningDialog (mainWindow, dataBase.Filename);

                switch (dialog.Run ())
                {
                case (int) ResponseType.Ok:
                    result = true;
                    Save ();
                    break;
                case (int) ResponseType.Cancel:
                    result = false;
                    break;
                case (int) ResponseType.No:
                    result = true;
                    break;
                }

                return result;
            }
            else
                return true;
        }
    }

    public void Close ()
    {
        Kurush.Keys.Toolbar = mainHandleBox.Visible;
        Kurush.Keys.Statusbar = mainStatusbar.Visible;
        Kurush.Keys.TransferForm = viewTransferForm.Active;

        if (!Kurush.Keys.Maximized)
        {
            Kurush.Keys.Width = mainWindow.Allocation.Width;
            Kurush.Keys.Height = mainWindow.Allocation.Height;
        }

        if (this.dataBase != null)
            Kurush.OpenRecentManager.Add (this.dataBase.Filename);

        Kurush.OpenRecentManager.RemoveClient (fileOpenRecent);

        mainWindow.Destroy ();
    }

    private void CancelSave ()
    {
        cancelSaveDialog =
            new MessageDialog (mainWindow,
                               DialogFlags.DestroyWithParent,
                               MessageType.Warning,
                               ButtonsType.YesNo,
                               "Currently saving " + dataBase.Filename +
                               "\nDo you want to cancel saving?");

        cancelSaveDialog.Response += new ResponseHandler (GenericResponse);
        cancelSaveDialog.Show ();
    }

    private void SaverAborted (bool success)
    {
        System.Diagnostics.Debug.WriteLine ("SaverAborted Handler");
    }

    private void SaverFinished (bool success)
    {
        System.Diagnostics.Debug.WriteLine ("SaverFinished Handler");

        mainStatusbar.Pop (idSaving);

        if (cancelSaveDialog != null)
        {
            cancelSaveDialog.Destroy ();
            cancelSaveDialog = null;
        }
    }

    public void Save ()
    {
        if (dataBase == null)
            return;

        if (dataBase.Modified)
        {
            if (dataBase.Filename != null)
            {
                if (saver.Start (dataBase))
                    mainStatusbar.Push (idSaving, "Saving " + dataBase.Filename);
            }
            else
                SaveAs ();
        }
    }

    public void SaveAs ()
    {
        if (dataBase == null)
            return;

        FileSelection dialog = new FileSelection ("Save As");
        dialog.TransientFor = mainWindow;
        dialog.DestroyWithParent = true;
        dialog.Response += save_project_dialog_response;
        dialog.Show ();

    }

    private void OpenProjectDialogResponse (object obj, ResponseArgs args)
    {
        FileSelection dialog = obj as FileSelection;
        bool destroy = true;

        if (args.ResponseId == ResponseType.Ok)
        {
            string name = dialog.Filename;
            if (!System.IO.Directory.Exists (name))
                Kurush.NewMainWindow (name);
        }

        dialog.Destroy ();
    }


    private void SaveProjectDialogResponse (object obj, ResponseArgs args)
    {
        FileSelection dialog = obj as FileSelection;
        bool destroy = true;

        if (args.ResponseId == ResponseType.Ok)
        {
            string name = dialog.Filename;

            if (!System.IO.Directory.Exists (name))
            {
                if (System.IO.File.Exists (name))
                {
                    MessageDialog md =
                        new MessageDialog (dialog,
                                           DialogFlags.DestroyWithParent,
                                           MessageType.Warning,
                                           ButtonsType.YesNo,
                                           "A file named \"" + name +
                                           "\" already exists.\nDo you want " +
                                           "to replace it with the one you " +
                                           "are saving?");

                    if (md.Run () == (int) ResponseType.Yes)
                    {
                        dataBase.Filename = name;
                        if (saver.Start (dataBase))
                            mainStatusbar.Push (idSaving, "Saving " + dataBase.Filename);
                        SetTitle (name);
                    }
                    else
                        destroy = false;

                    md.Destroy ();
                }
                else
                {
                    dataBase.Filename = name + Kurush.Extension;
                    if (saver.Start (dataBase))
                        mainStatusbar.Push (idSaving, "Saving " + dataBase.Filename);
                    SetTitle (dataBase.Filename);
                }
            }
            else
                destroy = false;

        }

        if (destroy)
            dialog.Destroy ();
    }

    /***************************************************************************
     EVENT HANDLERS
     **************************************************************************/

    private void on_mainWindow_state_event (object obj, WindowStateEventArgs args)
    {
        if (args.Event.NewWindowState == Gdk.WindowState.Maximized)
        {
            if (!Kurush.Keys.Maximized)
            {
                Kurush.Keys.Width = mainWindow.Allocation.Width;
                Kurush.Keys.Height = mainWindow.Allocation.Height;
            }
            Kurush.Keys.Maximized = true;

        }
        else
            if (args.Event.NewWindowState == 0)
                Kurush.Keys.Maximized = false;
    }

    private void on_mainWindow_delete_event (object obj, DeleteEventArgs args)
    {
        args.RetVal = true;

        if (CanClose)
            Kurush.CloseMainWindow (this);
    }

    private void on_fileNew_activate (object obj, EventArgs args)
    {
        Kurush.NewMainWindow ();
    }

    private void on_fileOpen_activate (object obj, EventArgs args)
    {
        FileSelection dialog = new FileSelection ("Open");
        dialog.TransientFor = mainWindow;
        dialog.DestroyWithParent = true;
        dialog.Response += open_project_dialog_response;
        dialog.Show ();
    }

    private void on_fileSave_activate (object obj, EventArgs args)
    {
        Save ();
    }


    private void on_fileSaveAs_activate (object obj, EventArgs args)
    {
        SaveAs ();
    }

    private void on_fileProperties_activate (object obj, EventArgs args)
    {
        PropertiesDialog dialog = new PropertiesDialog (mainWindow, dataBase);
    }

    private void on_filePrint_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.PrintReport ();
    }


    private void on_filePrintSetup_activate (object obj, EventArgs args)
    {
        PrintSetupDialog dialog = new PrintSetupDialog (Kurush.PrConfig);

        dialog.TransientFor = mainWindow;
        dialog.Response += new ResponseHandler (GenericResponse);
        dialog.Show ();
    }

    private void on_fileClose_activate (object obj, EventArgs args)
    {
        if (CanClose)
            Kurush.CloseMainWindow (this);
    }

    private void on_fileQuit_activate (object obj, EventArgs args)
    {
        Kurush.Quit();
    }

    private void on_editUndo_activate (object obj, EventArgs args)
    {
        if (undoManager != null)
            undoManager.Undo ();
    }

    private void on_editRedo_activate (object obj, EventArgs args)
    {
        if (undoManager != null)
            undoManager.Redo ();
    }

    private void on_editDetails_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.Details ();
    }

    private void on_editAdd_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.Add ();
    }

    private void on_editRemove_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.Remove ();
    }

    private void on_editProperties_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.Properties ();
    }

    private void on_editMoveUp_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.MoveUp ();
    }

    private void on_editMoveDown_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.MoveDown ();
    }

    private void on_editPreferences_activate (object obj, EventArgs args)
    {
        PreferencesDialog dialog = new PreferencesDialog (mainWindow);
    }

    private void on_viewToolbar_activate (object obj, EventArgs args)
    {
        mainHandleBox.Visible = viewToolbar.Active;
    }

    private void on_viewStatusbar_activate (object obj, EventArgs args)
    {
        mainStatusbar.Visible = viewStatusbar.Active;
    }

    private void on_viewTransferForm_activate (object obj, EventArgs args)
    {
        actions.TransferFormActive = viewTransferForm.Active;
    }

    private void on_viewExpandAll_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.ExpandAll ();
    }

    private void on_viewCollapseAll_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.CurrentFrame.CollapseAll ();
    }

    private void on_goBack_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.Back ();
    }

    private void on_goForward_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.Forward ();
    }

    private void on_goHome_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
            frameManager.Go (summaryFrame);
    }

    private void on_goAccountLedgers_activate (object obj, EventArgs args)
    {

    }

    private void on_goTrialBalance_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
        {
            if (!frameManager.Contains (trialBalanceFrame))
                trialBalanceFrame = new TrialBalanceFrame (frameManager);

            frameManager.Go (trialBalanceFrame);
        }
    }

    private void on_goBalanceSheet_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
        {
            if (!frameManager.Contains (balanceSheetFrame))
                balanceSheetFrame = new BalanceSheetFrame (frameManager);

            frameManager.Go (balanceSheetFrame);
        }
    }

    private void on_goIncomeStatement_activate (object obj, EventArgs args)
    {
        if (frameManager != null)
        {
            if (!frameManager.Contains (incomeStatementFrame))
                incomeStatementFrame = new IncomeStatementFrame (frameManager);

            frameManager.Go (incomeStatementFrame);
        }
    }

    private void on_helpContents_activate (object obj, EventArgs args)
    {
    }

    private void on_helpAbout_activate (object obj, EventArgs args)
    {
        AboutDialog aboutDialog = new AboutDialog (mainWindow);
    }



}

}
