// =============================================================================
//
//      --- kvi_string.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviString"

#define _KVI_STRING_CPP_

#include "kvi_debug.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_settings.h"
#include "kvi_string.h"

bool kvi_matchWildExpr(register const char *m1, register const char *m2)
{
	// Matches two regular expressions containing wildcards

	//          s1
	//          m1
	// mask1 : *xor
	// mask2 : xorand*xor
	//         m2
	//          s2

	//                        s2
	//                       m2
	//                       |
	// XorT!xor@111.111.111.11
	//
	// *!*@*.net
	//      |
	//      m1
	//      s1
	//

	if( !(m1 && m2 && (*m1)) ) return false;
	const char *savePos1 = 0;
	const char *savePos2 = m2;
	while( *m1 ) {
		// Loop managed by m1 (initially first mask)
		if( *m1 == '*' ) {
			// Found a wildcard in m1
			savePos1 = ++m1;              // Move to the next char and save the position. This is our jolly
			if( !*savePos1 ) return true; // Last was a wildcard, matches everything ahead...
			savePos2 = m2 + 1;            // Next return state for the second string
			continue;                     // And return
		}
		if( !(*m2) ) return false;        // m2 finished and we had something to match here!
		if( tolower(*m1) == tolower(*m2) ) {
			// Chars matched
			m1++;                         // Go ahead in the two strings
			m2++;                         //
			if( !(*m1) && *m2 && savePos1 ) {
				// m1 finished, but m2 not yet and we have a savePosition for m1 (there was a wildcard)...
				// Retry matching the string following the * from the savePos2 (one char ahead last time)
				m1 = savePos1;          // Back to char after wildcard
				m2 = savePos2;          // Back to last savePos2
				savePos2++;             // Next savePos2 will be next char
			}
		} else {
			if( *m2 == '*' ) {
				// A wildcard in the second string
				// Invert the game : mask1 <-> mask2
				// mask2 now leads the game...
				savePos1 = m1;          // Aux
				m1 = m2;                // ... swap
				m2 = savePos1;          // ... swap
				savePos1 = m1;          // Sync save pos1
				savePos2 = m2 + 1;      // Sync save pos2
				continue;               // ... and again
			}
			if( savePos1 ) {            // Have a jolly man. Allow not matching
				m1 = savePos1;          // Go back to char after wildcard. Need to rematch
				m2 = savePos2;          // Back to last savePos2
				savePos2++;             // And set next savePos2
			} else return false;        // No previous wildcards... not matched!
		}
	}
	return (!(*m2));                    // m1 surely finished, so for the match, m2 must be finished too

}

bool kvi_matchWildExprCS(register const char *m1, register const char *m2)
{
	if( !(m1 && m2 && (*m1)) ) return false;
	const char *savePos1 = 0;
	const char *savePos2 = m2;
	while( *m1 ) {
		// Loop managed by m1 (initially first mask)
		if( *m1 == '*' ) {
			// Found a wildcard in m1
			savePos1 = ++m1;              // Move to the next char and save the position. This is our jolly
			if( !*savePos1 ) return true; // Last was a wildcard, matches everything ahead...
			savePos2 = m2 + 1;            // Next return state for the second string
			continue;                     // And return
		}
		if( !(*m2) ) return false;        // m2 finished and we had something to match here!
		if( (*m1) == (*m2) ) {
			// Chars matched
			m1++;                         // Go ahead in the two strings
			m2++;                         //
			if( !(*m1) && *m2 && savePos1 ) {
				// m1 finished, but m2 not yet and we have a savePosition for m1 (there was a wildcard)...
				// Retry matching the string following the * from the savePos2 (one char ahead last time)
				m1 = savePos1;            // Back to char after wildcard
				m2 = savePos2;            // Back to last savePos2
				savePos2++;               // Next savePos2 will be next char
			}
		} else {
			if( *m2 == '*' ) {
				// A wildcard in the second string
				// Invert the game : mask1 <-> mask2
				// mask2 now leads the game...
				savePos1 = m1;          // Aux
				m1 = m2;                // ... swap
				m2 = savePos1;          // ... swap
				savePos1 = m1;          // Sync save pos1
				savePos2 = m2 + 1;      // Sync save pos2
				continue;               // ... and again
			}
			if( savePos1 ) {            // Have a jolly man. Allow not matching
				m1 = savePos1;          // Go back to char after wildcard. Need to rematch
				m2 = savePos2;          // Back to last savePos2
				savePos2++;             // And set next savePos2
			} else return false;        // No previous wildcards; not matched!
		}
	}
	return (!(*m2));                    // m1 surely finished, so for the match, m2 must be finished too
}

const char *kvi_extractToken(QString &str, const char *aux_ptr, char sep)
{
	KviStr _str(str);
	const char *token = kvi_extractToken(_str, aux_ptr, sep);
	str = QString(_str.ptr());
	return token;
}

const char *kvi_extractToken(KviStr &str, const char *aux_ptr, char sep)
{
	__range_valid(aux_ptr);
	while( *aux_ptr && (*aux_ptr == sep) )
		aux_ptr++;
	const char *p = aux_ptr;
	while( *p && (*p != sep) )
		p++;
	str.m_len = p-aux_ptr;
	str.m_ptr = (char *) kvi_realloc(str.m_ptr, str.m_len + 1);
	kvi_fastmove(str.m_ptr, aux_ptr, str.m_len);
	*(str.m_ptr + str.m_len) = '\0';
	while( *p && (*p == sep) )
		p++;
	return p;
}

const char *kvi_extractUpTo(KviStr &str, const char *aux_ptr, char sep)
{
	__range_valid(aux_ptr);
	const char *p = aux_ptr;
	while( *p && (*p != sep) ) p++;
	str.m_len = p - aux_ptr;
	str.m_ptr = (char *) kvi_realloc(str.m_ptr, str.m_len + 1);
	kvi_fastmove(str.m_ptr, aux_ptr, str.m_len);
	*(str.m_ptr + str.m_len) = '\0';
	return p;
}

int kvi_vsnprintf(char *buffer, int len, const char *fmt, va_list list)
{
	__range_valid(fmt);
	__range_valid(buffer);
	__range_valid(len > 0); // Printing 0 characters is senseless
	register char *p = buffer;
	char *argString;
	long argValue;
	unsigned long argUValue;
	// 123456789 123456789 123456789 1\0
	char numberBuffer[32]; // Enough? 10 is enough for 32bit unsigned int...
	char *pNumBuf;
	unsigned int tmp;

	for( p = buffer; *fmt; ++fmt ) {
		if( len < 1 ) return (-1); // Not enough space... (in fact this could be len < 2 for the terminator)
		// Copy up to a '%'
		if( *fmt != '%' ) {
			*p = *fmt;
			p++;
			--len;
			continue;
		}
		++fmt; // Skip this '%'
		switch( *fmt ) {
			case 's': // String
				argString = va_arg(list, char *);
				if( !argString ) {
					argString = "[!NULL!]";
					argValue = 8;
				} else argValue = strlen(argString);
				// Check for space...
				if( len <= argValue ) return (-1); // Not enough space for buffer and terminator
				while( *argString ) {
					*p = *argString;
					p++;
					argString++;
				}
				len -= argValue;
				continue;
			case 'd': // Signed integer
				argValue = va_arg(list, int);
				if( argValue < 0 ) { // Negative integer
					*p = '-';
					p++;
					if( --len == 0 ) return (-1);
					argValue = -argValue; // Need to have it positive
					// Most negative integer exception (avoid completely senseless (non digit) resposnes)
					if( argValue < 0 ) argValue = 0;  // We get -0 here
				}
				// Write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argValue / 10;
					*pNumBuf = argValue - (tmp * 10) + '0';
					pNumBuf++;
				} while( (argValue = tmp) );
				// Copy now
				argUValue = pNumBuf - numberBuffer; // Length of the number string
				if( ((uint) len) <= argUValue ) return (-1); // Not enough space for number and terminator
				do {
					pNumBuf--;
					*p = *pNumBuf;
					p++;
				} while( pNumBuf != numberBuffer );
				len -= argUValue;
				continue;
			case 'u': // Unsigned integer
				argUValue = va_arg(list, unsigned int); // Many implementations place int here
				// Write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argUValue / 10;
					*pNumBuf = argUValue - (tmp * 10) + '0';
					pNumBuf++;
				} while( (argUValue = tmp) );
				// Copy now
				argValue = pNumBuf - numberBuffer; // Length of the number string
				if( len <= argValue ) return (-1); // Not enough space for number and terminator
				do {
					pNumBuf--;
					*p = *pNumBuf;
					p++;
				} while( pNumBuf != numberBuffer );
				len -= argValue;
				continue;
			case 'c': // Char
				//
				// I am not sure about this...
				// In the linux kernel source the
				// unsigned char is extracted from an integer type.
				// We assume that gcc stacks a char argument
				// as sizeof(int) bytes value.
				// Is this always true?
				//
				*p = (char) va_arg(list, int);
				p++;
				--len;
				continue;
			default: // A normal percent
				*p = '%';  // Write it
				p++;
				if( --len == 0 ) return (-1); // Not enough space for next char or terminator
				if( *fmt ) {
					// This if is just in case we have a % at the end of the string.
					*p = *fmt; // And write this char
					p++;
					--len;
				}
				continue;
		}
	}
	if( len < 1 ) return (-1); // Missing space for terminator
	*p = '\0';
	return p-buffer;
}

//
// Nearly the same as the above function...
//
int kvi_irc_vsnprintf(char *buffer, const char *fmt, va_list list, bool *bTruncated)
{
	__range_valid(fmt);
	__range_valid(buffer);
	register char *p = buffer;
	char *argString;
	long argValue;
	unsigned long argUValue;
	char numberBuffer[64]; // Enough? 10 is enough for 32bit unsigned int...
	char *pNumBuf;
	unsigned int tmp;
	*bTruncated = false;
	int len = 512;

	for( p = buffer; *fmt; ++fmt ) {
		if( len < 3 ) goto truncate;
		// Copy up to a '%'
		if( *fmt != '%' ) {
			*p = *fmt;
			p++;
			--len;
			continue;
		}
		++fmt; // Skip this '%'
		switch( *fmt ) {
			case 's': // String
				argString = va_arg(list, char *);
				if( !argString ) argString = "[!NULL!]";
				// Check for space...
				while( *argString ) {
					*p = *argString;
					p++;
					argString++;
					if( --len < 3 ) goto truncate;
				}
				continue;
			case 'd': // Signed integer
				argValue = va_arg(list, int);
				if( argValue < 0 ) { // Negative integer
					*p = '-';
					p++;
					if( --len < 3 ) goto truncate; // Place just for CRLF
					argValue = -argValue; // Need to have it positive
					if( argValue < 0 ) argValue = 0; // -0 (hack the exception)
				}
				// Write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argValue / 10;
					*pNumBuf = argValue - (tmp * 10) + '0';
					pNumBuf++;
				} while( (argValue = tmp) );
				// Copy now
				do {
					pNumBuf--;
					*p = *pNumBuf;
					p++;
					if( --len < 3 ) goto truncate;
				} while( pNumBuf != numberBuffer );
				continue;
			case 'u': // Unsigned integer
				argUValue = va_arg(list, unsigned int); // Many implementations place int here
				// Write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argUValue / 10;
					*pNumBuf = argUValue - (tmp * 10) + '0';
					pNumBuf++;
				} while( (argUValue = tmp) );
				// Copy now
				if( --len < 3 ) goto truncate; // No place for digits
				do {
					pNumBuf--;
					*p = *pNumBuf;
					p++;
					if( --len < 3 ) goto truncate;
				} while( pNumBuf != numberBuffer );
				continue;
			case 'c': // Char
				*p = (char) va_arg(list, int);
				p++;
				--len;
				continue;
			default: // A normal percent
				*p = '%';  // Write it
				p++;
				if( --len < 3 ) goto truncate; // Not enough space for next char
				if( *fmt ) {     // This if is just in case we have a % at the end of the string.
					*p = *fmt; // And write this char
					p++;
					--len;
				}
				continue;
		}
	}
	// Successful finish
	__range_valid(len >= 2);
	*p = '\r';
	p++;
	*p   = '\n';
	return ((p-buffer) + 1);
truncate:
	__range_valid(len >= 2);
	*bTruncated = true;
	*p = '\r';
	p++;
	*p   = '\n';
	return ((p-buffer) + 1);
}

#ifndef COMPILE_i386_ASM_CODE
bool kvi_strEqualCS(const char *str1, const char *str2)
{
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *) str1;
	register unsigned char *s2 = (unsigned char *) str2;
	while( *s1 )
		if( *s1++ != *s2++ ) return false;
	return (*s1 == *s2);
}

bool kvi_strEqualCSN(const char *str1, const char *str2, int len)
{
	__range_valid(str1);
	__range_valid(str2);
	__range_valid(len >= 0);
	register unsigned char *s1 = (unsigned char *) str1;
	register unsigned char *s2 = (unsigned char *) str2;
	while( len-- && *s1 )
		if( *s1++ != *s2++ ) return false;
	return (len < 0);
}
#endif // COMPILE_i386_ASM_CODE

bool kvi_strEqualCIN(const char *str1, const char *str2, int len)
{
	__range_valid(str1);
	__range_valid(str2);
	__range_valid(len >= 0);
	register unsigned char *s1 = (unsigned char *) str1;
	register unsigned char *s2 = (unsigned char *) str2;
	while( len-- && *s1 )
		if( tolower(*s1++) != tolower(*s2++) ) return false;
	return (len < 0);
}

bool kvi_strEqualCI(const char *str1, const char *str2)
{
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *) str1;
	register unsigned char *s2 = (unsigned char *) str2;
	while( *s1 )
		if( tolower(*s1++) != tolower(*s2++) ) return false;
	return (*s1 == *s2);
}

// Note that greater here means that it comes AFTER in alphabetic order
int kvi_strcmpCI(const char *str1, const char *str2)
{
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *) str1;
	register unsigned char *s2 = (unsigned char *) str2;
	int diff;
	unsigned char rightchar;
	while( !(diff = (rightchar = tolower(*s2++)) - tolower(*s1++)) )
		if( !rightchar ) break;
	return diff; // Diff is nonzero or end of both was reached (it is positive if *s2 > *s1)
}

int kvi_strcmpCS(const char *str1, const char *str2)
{
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *) str1;
	register unsigned char *s2 = (unsigned char *) str2;
	int diff;
	while( !(diff = (*s2) - (*s1++)) )
		if( !*s2++ ) break;
	return diff; // Diff is nonzero or end of both was reached (it is positive if *s2 > *s1)
}

int kvi_strMatchRevCS(const char *str1, const char *str2, int index)
{
	__range_valid(str1);
	__range_valid(str2);
	register char *s1=(char *) str1;
	register char *s2=(char *) str2;

	int curlen = strlen(str1);
	int diff;

	if( index < 0 || index >= curlen )
		index = curlen - 1;

	s1 += index;
	while( *s2 ) s2++;
	s2--;

	// Now start comparing
	while( 1 ) {
		// In this case, we have str1 = "lo" and str2 = "hello"
		if( s1 < str1 && !(s2 < str2) ) return 256;
		if( s2 < str2 ) return 0;
		if( (diff = (*s1) - (*s2)) ) return diff;
		s1--;
		s2--;
	}
}

KviStr::KviStr()
{
	m_ptr = (char *) kvi_malloc(1);
	*m_ptr = '\0';
	m_len = 0;
}

KviStr::KviStr(const char *str)
{
	// Deep copy constructor
	if( str ) {
		// Deep copy
		m_len = strlen(str);
		m_ptr = (char *) kvi_malloc(m_len + 1);
		kvi_fastmove(m_ptr, str, m_len + 1);
	} else {
		m_ptr = (char *) kvi_malloc(1);
		*m_ptr = '\0';
		m_len = 0;
	}
}

KviStr::KviStr(const char *str, int len)
{
	__range_valid(str);
	__range_valid(len <= ((int) strlen(str)));
	__range_valid(len >= 0);
	m_len = len;
	m_ptr = (char *) kvi_malloc(m_len + 1);
	kvi_fastmove(m_ptr, str, m_len);
	*(m_ptr + m_len) = '\0';
}

KviStr::KviStr(const char *bg, const char *end)
{
	__range_valid(bg);
	__range_valid(end);
	__range_valid(bg <= end);
	m_len = end - bg;
	m_ptr = (char *) kvi_malloc(m_len + 1);
	kvi_fastmove(m_ptr, bg, m_len);
	*(m_ptr + m_len) = '\0';
}

KviStr::KviStr(KviFormatConstructorTag tag, const char *fmt, ...)
{
	m_ptr = (char *) kvi_malloc(256);
	// First try
	va_list list;
	va_start(list, fmt);
	// Print with max 256 chars
	m_len = kvi_vsnprintf(m_ptr, 256, fmt, list);
	va_end(list);

	// Check if we failed
	if( m_len < 0 ) {
		// Yes, failed
		int dummy = 256;
		do { // We failed, so retry with 256 more chars
			dummy += 256;
			// Realloc
			m_ptr = (char *) kvi_realloc(m_ptr, dummy);
			// Print...
			va_start(list, fmt);
			m_len = kvi_vsnprintf(m_ptr, dummy, fmt, list);
			va_end(list);
		} while( m_len < 0 );
	}
	// Done...
	// Now m_len is the length of the written string not including the terminator...
	// Perfect! :)
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
}

KviStr::KviStr(const KviStr &str)
{
	__range_valid(str.m_ptr);
	m_len = str.m_len;
	m_ptr = (char *) kvi_malloc(m_len + 1);
	kvi_fastmove(m_ptr, str.m_ptr, m_len + 1);
}

KviStr::KviStr(const QString &str)
{
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	QCString tmp = str.local8Bit();
	const char *ptr = tmp.data();
#else
	const QCString ptr = str.utf8();
#endif
	if( ptr ) {
		m_len = strlen(ptr);
		m_ptr = (char *) kvi_malloc(m_len + 1);
		kvi_fastmove(m_ptr, ptr, m_len + 1);
	} else {
		m_ptr = (char *) kvi_malloc(1);
		*m_ptr = '\0';
		m_len = 0;
	}
}

KviStr::KviStr(char c, int fillLen)
{
	__range_valid(fillLen >= 0);
	m_len = fillLen;
	m_ptr = (char *) kvi_malloc(m_len + 1);
	register char *p = m_ptr;
	while( fillLen-- ) {
		*p = c;
		p++;
	}
	*p = '\0';
}

KviStr::~KviStr()
{
	kvi_free(m_ptr);
}

KviStr & KviStr::operator=(const KviStr &str)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	m_len = str.m_len;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	kvi_fastmove(m_ptr, str.m_ptr, m_len + 1);
	return (*this);
}

KviStr & KviStr::operator=(const char *str)
{
	if( str ) {
		m_len = strlen(str);
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		kvi_memmove(m_ptr, str, m_len + 1);
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, 1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}

bool KviStr::hasNonWhiteSpaceData() const
{
	const char *aux = m_ptr;
	while( *aux )
	{
		if( ((*aux) != ' ') && ((*aux) != '\t') )
			return true;
		aux++;
	}
	return false;
}

static char digits[16] = { '0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F' };

void KviStr::bufferToHex(char *buffer, int len)
{
	__range_valid(buffer);
	m_len = len << 1;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	char *aux = m_ptr;
	while( len )
	{
		*aux = digits[(unsigned int) (((unsigned char) (*buffer)) / 16)];
		aux++;
		*aux = digits[(unsigned int) (((unsigned char) (*buffer)) % 16)];
		aux++;
		len--;
		buffer++;
	}
	*(m_ptr + m_len) = '\0';
}

static char get_decimal_from_hex_digit_char(char dgt)
{
	if( (dgt >= '0') && (dgt <= '9') ) return (dgt - '0');
	if( (dgt >= 'A') && (dgt <= 'F') ) return (10 + (dgt - 'A'));
	if( (dgt >= 'a') && (dgt <= 'f') ) return (10 + (dgt - 'a'));
	return 0;
}

int KviStr::hexToBuffer(char **buffer, bool bNullToNewlines)
{
	int len;
	if( m_len % 2 )
		len = (m_len >> 1) + 1;
	else
		len = (m_len >> 1);
	*buffer = (char *) kvi_malloc(len);

	char *ptr = *buffer;

	char *aux = m_ptr;
	while( *aux ) {
		*ptr = get_decimal_from_hex_digit_char(*aux) * 16;
		aux++;
		if( *aux ) {
			*ptr += get_decimal_from_hex_digit_char(*aux);
			aux++;
		}
		if( bNullToNewlines )
			if( !(*ptr) ) *ptr = '\n';
		ptr++;
	}
	return len;
}

KviStr &KviStr::setStr(const char *str, int len)
{
	int alen = strlen(str);
	if( (len < 0) || (len > alen) )
		m_len = alen;
	else
		m_len = len;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	kvi_memmove(m_ptr, str, m_len);
	*(m_ptr + m_len) = '\0';
	return (*this);
}

KviStr & KviStr::operator=(const QString &str)
{
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	QCString tmp = str.local8Bit();
	const char *ptr = tmp.data();
#else
	const QCString ptr = str.utf8();
#endif
	if( ptr ) {
		m_len = strlen(ptr);
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		kvi_fastmove(m_ptr, ptr, m_len + 1);
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, 1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}

KviStr &KviStr::operator=(char c)
{
	m_len = 1;
	m_ptr = (char *) kvi_realloc(m_ptr, 2);
	*m_ptr = c;
	*(m_ptr + 1) = '\0';
	return (*this);
}

void KviStr::append(char c)
{
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 2);
	*(m_ptr + m_len) = c;
	m_len++;
	*(m_ptr + m_len) = '\0';
}

void KviStr::append(const KviStr &str)
{
	__range_valid(str.m_ptr);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + str.m_len + 1);
	kvi_fastmove(m_ptr + m_len, str.m_ptr, str.m_len + 1);
	m_len += str.m_len;
}

void KviStr::append(const char *str)
{
	if( !str )return;
	int len = strlen(str);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + len + 1);
	kvi_fastmove(m_ptr + m_len, str, len + 1);
	m_len += len;
}

void KviStr::append(const QString &str)
{
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	QCString tmp = str.local8Bit();
	const char *ptr = tmp.data();
#else
	const QCString ptr = str.utf8();
#endif
	if( !ptr ) return;
	int len = strlen(ptr);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + len + 1);
	kvi_fastmove(m_ptr + m_len, ptr, len + 1);
	m_len += len;
}

void KviStr::append(const char *str, int len)
{
	__range_valid(str);
	__range_valid(len <= ((int) strlen(str)));
	__range_valid(len >= 0);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + len + 1);
	kvi_fastmove(m_ptr + m_len, str, len);
	m_len += len;
	*(m_ptr + m_len) = '\0';
}

void KviStr::append(KviFormatConstructorTag tag, const char *fmt, ...)
{
	int auxLen;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 256);
	// First try
	va_list list;
	va_start(list, fmt);
	// Print... with max 256 chars
	auxLen = kvi_vsnprintf(m_ptr + m_len, 256, fmt, list);
	va_end(list);

	// Check if we failed
	if( auxLen < 0 ) {
		// Yes, failed
		int dummy = 256;
		do { // We failed, so retry with 256 more chars
			dummy += 256;
			// Realloc
			m_ptr = (char *) kvi_realloc(m_ptr, m_len + dummy);
			// Print...
			va_start(list, fmt);
			auxLen = kvi_vsnprintf(m_ptr + m_len, dummy, fmt, list);
			va_end(list);
		} while( auxLen < 0 );
	}
	m_len += auxLen;
	// Done...
	// Now m_len is the length of the written string not including the terminator...
	// Perfect! :)
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
}

void KviStr::extractFromString(const char *begin, const char *end)
{
	__range_valid(begin);
	__range_valid(end);
	__range_valid(end >= begin);
	m_len = end - begin;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	kvi_fastmove(m_ptr, begin, m_len);
	*(m_ptr + m_len) = '\0';
}

void KviStr::prepend(const KviStr &str)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + str.m_len + 1);
	kvi_memmove(m_ptr + str.m_len, m_ptr, m_len + 1); // Move self
	kvi_fastmove(m_ptr, str.m_ptr, str.m_len);
	m_len += str.m_len;
}

void KviStr::prepend(const char *str)
{
	if( !str ) return;
	int len = strlen(str);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + len + 1);
	kvi_memmove(m_ptr + len, m_ptr, m_len + 1); // Move self
	kvi_fastmove(m_ptr, str, len);
	m_len += len;
}

void KviStr::prepend(const char *str, int len)
{
	__range_valid(str);
	__range_valid(len <= ((int) strlen(str)));
	__range_valid(len >= 0);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + len + 1);
	kvi_memmove(m_ptr + len, m_ptr, m_len + 1); // Move self
	kvi_fastmove(m_ptr, str, len);
	m_len += len;
}

void KviStr::toUpper()
{
	register char *p = m_ptr;
	while( *p ) {
		*p = toupper(*p);
		p++;
	}
}

void KviStr::toLower()
{
	register char *p = m_ptr;
	while( *p ) {
		*p = tolower(*p);
		p++;
	}
}

KviStr KviStr::upper() const
{
	KviStr tmp(*this);
	tmp.toUpper();
	return tmp;
}

KviStr KviStr::lower() const
{
	KviStr tmp(*this);
	tmp.toLower();
	return tmp;
}

KviStr KviStr::left(int maxLen)
{
	if( maxLen <= 0 ) {
		KviStr empty;
		return empty;
	}
	if( maxLen > m_len ) maxLen = m_len;
	KviStr str(m_ptr, maxLen);
	return str;
}

KviStr KviStr::right(int maxLen)
{
	if( maxLen <= 0 ) {
		KviStr empty;
		return empty;
	}
	if( maxLen > m_len ) maxLen = m_len;
	KviStr str(m_ptr + (m_len - maxLen), maxLen);
	return str;
}

KviStr KviStr::middle(int idx, int maxLen)
{
	__range_valid(maxLen >= 0);
	__range_valid(idx >= 0);
	if( (maxLen <= 0) || (idx < 0) ) { // Max len negative; invalid params
		KviStr ret;
		return ret;
	}
	if( (maxLen + idx) <= m_len ) { // Valid params
		KviStr str(m_ptr + idx, maxLen);
		return str;
	}
	if( idx < m_len ) { // String shorter than requested
		KviStr str(m_ptr + idx);
		return str;
	}
	// idx out of bounds
	KviStr ret;
	return ret;
}

KviStr &KviStr::insert(int idx, const char *data)
{
	__range_valid(data);
	if( idx <= m_len ) {
		int len = strlen(data);
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + len + 1);
		kvi_memmove(m_ptr + idx + len, m_ptr + idx, m_len - idx + 1);
		kvi_fastmove(m_ptr + idx, data, len);
		m_len += len;
	}
	return (*this);
}

KviStr &KviStr::insert(int idx, char c)
{
	if( idx <= m_len ) {
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 2);
		kvi_memmove(m_ptr + idx + 1, m_ptr + idx, m_len - idx + 1);
		m_len++;
		*(m_ptr + idx) = c;
	}
	return (*this);
}

KviStr &KviStr::replaceAll(char c, const char *str)
{
	int idx = findFirstIdx(c);
	KviStr tmp;
	while( idx >= 0 ) {
		if( idx > 0 ) tmp += left(idx);
		cutLeft(idx + 1);
		tmp.append(str);
		idx = findFirstIdx(c);
	}
	tmp.append(*this);
	// Now copy
	m_len = tmp.m_len;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	kvi_fastmove(m_ptr, tmp.m_ptr, m_len + 1);
	return (*this);
}

KviStr &KviStr::replaceAll(char *toFind, const char *str, bool bCaseS)
{
	int len = strlen(toFind);
	int idx = findFirstIdx(toFind, bCaseS);
	KviStr tmp;
	while( idx >= 0 ) {
		if( idx > 0 ) tmp += left(idx);
		cutLeft(idx + len);
		tmp.append(str);
		idx = findFirstIdx(toFind, bCaseS);
	}
	tmp.append(*this);
	// Now copy
	m_len = tmp.m_len;
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	kvi_fastmove(m_ptr, tmp.m_ptr, m_len + 1);
	return (*this);
}

int KviStr::contains(char c, bool caseS)
{
	register char *p = m_ptr;
	int cnt = 0;
	if( caseS ) {
		while( *p ) {
			if( *p == c ) cnt++;
			p++;
		}
	} else {
		char b = tolower(c);
		while( *p ) {
			if( tolower(*p) == b ) cnt++;
			p++;
		}
	}
	return cnt;
}

int KviStr::contains(const char *str, bool caseS)
{
	__range_valid(str);
	register char *p = m_ptr;
	int cnt = 0;
	int len = strlen(str);
	if( caseS ) {
		while( *p ) {
			if( *p == *str ) {
				if( kvi_strEqualCSN(p, str, len) )
					cnt++;
			}
			p++;
		}
	} else {
		while( *p ) {
			char c = tolower(*str);
			if( tolower(*p) == c ) {
				if( kvi_strEqualCIN(p, str, len) )
					cnt++;
			}
			p++;
		}
	}
	return cnt;
}

KviStr &KviStr::setNum(long num)
{
	char numberBuffer[30];
	bool bNegative = false;
	long tmp;
	register char *p;
	register char *pNumBuf = numberBuffer;

	if( num < 0 ) { // Negative integer
		bNegative = true;
		num = -num; // Need to have it positive
		if( num < 0 ) { // 2^31 exception
			// We need to avoid absurd responses like ".(./),." :)
			num = 0; // We get a negative zero here; it is still an exception
		}
	}

	// Write the number in a temporary buffer (at least '0')
	do {
		tmp = num / 10;
		*pNumBuf = num - (tmp * 10) + '0';
		pNumBuf++;
	} while( (num = tmp) );

	// Copy now
	m_len = pNumBuf - numberBuffer; // Length of the number string
	if( bNegative ) {
		m_len++;
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		p = m_ptr;
		*p = '-';
		p++;
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		p = m_ptr;
	}
	do {
		pNumBuf--;
		*p = *pNumBuf;
		p++;
	} while( pNumBuf != numberBuffer );
	*(m_ptr + m_len) = '\0';
	return (*this);
}

KviStr &KviStr::setNum(unsigned long num)
{
	char numberBuffer[30];
	unsigned long tmp;
	register char *p;
	register char *pNumBuf = numberBuffer;

	// Write the number in a temporary buffer (at least '0')
	do {
		tmp = num / 10;
		*pNumBuf = num - (tmp * 10) + '0';
		pNumBuf++;
	} while( (num = tmp) );

	// Copy now
	m_len = pNumBuf - numberBuffer; // Length of the number string
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	p = m_ptr;
	do {
		pNumBuf--;
		*p = *pNumBuf;
		p++;
	} while( pNumBuf != numberBuffer );
	*(m_ptr + m_len) = '\0';
	return (*this);
}

long KviStr::toLong(bool *bOk) const
{
	long result = 0;
	if( bOk ) *bOk = false;
	register char *p = m_ptr;
	bool bNeg = false;
	while( isspace(*p) ) p++; // Skip spaces
	if( *p == '-' ) {
		bNeg = true;
		p++;
	} else {
		if( *p == '+' ) p++;
	}
	if( isdigit(*p) ) { // Point to something interesting?
		do {
			result = (result * 10) + (*p - '0');
			p++;
		} while( isdigit(*p) );
		if( bNeg ) result = -result;
		while( isspace(*p) ) p++;     // Skip trailing spaces
		if( *p ) return 0;            // If this is not the end, die.
		if( bOk ) *bOk = true;
		return result;
	}
	return 0;
}

unsigned long KviStr::toULong(bool *bOk) const
{
	unsigned long result = 0;
	if( bOk ) *bOk = false;
	register char *p = m_ptr;
	while( isspace(*p) ) p++; // Skip spaces
	if( isdigit(*p) ) { // Point to something interesting?
		do {
			result = (result * 10) + (*p - '0');
			p++;
		} while( isdigit(*p) );
		while( isspace(*p) ) p++;     // Skip trailing spaces
		if( *p ) return 0;            // If this is not the end, die.
		if( bOk ) *bOk = true;
		return result;
	}
	return 0;
}

long KviStr::toLongExt(bool *bOk, int base)
{
	if( m_len == 0 ) {
		if( bOk ) *bOk = false;
		return 0;
	}
	char *endptr;
	long result = strtol(m_ptr, &endptr, base);
	if( *endptr ) {
		// Must be whitespace, otherwise there is trailing garbage inside
		while( isspace(*endptr) && (*endptr) ) endptr++;
		if( *endptr ) {
			// Still not at the end
			// Trailing garbage not allowed
			if( bOk ) *bOk = false;
			return result;
		}
	}
	if( bOk ) *bOk = true;
	return result;
}

KviStr &KviStr::cutLeft(int len)
{
	__range_valid(len >= 0);
	if( len <= m_len ) {
		m_len -= len;
		kvi_memmove(m_ptr, m_ptr + len, m_len + 1);
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, 1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}

KviStr &KviStr::cutRight(int len)
{
	__range_valid(len >= 0);
	if( len <= m_len ) {
		m_len -= len;
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		*(m_ptr + m_len) = '\0';
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, 1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}

KviStr &KviStr::cut(int idx, int len)
{
	__range_valid(idx >= 0);
	__range_valid(len >= 0);
	if( idx < m_len ) {
		// idx = 3, len, = 3, m_len = 10
		// 0123456789
		// abcdefghij
		//    ^  ^
		//   p1  p2
		char *p1 = m_ptr + idx;
		if( len + idx > m_len )
			len = m_len - idx;
		char *p2 = p1 + len;
		kvi_memmove(p1, p2, m_len - (len + idx) + 1);
		m_len -= len;
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	}
	return (*this);
}

KviStr &KviStr::cutToFirst(char c, bool bIncluded)
{
	int idx = findFirstIdx(c);
	if( idx != -1 )
		cutLeft(bIncluded ? idx + 1 : idx);
	return (*this);
}

KviStr &KviStr::cutFromFirst(char c, bool bIncluded)
{
	int idx = findFirstIdx(c);
	if( idx != -1 )
		cutRight(bIncluded ? (m_len - idx) : (m_len - (idx + 1)));
	return (*this);
}

KviStr &KviStr::cutToLast(char c, bool bIncluded)
{
	int idx = findLastIdx(c);
	if( idx != -1 )
		cutLeft(bIncluded ? idx + 1 : idx);
	return (*this);
}

KviStr &KviStr::setLen(int len)
{
	__range_valid(len >= 0);
	m_ptr = (char *) kvi_realloc(m_ptr, len + 1);
	*(m_ptr + len) = '\0';
	m_len = len;
	return (*this);
}

KviStr &KviStr::stripLeftWhiteSpace()
{
	register char *p = m_ptr;
	while( isspace(*p) ) p++;
	m_len -= p - m_ptr;
	kvi_memmove(m_ptr, p, m_len + 1);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	return (*this);
}

KviStr &KviStr::stripLeft(char c)
{
	__range_valid(c != '\0');
	register char *p = m_ptr;
	while( *p == c ) p++;
	m_len -= p - m_ptr;
	kvi_memmove(m_ptr, p, m_len + 1);
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	return (*this);
}

bool KviStr::getToken(KviStr &str, char sep)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	register char *p = m_ptr;
	// Skip to the end
	while( *p && (*p != sep) ) p++;
	// 0123456789
	// abcd xyz
	// ^   ^
	str.m_len = p - m_ptr;
	str.m_ptr = (char *) kvi_realloc(str.m_ptr, str.m_len + 1);
	kvi_fastmove(str.m_ptr, m_ptr, str.m_len);
	*(str.m_ptr + str.m_len) = '\0';
	while( *p && (*p == sep) ) p++;
	cutLeft(p - m_ptr);
	return (m_len != 0);
}

bool KviStr::getLine(KviStr &str)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	if( m_len == 0 ) return false;
	register char *p = m_ptr;
	// Skip to the end
	while( *p && (*p != '\n') ) p++;
	// 0123456789
	// abcd xyz
	// ^   ^
	str.m_len = p - m_ptr;
	str.m_ptr = (char *) kvi_realloc(str.m_ptr, str.m_len + 1);
	kvi_fastmove(str.m_ptr, m_ptr, str.m_len);
	*(str.m_ptr + str.m_len) = '\0';
	p++;
	cutLeft(p - m_ptr);
	return true;
}

KviStr KviStr::getToken(char sep)
{
	register char *p = m_ptr;
	while( *p && (*p != sep) ) p++;
	KviStr ret(m_ptr, p);
	while( *p && (*p == sep) ) p++;
	cutLeft(p - m_ptr);
	return ret;
}

KviStr &KviStr::sprintf(const char *fmt, ...)
{
	m_ptr = (char *) kvi_realloc(m_ptr, 256);
	// First try
	va_list list;
	va_start(list, fmt);
	// Print with max 256 chars
	m_len = kvi_vsnprintf(m_ptr, 256, fmt, list);
	va_end(list);

	// Check if we failed
	if( m_len < 0 ) {
		// Yes, failed
		int dummy = 256;
		do { // We failed, so retry with 256 more chars
			dummy += 256;
			// Realloc
			m_ptr = (char *) kvi_realloc(m_ptr, dummy);
			// Print...
			va_start(list, fmt);
			m_len = kvi_vsnprintf(m_ptr, dummy, fmt, list);
			va_end(list);
		} while( m_len < 0 );
	}
	// Done...
	// Now m_len is the length of the written string not including the terminator...
	// Perfect! :)
	m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
	return (*this);
}

int KviStr::findFirstIdx(char c)
{
	register char *p = m_ptr;
	while( *p && (*p != c) ) p++;
	return (*p ? p - m_ptr : -1);
}

int KviStr::findFirstIdx(const char *str, bool caseS)
{
	__range_valid(str);
	register char *p = m_ptr;
	int len = strlen(str);
	if( caseS ) {
		for( ;; ) {
			while( *p && (*p != *str) ) p++;
			if( *p ) {
				if( kvi_strEqualCSN(str, p, len) )
					return (p - m_ptr);
				p++;
			} else return -1;
		}
	} else {
		for( ;; ) {
			char tmp = toupper(*str);
			while( *p && (toupper(*p) != tmp) ) p++;
			if( *p ) {
				if( kvi_strEqualCIN(str, p, len) )
					return (p - m_ptr);
				p++;
			} else return -1;
		}
	}
}

int KviStr::findLastIdx(char c)
{
	// Empty string?
	if( m_len < 1 ) return -1;
	// p points to the last character in the string
	register char *p = (m_ptr + m_len) - 1;
	// Go back until we find a match or we run to the first char in the string.
	while( (*p != c) && (p > m_ptr) ) p--;
	// If *p == c --> matched, else we are at the beginning of the string.
	return ((*p == c) ? p - m_ptr : -1);
}

int KviStr::findLastIdx(const char *str, bool caseS)
{
	__range_valid(str);
	// Calculate the length of the searched string
	int len = strlen(str);
	// Too long?
	if( m_len < len ) return -1;
	// p points to the last character in the string
	register char *p = (m_ptr + m_len) - 1;
	if( caseS ) {
		for( ;; ) {
			// Go back until we find a character that matches or we run to the first char.
			while( (*p != *str) && (p > m_ptr) ) p--;
			if( *p == *str ) {
				// Maybe occurrence
				if( kvi_strEqualCSN(str, p, len) )
					return (p - m_ptr);
				// Nope. Continue if there is more data to check
				if( p == m_ptr ) return -1;
				p--;
			} else return -1; // Beginning of the string
		}
	} else {
		// Case insensitive
		for( ;; ) {
			// Go back until we find a character that matches or we run to the first char.
			char tmp = toupper(*str);
			while( (toupper(*p) != tmp ) && (p > m_ptr) )
				p--;
			if( toupper(*p) == tmp ) {
				// Maybe occurrence
				if( kvi_strEqualCIN(str, p, len))
					return (p - m_ptr);
				// Nope. Continue if there is more data to check
				if( p == m_ptr ) return -1;
				p--;
			} else return -1; // Beginning of the string
		}
	}
}

KviStr &KviStr::stripWhiteSpace()
{
	// 0123456789
	//    abcd   0
	// ^        ^
	// left   right
	register char *left  = m_ptr;
	register char *right = m_ptr + m_len - 1;
	// Skip initial spaces
	while( isspace(*left) ) left++;
	if( *left ) {
		// Valid string, left points to first non-space
		while( (right >= left) && isspace(*right) ) right--;
		// 0123456789
		//    abcd   0
		//    ^  ^
		// left   right
		m_len = (right - left) + 1;
		kvi_memmove(m_ptr, left, m_len);
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		*(m_ptr + m_len) = '\0';
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, 1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}

KviStr &KviStr::stripSpace()
{
	// 0123456789
	//    abcd   0
	// ^        ^
	// left   right
	register char *left  = m_ptr;
	register char *right = m_ptr + m_len - 1;
	// Skip initial spaces
	while( (*left == ' ') || (*left == '\t') ) left++;
	if( *left ) {
		// Valid string, left points to first non-space
		while( (right >= left) && ((*right == ' ') || (*right == '\t')) ) right--;
		// 0123456789
		//    abcd   0
		//    ^  ^
		// left   right
		m_len = right - left + 1;
		kvi_memmove(m_ptr, left, m_len);
		m_ptr = (char *) kvi_realloc(m_ptr, m_len + 1);
		*(m_ptr + m_len) = '\0';
	} else {
		m_ptr = (char *) kvi_realloc(m_ptr, 1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}


bool KviStr::isNum()
{
	register char *p = m_ptr;
	while( isspace(*p) ) p++;
	if( *p == '-' ) p++;
	if( !isdigit(*p) ) return false;
	while( isdigit(*p) ) p++;
	while( isspace(*p) ) p++;
	return (*p == '\0');
}

bool KviStr::isUnsignedNum()
{
	register char *p = m_ptr;
	while( isspace(*p) ) p++;
	if( !isdigit(*p) ) return false;
	while( isdigit(*p) ) p++;
	while( isspace(*p) ) p++;
	return (*p == '\0');
}
