#ifndef _KVI_STRING_H_INCLUDED_
#define _KVI_STRING_H_INCLUDED_

// =============================================================================
//
//      --- kvi_string.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

//
//   KviStr class implementation and string utility functions
//

#include <ctype.h>
#include <stdarg.h>

#include <qstring.h>

#include "kvi_settings.h"

//
//  Sigh...
//  IRC is not Unicode...(yet) :(
//
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	#define _CHAR_2_QSTRING(__chptr) QString::fromLocal8Bit(__chptr)
#else
	#define _CHAR_2_QSTRING(__chptr) QString(__chptr)
#endif

#undef __kvi_extern
#ifdef _KVI_STRING_CPP_
	#define __kvi_extern
#else
	#define __kvi_extern extern
#endif

/**
 * A simple string class.<br>
 * - No data sharing.<br>
 * - Not UNICODE.<br>
 * - ALWAYS has non-null DATA.<br>
 * - (Maybe) unsafe :)<br>
 * WARNING : Handle with care and use at own risk :)<br>
 *
 * @short A simple string class implementation
 * @author Szymon Stefanek
 */
class KviStr
{
public:
	// No particular reason for these two names...
	// It is just because I like it :)
	enum KviFormatConstructorTag { Format, Sprintf };

	/**
	 * Empty string == ""<br>
	 * len = 0;<br>
	 * 1 byte allocated
	 */
	KviStr();

	/**
	 * Deep copy of the string<br>
	 * If str is 0 an empty (ptr == "", len = 0) string is created.<br>
	 * If str is non zero a deep copy is created. (len = strlen(str))<br>
	 * Note that str MUST be null terminated in this case.<br>
	 * len+1 bytes allocated
	 */
	KviStr(const char *str);

	/**
	 * Deep copy of len bytes of str.<br>
	 * Note that str can NOT be 0 here,
	 * and must be also at least len characters long.<br>
	 * len+1 bytes allocated
	 */
	KviStr(const char *str, int len);

	/**
	 * Special constructor.<br>
	 * bg and end are pointers to a SINGLE string.<br>
	 * A string is extracted starting from bg and ending at end (not included).<br>
	 * Example:<br>
	 * str=" this is a sample string !";<br>
	 * bg = str + 11; end = str + 24;<br>
	 * KviStr s(bg, end);<br>
	 * now s == "sample string"<br>
	 * len is calculated as (int) (end-bg)<br>
	 * len+1 bytes allocated
	 */
	KviStr(const char *bg, const char *end);

	/**
	 * Format constructor.<br>
	 * tag is... yes... a dummy number used to resolve ambiguities.<br>
	 * It is SAFE: will fail only if we run out of memory,<br>
	 * but can handle only %s %d %u and %c.
	 */
	KviStr(KviFormatConstructorTag tag, const char *fmt, ...);

	/**
	 * Carbon copy :)
	 */
	KviStr(const KviStr &str);

	/**
	 * Compat with Qt...<br>
	 * WARNING : With Qt2.0 it WILL loose UNICODE data.<br>
	 * Safe even if the QString is null.
	 */
	KviStr(const QString &str);

	/**
	 * Fill constructor.<br>
	 * Creates a string long fillLen characters filled with character c.<br>
	 * Avoid passing '\0' as 'c' :)
	 */
	KviStr(char c, int fillLen = 1);

	/**
	 * Just free(m_ptr);
	 */
	~KviStr();

private:
	char *m_ptr; // Pointer to allocated buffer
	int   m_len; // String data length not including the terminator

public:
	/**
	 * Free access to data<br>
	 * (inline)
	 */
	char *ptr() const;

	/**
	 * Direct len, does not use strlen.<br>
	 * Simply fast. :)<br>
	 * (inline)
	 */
	int len() const;

public:
	/**
	 * Simple (len == 0)<br>
	 * (inline)
	 */
	bool isEmpty() const;

	/**
	 * Simple (len != 0)<br>
	 * Why not?<br>
	 * (inline)
	 */
	bool hasData() const;

	bool hasNonWhiteSpaceData() const;

	/**
	 * Safe sprintf.<br>
	 * This one will never write past the end of the string.<br>
	 * It can handle only %s %d %u and %c format flags.<br>
	 */
	KviStr &sprintf(const char *fmt, ...);

	/**
	 * Appends the string str to this string.
	 * Just a memmove and realloc.
	 */
	void append(const KviStr &str);

	/**
	 * Appends the string str to this string.
	 */
	void append(const QString &str);

	/**
	 * Appends the char c to this string.
	 */
	void append(char c);

	/**
	 * Appends the string str to this string unless str is 0.
	 * str can be null here.
	 */
	void append(const char *str);

	/**
	 * Appends exactly len characters from str to this string.<br>
	 * str can NOT be 0, and MUST be at least len characters long.
	 */
	void append(const char *str, int len);
	void append(KviFormatConstructorTag dummy, const char *fmt, ...);

	/**
	 * Prepends the string str to this string.
	 * Two memmoves and a realloc.
	 */
	void prepend(const KviStr &str);

	/**
	 * Prepends the string str to this string unless str is 0.
	 * str can be null here.
	 */
	void prepend(const char *str);

	/**
	 * Prepends exactly len characters from str to this string.<br>
	 * str can NOT be 0, and MUST be at least len characters long.
	 */
	void prepend(const char *str, int len);

	/**
	 * Changes the characters in this string to uppercase.
	 */
	void toUpper();

	/**
	 * Changes the characters in this string to lowercase.
	 */
	void toLower();

	/**
	 * Returns an uppercase copy.<br>
	 * 'this' string is not changed.
	 * Copy and then to upper.
	 */
	KviStr upper() const;

	/**
	 * Returns a lowercase copy.<br>
	 * 'this' string is not changed.
	 * Copy and then to lower.
	 */
	KviStr lower() const;

	/**
	 * Returns true if the last character of the string is ch.
	 */
	bool  lastCharIs(char ch) const;
	bool firstCharIs(char ch) const;

	void bufferToHex(char *buffer, int len);
	int hexToBuffer(char **buffer, bool bNullToNewlines = false);

	/**
	 * If lastCharIs ch does nothing otherwise appends it
	 */
	void ensureLastCharIs(char ch);

	KviStr  left(int maxLen);              // Safe, but maxLen must be >= 0!
	KviStr right(int maxLen);              // Safe, but maxLen must be >= 0!
	KviStr middle(int idx, int maxLen);

	KviStr &cutLeft(int len);              // Removes the first len characters
	KviStr &cutRight(int len);             // Removes the last len characters
	KviStr &cut(int idx, int len);
	/**
	 * Cuts the left part of the string up to the first character c
	 * or does nothing if the char c is not in the string.
	 */
	KviStr   &cutToFirst(char c, bool bIncluded = true);
	KviStr    &cutToLast(char c, bool bIncluded = true);
	KviStr &cutFromFirst(char c, bool bIncluded = true);

	KviStr &insert(int idx, const char *data);
	KviStr &insert(int idx, char c);

	/**
	 * Replaces all occurrences of char c with the string str.
	 */
	KviStr &replaceAll(char c, const char *str);

	/**
	 * Replaces all occurrences of string toFind with the string str.
	 */
	KviStr &replaceAll(char *toFind, const char *str, bool bCaseS = true);

	KviStr &setLen(int len); // Null terminator NOT included!!!

	KviStr &setStr(const char *str, int len = -1); // Safe; len can be anything

	char &at(int idx) const;

	/**
	 * Strips whitespace characters from beginning of this string.
	 */
	KviStr &stripLeftWhiteSpace();

	/**
	 * Strips initial and final WHITESPACE characters (see man isspace),<br>
	 * and returns a reference to this string.
	 */
	KviStr &stripWhiteSpace();

	/**
	 * Strips spaces and tabs only.
	 */
	KviStr &stripSpace();

	/**
	 * Strips all occurrences of the character c from the beginning of the string.<br>
	 * Note that c cannot be '\0' :)
	 */
	KviStr &stripLeft(char c);

	/**
	 * Extracts (copy to str and remove) a token from this string,<br>
	 * and returns true if there are more tokens to extract<br>
	 * Does not strip initial separators!!<br>
	 * str can NOT be this string.
	 */
	bool getToken(KviStr &str, char sep);

	/**
	 * Does not strip initial separators!<br>
	 * Can assign to this string as well.
	 * getToken(' ') => "string    with tokens" -> "string" + "with tokens"
	 */
	KviStr getToken(char sep);

	/**
	 * Extracts a line from the string.<br>
	 * Returns false if there was no data to extract
	 */
	bool getLine(KviStr &str);

	/**
	 * Retuns true if the string contains only digits and an optional '-' character
	 * at the beginning.<br>
	 * Space characters are allowed at the beginning and end.<br>
	 * There is no overflow check!
	 */
	bool isNum();

	/**
	 * Retuns true if the string contains only digits.<br>
	 * Space characters are allowed at the beginning and end.<br>
	 * There is no overflow check!
	 */
	bool isUnsignedNum();

	/**
	 * Returns the number of occurrences of string str in this string.<br>
	 * Overlapped matches are counted.<br><br>
	 * KviStr str("1231231");<br>
	 * str.contains("1231") == 2
	 */
	int contains(const char *str, bool caseS = true);

	/**
	 * Returns the number of occurrences of character c in this string
	 */
	int contains(char c, bool caseS = true);

	/**
	 * Finds the first occurrence of the character c in this string,<br>
	 * and returns its zero-based index or -1 if c cannot be found.<br>
	 * c can NOT be '\0' here.
	 */
	int findFirstIdx(char c);

	/**
	 * Finds the first occurrence of the sub-string str in this string,<br>
	 * and returns its zero-based index or -1 if the sub-string cannot be found.<br>
	 * str can NOT be 0 here.
	 */
	int findFirstIdx(const char *str, bool caseS = true);

	/**
	 * Finds the last occurrence of the character c in this string,<br>
	 * and returns its zero-based index or -1 if the character cannot be found.
	 */
	int findLastIdx(char c);

	/**
	 * Finds the last occurrence of the sub-string str in this string,<br>
	 * and returns its zero-based index or -1 if the sub-string cannot be found.<br>
	 * str can NOT be 0 here.
	 */
	int findLastIdx(const char *str, bool caseS = true);

	/**
	 * Like the special constructor that gets the same args.
	 */
	void extractFromString(const char *begin, const char *end);

	// Everything in base 10
	          char   toChar(bool *bOk = 0) const; // No overflow checks here
	unsigned  char  toUChar(bool *bOk = 0) const; //
	           int    toInt(bool *bOk = 0) const; //
	unsigned   int   toUInt(bool *bOk = 0) const; //
	         short  toShort(bool *bOk = 0) const; //
	unsigned short toUShort(bool *bOk = 0) const; //
	          long   toLong(bool *bOk = 0) const; //
	unsigned  long  toULong(bool *bOk = 0) const; //

	KviStr &setNum(long num);           //
	KviStr &setNum(unsigned long num);  //
	KviStr &setNum(int num);            //
	KviStr &setNum(unsigned int num);   //
	KviStr &setNum(short num);          //
	KviStr &setNum(unsigned short num); //
	KviStr &setNum(char num);           //
	KviStr &setNum(unsigned char num);  //

	// Special functions for multiple bases
	long toLongExt(bool *bOk = 0, int base = 0);

	KviStr &operator=(const KviStr &str);  // Deep copy
	KviStr &operator=(const char *str);    // str can be null here
	KviStr &operator=(char c);             // 2 bytes allocated, m_len = 1
	KviStr &operator=(const QString &str);
	KviStr &operator+=(const KviStr &str); // See related append()
	KviStr &operator+=(const char *str);   // See related append()
	KviStr &operator+=(char c);            // See related append()
	KviStr &operator+=(const QString &str);

	friend inline bool  operator==(const KviStr &, const KviStr &);
	friend inline bool  operator==(const KviStr &, const char *);
	friend inline bool  operator==(const char *,   const KviStr &);

	friend inline bool  operator!=(const KviStr &, const KviStr &);
	friend inline bool  operator!=(const KviStr &, const char *);
	friend inline bool  operator!=(const char *,   const KviStr &);

	friend inline KviStr operator+(const KviStr &, const KviStr &);
	friend inline KviStr operator+(const KviStr &, const char *);
	friend inline KviStr operator+(const char *,   const KviStr &);
	friend inline KviStr operator+(const KviStr &, char);
	friend inline KviStr operator+(char,           const KviStr &);

	friend const char *kvi_extractToken(QString &str, const char *aux_ptr, char sep);
	friend const char *kvi_extractToken(KviStr  &str, const char *aux_ptr, char sep);
	friend const char *kvi_extractUpTo (KviStr  &str, const char *aux_ptr, char sep);
};

// ===========================================================================
//
// Inline stuff...
//
// ===========================================================================

inline char *KviStr::ptr() const                        { return m_ptr; }
inline int KviStr::len() const                          { return m_len; }
inline char &KviStr::at(int idx) const                  { return ((m_len > idx) ? m_ptr[idx] : m_ptr[0]); };

inline bool KviStr::isEmpty() const                     { return (m_len == 0); }
inline bool KviStr::hasData() const                     { return (m_len != 0); }

inline bool KviStr::firstCharIs(char c) const           { return (*m_ptr == c); }
inline bool KviStr::lastCharIs(char c) const            { return (m_len > 0) ? (*(m_ptr + m_len - 1) == c) : false; }
inline void KviStr::ensureLastCharIs(char c)            { if( !lastCharIs(c)) append(c); }

inline KviStr &KviStr::operator+=(const KviStr &str)    { append(str); return (*this); }
inline KviStr &KviStr::operator+=(const char *str)      { append(str); return (*this); }
inline KviStr &KviStr::operator+=(const QString &str)   { append(str); return (*this); }
inline KviStr &KviStr::operator+=(char c)               { append(c);   return (*this); }

inline KviStr &KviStr::setNum(char num)                 { return setNum((long) num); }
inline KviStr &KviStr::setNum(unsigned char num)        { return setNum((unsigned long) num); }
inline KviStr &KviStr::setNum(short num)                { return setNum((long) num); }
inline KviStr &KviStr::setNum(unsigned short num)       { return setNum((unsigned long) num); }
inline KviStr &KviStr::setNum(int num)                  { return setNum((long) num); }
inline KviStr &KviStr::setNum(unsigned int num)         { return setNum((unsigned long) num); }

inline           char KviStr::toChar  (bool *bOk) const { return (char) toLong(bOk); }
inline unsigned  char KviStr::toUChar (bool *bOk) const { return (unsigned char) toULong(bOk); }
inline            int KviStr::toInt   (bool *bOk) const { return (int) toLong(bOk); }
inline unsigned   int KviStr::toUInt  (bool *bOk) const { return (unsigned int) toULong(bOk); }
inline          short KviStr::toShort (bool *bOk) const { return (short) toLong(bOk); }
inline unsigned short KviStr::toUShort(bool *bOk) const { return (unsigned short) toULong(bOk); }

/**
 * Cool string parsing function.<br>
 * It will extract the first found token from the string aux_ptr, and return<br>
 * a pointer to the beginning of the next token, or end of the string.<br>
 * It skips the initial sep characters!<br>
 * Example of use:<br>
 * <br>
 *     KviStr str("String full of tokens");<br>
 *     char *ptr = str.data();<br>
 *     KviStr word;<br>
 *     while( *ptr ) {<br>
 *     &nbsp;&nbsp;&nbsp;&nbsp;ptr = kvi_extractToken(word, ptr);<br>
 *     &nbsp;&nbsp;&nbsp;&nbsp;messWithTheToken(word);<br>
 *     }<br>
 * <br>
 * Another example...
 *     KviStr word;
 *     char *ptr_to_World = kvi_extractToken(word, "Hello World");
 *     // Now word is equal to "Hello"
 */
__kvi_extern const char *kvi_extractToken(QString &str, const char *aux_ptr, char sep = ' ');
__kvi_extern const char *kvi_extractToken(KviStr  &str, const char *aux_ptr, char sep = ' ');

/**
 * Does not skip the beginning separators!<br>
 * Extracts data from the string up to the next separator character or the end of the string.
 * and returns a pointer to that separator (or string end).
 */
__kvi_extern const char *kvi_extractUpTo(KviStr &str, const char *aux_ptr, char sep = ' ');

/**
 * Reduced vsnprintf...
 * Handles %s, %c, %d, %u  (%% are TWO percents here and not one.)
 * Returns -1 if the formatted string exceeded the buffer length.
 * Otherwise returns the length of the formatted buffer... not including '\0'
 */
__kvi_extern int kvi_vsnprintf(char *buffer, int len, const char *fmt, va_list list);

/**
 * Reduced vsnprintf: special version for IRC.
 * Handles %s, %c, %d, %u  (%% are TWO percents here and not one.)
 * Writes up to 510 characters and terminates the string with a CRLF
 * Sets bTruncated if the requested format string was too large to fit in 512 bytes
 * otherwise sets it to false; the buffer MUST be at least 512 bytes long.
 * Always returns the length of the formatted buffer... max 512 - min 2 = CRLF
 */
__kvi_extern int kvi_irc_vsnprintf(char *buffer, const char *fmt, va_list list, bool *bTruncated);

#ifdef COMPILE_i386_ASM_CODE
	#include "kvi_strasm.h" // Include inlined assembly implementations
#else
	/**
	 * Returns true if the string str1 is equal to str2.
	 * Case sensitive.
	 */
	__kvi_extern bool kvi_strEqualCS(const char *str1, const char *str2);

	/**
	 * Returns true if the first len characters of string str1 are equal to str2.
	 * Case sensitive.
	 * Note that if str1 or str2 are shorter than len characters then are considered as NOT equal!
	 */
	__kvi_extern bool kvi_strEqualCSN(const char *str1, const char *str2, int len);

	#define kvi_strEqualNoLocaleCI(str1,str2)      kvi_strEqualCI(str1,str2)
	#define kvi_strEqualNoLocaleCIN(str1,str2,len) kvi_strEqualCIN(str1,str2,len)
	#define kvi_strLen(str)                        strlen(str)
#endif

/**
 * Returns true if the string str1 is equal to str2.
 * Case insensitive.
 */
__kvi_extern bool kvi_strEqualCI(const char *str1, const char *str2);

/**
 * Returns true if the first len characters of string str1 are equal to str2.
 * Case insensitive.
 * Note that if str1 or str2 are shorter than len characters then they are considered as NOT equal!
 */
__kvi_extern bool kvi_strEqualCIN(const char *str1, const char *str2, int len);

/**
 * My own implementation of strcmp and strncasecmp.
 * Once I write it, I KNOW what it does: ALWAYS :)
 * Note that greater here means that it comes AFTER in alphabetic order.
 */
__kvi_extern int kvi_strcmpCI(const char *str1, const char *str2);
__kvi_extern int kvi_strcmpCS(const char *str1, const char *str2);

/**
 * Returns true if the two regular expressions with wildcards match.
 */
__kvi_extern bool kvi_matchWildExpr(register const char *m1, register const char *m2);

/**
 * Returns true if the two regular expressions with wildcards match, case sensitive.
 */
__kvi_extern bool kvi_matchWildExprCS(register const char *m1, register const char *m2);

/**
 * This function work like a particular case of strncmp.
 * It evaluates whether str2 is the terminal part of str1.
 * Example: if str1 is "this is an experiment" and str2 is "xperiment"
 * return 0.
 * With the index parameter, the match starts on str1 from the specified
 * index. For example:
 * If str1 is "this is an experiment" and str2 is "an" we have return != 0
 * but "this is an experiment"
 *      012345678901234567890
 *               ^
 * if we call kvi_strsubRevCS("this is an experiment", "an", 9)
 * we have a match.
 */
__kvi_extern int kvi_strMatchRevCS(const char *str1, const char *str2, int index = -1);

// ===========================================================================
//
// KviStr comparison non-member operators
//
// ===========================================================================
__kvi_extern inline bool operator==(const KviStr &left, const KviStr &right)
	{ return kvi_strEqualCS(left.m_ptr, right.m_ptr); }
__kvi_extern inline bool operator==(const KviStr &left, const char *right)
	{ return kvi_strEqualCS(left.m_ptr, right); }
__kvi_extern inline bool operator==(const char *left, const KviStr &right)
	{ return kvi_strEqualCS(left, right.m_ptr); }

__kvi_extern inline bool operator!=(const KviStr &left, const KviStr &right)
	{ return !kvi_strEqualCS(left.m_ptr, right.m_ptr); }
__kvi_extern inline bool operator!=(const KviStr &left, const char *right)
	{ return !kvi_strEqualCS(left.m_ptr, right); }
__kvi_extern inline bool operator!=(const char *left, const KviStr &right)
	{ return !kvi_strEqualCS(left, right.m_ptr); }

__kvi_extern inline KviStr operator+(const KviStr &left, const KviStr &right)
	{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(const KviStr &left, const char *right)
	{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(const char *left, const KviStr &right)
	{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(const KviStr &left, char right)
	{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(char left, const KviStr &right)
	{ KviStr ret(left); ret += right; return ret; }

#endif // _KVI_STRING_H_INCLUDED_
