// =============================================================================
//
//      --- kvi_imagelibrary_qt.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviImageLibrary"

#include <qbitmap.h>

#include "kvi_debug.h"
#include "kvi_fileutils.h"
#include "kvi_imagelibrary_qt.h"
#include "kvi_locale.h"

KviImageLibrary::KviImageLibrary(const char *path1, const char *path2, int imageWidth, int imageHeight)
{
	__range_valid(path1);
	__range_valid(path2);
	m_count    = -1;
	m_pLibrary =  0;
	if( kvi_fileExists(path1) ) {
		loadLibrary(path1);
		if( !m_pLibrary ) loadLibrary(path2);
	} else loadLibrary(path2);
	setImageSize(imageWidth, imageHeight);
}

KviImageLibrary::~KviImageLibrary()
{
	unloadLibrary();
}

void KviImageLibrary::unloadLibrary()
{
	if( m_pLibrary ) {
		delete m_pLibrary;
		m_pLibrary = 0;
	}
	m_count = 0;
}

void KviImageLibrary::setImageSize(int imageWidth, int imageHeight)
{
	__range_valid(imageWidth  > 0);
	__range_valid(imageHeight > 0);
	m_iWidth  = ((imageWidth  > 0) ? imageWidth  : 16);
	m_iHeight = ((imageHeight > 0) ? imageHeight : 16);
}

bool KviImageLibrary::loadLibrary(const char *path)
{
	__range_valid(path);
	unloadLibrary();
	m_count = -1;
	m_pLibrary = new QPixmap(path);
	if( m_pLibrary->isNull() ) {
		delete m_pLibrary;
		m_pLibrary = 0;
		m_count    = 0;
		debug(__tr("WARNING: cannot load image library %s"), path);
	}
	return (m_pLibrary != 0);
}

int KviImageLibrary::imageCount()
{
	if( !m_pLibrary ) return 0;
	if( m_count != -1 )
		return m_count;
	if( (m_iWidth < 1) || (m_iHeight < 1) ) return 0;
	int nRows = m_pLibrary->width() / m_iWidth;
	return (nRows * (m_pLibrary->height() / m_iHeight));
}

bool KviImageLibrary::libraryLoaded()
{
	return (m_pLibrary != 0);
}

QPixmap KviImageLibrary::getImage(int zeroBasedIndex)
{
	__range_valid(zeroBasedIndex >= 0);
	__range_valid(zeroBasedIndex < imageCount());

	QPixmap image(m_iWidth, m_iHeight, m_pLibrary->depth());
	if( (zeroBasedIndex >= imageCount() ) || (zeroBasedIndex < 0) ) {
		image.fill(); // White fill
		return image;
	}

	// imPerRow is not zero... because imageCount returned non zero.
	int imPerRow = (m_pLibrary->width() / m_iWidth);
	int xOffset  = (zeroBasedIndex % imPerRow) * m_iWidth;
	int yOffset  = (zeroBasedIndex / imPerRow) * m_iHeight;
	// Copy the image data
	bitBlt(&image, 0, 0, m_pLibrary, xOffset, yOffset, m_iWidth, m_iHeight, Qt::CopyROP);
	if( m_pLibrary->mask() ) {
		// Copy the mask too
		QBitmap bmp(m_iWidth, m_iHeight);
		bitBlt(&bmp, 0, 0, m_pLibrary->mask(), xOffset, yOffset, m_iWidth, m_iHeight, Qt::CopyROP);
		image.setMask(bmp);
	}
	return image;
}
