// =============================================================================
//
//      --- kvi_dcc_chat.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccChat"

#include <qsplitter.h>

#include "kvi_dcc_chat.h"
#include "kvi_dcc_event.h"
#include "kvi_dcc_manager.h"
#include "kvi_defines.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_input.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_view.h"
#include "kvi_locale.h"
#include "kvi_mutex.h"
#include "kvi_netutils.h"
#include "kvi_options.h"
#include "kvi_userparser.h"
#include "kvi_userpopupmenu.h"

// Declared in kvi_app.cpp and managed by KviApp class:
extern KviUserPopupMenu *g_pDccChatPopup;
extern QPixmap          *g_pixViewOut[KVI_OUT_NUM_IMAGES];

extern KviEventManager *g_pEventManager;

// TODO: Override saveProperties()

/**
 * ============ KviDccChat ============
 */
KviDccChat::KviDccChat(KviFrame *frame, const char *name)
	: KviWindow(name, KVI_WND_TYPE_CHAT, frame)
{
	m_pSplitter  = new QSplitter(QSplitter::Horizontal, this);
	m_pSplitter->setOpaqueResize();
	m_pView      = new KviIrcView(m_pSplitter, frame, this);
	m_pInput     = new KviInput(this, frame->m_pUserParser);
	connect(m_pView, SIGNAL(contextPopupRequested(KviIrcView *)), this, SLOT(viewRightClicked(KviIrcView *)));

	m_thread = 0;

	m_pDccChatData = new KviDccChatData();
	m_pDccChatData->parent = this;
	m_pDccChatData->outQueue = new QPtrQueue<KviStr>;
	m_pDccChatData->outQueue->setAutoDelete(true);
	m_pDccChatData->outMutex = new KviMutex();

	m_szLocalNick = m_pFrm->m_global.szCurrentNick;
	m_szLocalMask = m_pFrm->m_global.szCurrentMaskFromServer;
	if( m_szLocalNick.isEmpty() ) m_szLocalNick = __tr("me");
	if( m_szLocalMask.isEmpty() ) m_szLocalMask = __tr("me!me@localhost");
}

/**
 * ============ ~KviDccChat ============
 */
KviDccChat::~KviDccChat()
{
	abortThread();

	if( m_pDccChatData ) {
		if( m_pDccChatData->outMutex ) {
			delete m_pDccChatData->outMutex;
			m_pDccChatData->outMutex = 0;
		}
		if( m_pDccChatData->outQueue ) {
			delete m_pDccChatData->outQueue;
			m_pDccChatData->outQueue = 0;
		}
		delete m_pDccChatData;
		m_pDccChatData = 0;
	}
}

void KviDccChat::abortThread()
{
	if( m_thread ) {
		m_thread->stop();
		m_thread->wait();
		delete m_thread;
		m_thread = 0;
	}
}
void KviDccChat::closeEvent(QCloseEvent *e)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnDccChatTerminated) && m_szRemoteInfo.hasData() )
		m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatTerminated, this, m_szRemoteInfo);
	KviWindow::closeEvent(e);
}

void KviDccChat::viewRightClicked(KviIrcView *)
{
	g_pDccChatPopup->doPopup(this, SLOT(viewPopupClicked(const KviStr &)));
}

void KviDccChat::viewPopupClicked(const KviStr &dataBuffer)
{
	m_pFrm->m_pUserParser->parseCommand(dataBuffer.ptr(), this);
}

/**
 * ================ myIconPtr =================
 */
QPixmap *KviDccChat::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_CHAT];
}

/**
 * =============== applyOptions ================
 */
void KviDccChat::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages, false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
}

bool KviDccChat::event(QEvent *e)
{
	if( e->type() != QEvent::User )
		return KviWindow::event(e);

	KviDccEvent *ev = (KviDccEvent *) e;
	switch( ev->m_type ) {
		case KVI_DCC_EVENT_ERROR:
			outputNoFmt(KVI_OUT_DCCERROR, ev->m_dataString.ptr());
			abortThread();
			// The thread is dead now... m_pDccChatData is safe.
			// m_szRemoteIpAddress.hasData() is true if we were connected!
			if( g_pEventManager->eventEnabled(KviEvent_OnDccChatTerminated) && m_szRemoteInfo.hasData() ) {
				KviStr tmp(KviStr::Format, "%s %s", m_szRemoteInfo.ptr(), ev->m_dataString.ptr());
				if( m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatTerminated, this, tmp) )
					return true;
			}
			output(KVI_OUT_DCCINFO,
				__tr("Connection terminated (%s:%d)"),
				m_pDccChatData->szAddress.ptr(), m_pDccChatData->uPort
			);
			break;
		case KVI_DCC_EVENT_MSG:
			outputNoFmt(KVI_OUT_DCCINFO, ev->m_dataString.ptr());
			break;
		case KVI_DCC_EVENT_DATA:
			if( g_pEventManager->eventEnabled(KviEvent_OnDccChatMessage) ) {
				KviStr tmp(KviStr::Format, "%s %s", m_szRemoteInfo.ptr(), ev->m_dataString.ptr());
				if( m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatMessage, this, tmp) )
					return true;
			}
			if( *(ev->m_dataString.ptr()) == 0x01 ) {
				// CTCP? Only ACTION supported
				const char *aux = ev->m_dataString.ptr();
				aux++;
				if( kvi_strEqualCIN(aux, "ACTION", 6) ) {
					while( *aux ) aux++;
					aux--;
					if( *aux == 0x01 )
						ev->m_dataString.cutRight(1);
					ev->m_dataString.cutLeft(7);
					ev->m_dataString.stripWhiteSpace();
					output(KVI_OUT_ACTION, "%s %s", m_szRemoteNick.ptr(), ev->m_dataString.ptr());
					return true;
				}
			}
			m_pFrm->outputPrivmsg(this, KVI_OUT_NONE, m_szRemoteNick.ptr(), m_szRemoteMask.ptr(), ev->m_dataString.ptr());
			break;
		case KVI_DCC_EVENT_CONNECTIONESTABLISHED:
			m_szRemoteInfo = ev->m_dataString.ptr();
			if( g_pEventManager->eventEnabled(KviEvent_OnDccChatConnected) ) {
				m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatConnected, this, m_szRemoteInfo);
			}
			// Following toggles beeping from default of false to true.
			if( m_pView->initBeeping(m_szRemoteNick.ptr()) )
				m_pView->toggleBeeping();
			break;
		case KVI_DCC_EVENT_LISTENING:
			m_pFrm->m_pSocket->sendData(ev->m_dataString.ptr(), ev->m_dataString.len());
			outputNoFmt(KVI_OUT_DCCINFO, __tr("Sent DCC chat request, waiting for reply..."));
			break;
		default:
			return KviWindow::event(e);
			break;
	}
	return true;
}

bool KviDccChat::sendData(const char *buffer)
{
	if( !m_thread )
		return false; // No connection at all
	// Otherwise the socket is connected,
	// or is waiting for connection.
	// If it is waiting, just store the text in the queue.
	// It will be all sent when a connection is established
	KviStr *tmp = new KviStr(buffer);
	// Need to lock the mutex while appending...
	m_pDccChatData->outMutex->lock();
	m_pDccChatData->outQueue->enqueue(tmp);
	m_pDccChatData->outMutex->unlock();
	return true;
}

/**
 * ================ resizeEvent ===============
 */
void KviDccChat::resizeEvent(QResizeEvent *)
{
	int inputSize = m_pInput->heightHint();
	m_pSplitter->setGeometry(0, 0, width(), height() - inputSize);
	m_pInput->setGeometry(0, height() - inputSize, width(), inputSize);
}

void KviDccChat::acceptDccRequest(
	const char *nick, const char *username, const char *host, unsigned long uAddress, unsigned short uPort)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s", nick, username, host);
	// Called from KviDccManager
	// A user sent us a CTCP DCC, and we accepted it.
	// Connect to the originating host
	if( m_thread ) {
		debug("WARNING: acceptDccRequest was called twice!");
		return;
	}
	m_pDccChatData->nick      = nick;
	m_pDccChatData->username  = username;
	m_pDccChatData->host      = host;
	m_pDccChatData->uAddress  = uAddress;
	m_pDccChatData->uPort     = uPort;
	struct in_addr addr;
	addr.s_addr = uAddress;
	kvi_binaryIpToString(addr, m_pDccChatData->szAddress);

	m_thread = new KviDccChatAcceptThread(m_pDccChatData);
	if( m_thread )
		m_thread->start();
	else
		outputNoFmt(KVI_OUT_DCCERROR, __tr("Cannot create slave thread. DCC chat failed"));
}

void KviDccChat::requestDcc(const char *nick, const char *username, const char *host)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s", nick, username, host);
	// Called from KviDccManager
	// Request a DCC connection to a user
	if( m_thread ) {
		debug("WARNING: requestDcc was called twice!");
		return;
	}
	m_pDccChatData->nick      = nick;
	m_pDccChatData->username  = username;
	m_pDccChatData->host      = host;
	m_pDccChatData->uAddress  = m_pFrm->m_pSocket->getSockAddress();
	m_pDccChatData->uPortToListenOn = m_pFrm->m_pDccManager->getDccSendListenPort();

	if( g_pOptions->m_bUseUserDefinedIpForDccRequests ) {
		struct in_addr addr;
		if( kvi_stringIpToBinaryIp(g_pOptions->m_szDccLocalIpAddress.ptr(), &addr) ) {
			output(KVI_OUT_DCCINFO, __tr("Using %s as local host address"), g_pOptions->m_szDccLocalIpAddress.ptr());
			m_pDccChatData->uAddress = addr.s_addr;
		} else {
			output(
				KVI_OUT_DCCERROR, __tr("IP %s is invalid: using standard IP address"),
				g_pOptions->m_szDccLocalIpAddress.ptr()
			);
			g_pOptions->m_bUseUserDefinedIpForDccRequests = false;
		}
	}

	if( m_pDccChatData->uAddress == 0 ) {
		outputNoFmt(KVI_OUT_DCCERROR, __tr("Cannot resolve local host. DCC chat failed"));
		return;
	}
	m_pDccChatData->szAddress = __tr("unknown");
	m_pDccChatData->uPort     = 0;

	// Create the slave thread...
	m_thread = new KviDccChatRequestThread(m_pDccChatData);
	if( m_thread )
		m_thread->start();
	else
		outputNoFmt(KVI_OUT_DCCERROR, __tr("Cannot create slave thread. DCC chat failed"));
}

#include "m_kvi_dcc_chat.moc"
