// =============================================================================
//
//      --- kvi_dcc_send.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccSend"

#include <unistd.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qpixmap.h>
#include <qsplitter.h>

#include "kvi_chanlabel.h"
#include "kvi_console.h"
#include "kvi_dcc_event.h"
#include "kvi_dcc_manager.h"
#include "kvi_dcc_send.h"
#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_view.h"
#include "kvi_locale.h"
#include "kvi_mutex.h"
#include "kvi_netutils.h"
#include "kvi_options.h"
#include "kvi_progress.h"
#include "kvi_userparser.h"

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap         *g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern KviEventManager *g_pEventManager;

/**
 * ============ KviDccSend ============
 */
KviDccSend::KviDccSend(KviFrame *lpFrm, const char *name)
	: KviWindow(name, KVI_WND_TYPE_SEND, lpFrm)
{
	m_pSplitter    = new QSplitter(QSplitter::Horizontal, this);
	m_pSplitter->setOpaqueResize();
	m_pView        = new KviIrcView(m_pSplitter, lpFrm, this);
	m_pProgress    = new KviProgress(100, this);
	m_pStatLabel   = new KviChanLabel(__tr("No connection"), this);
	m_pFileLabel   = new KviChanLabel(__tr("File: none"), this);
	m_pInput       = 0;
	m_thread       = 0;
	m_pDccSendData = 0;

	m_pProgress->setFrameStyle(QFrame::Panel | QFrame::Sunken);

	applyLabelsOptions();
	setFileProgress(0);
}

/**
 * ============ ~KviDccSend ============
 */
KviDccSend::~KviDccSend()
{
	abortThread();
	destroyDccSendData();
}

/**
 * Remove the child thread if it is still here
 */
void KviDccSend::abortThread()
{
	if( !m_thread )
		return;

	m_thread->stop();
	m_thread->wait();
	delete m_thread;
	m_thread = 0;
}

void KviDccSend::destroyDccSendData()
{
	if( !m_pDccSendData )
		return;

	if( m_pDccSendData->resumeMutex ) {
		delete m_pDccSendData->resumeMutex;
		m_pDccSendData->resumeMutex = 0;
	}
	if( m_pDccSendData->file ) {
		m_pDccSendData->file->close();
		delete m_pDccSendData->file;
		m_pDccSendData->file = 0;
	}
	delete m_pDccSendData;
	m_pDccSendData = 0;
}

void KviDccSend::setFileProgress(int i)
{
	setProgress(i);
	m_pProgress->setProgress(i);
}

/**
 * ================ myIconPtr =================
 */
QPixmap *KviDccSend::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_SEND];
}

/**
 * =============== applyOptions ================
 */
void KviDccSend::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages, false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
	applyLabelsOptions();
}

/**
 * =============== applyOptions ================
 */
void KviDccSend::applyLabelsOptions()
{
	if( !g_pOptions->m_pixLabelsBack->isNull() ) {
		m_pFileLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
		m_pStatLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
	} else {
		m_pFileLabel->setBackColor(g_pOptions->m_clrLabelsBack);
		m_pStatLabel->setBackColor(g_pOptions->m_clrLabelsBack);
	}
	m_pFileLabel->setTextColor(g_pOptions->m_clrLabelsActiveFore);
	m_pStatLabel->setTextColor(g_pOptions->m_clrLabelsActiveFore);
	m_pFileLabel->setFont(g_pOptions->m_fntLabels);
	m_pStatLabel->setFont(g_pOptions->m_fntLabels);
}

bool KviDccSend::event(QEvent *e)
{
	if( e->type() != QEvent::User )
		return KviWindow::event(e);

	KviDccEvent *ev = (KviDccEvent *) e;
	switch( ev->m_type ) {
		case KVI_DCC_EVENT_ERROR:
			outputNoFmt(KVI_OUT_DCCERROR, ev->m_dataString.ptr());
			abortThread();
			// The thread is dead now... m_pDccSendData is safe.
			output(KVI_OUT_DCCINFO,
				__tr("Connection terminated (%s:%d)"), m_pDccSendData->szAddress.ptr(), m_pDccSendData->uPort
			);
			destroyDccSendData();
			break;
		case KVI_DCC_EVENT_MSG:
			outputNoFmt(KVI_OUT_DCCINFO, ev->m_dataString.ptr());
			break;
		case KVI_DCC_EVENT_DCCSENDSTATUS:
			m_pStatLabel->setText(ev->m_dataString.ptr());
			setFileProgress(ev->m_param);
			break;
		case KVI_DCC_EVENT_FINISHED:
			outputNoFmt(KVI_OUT_INTERNAL, ev->m_dataString.ptr());
			abortThread();
			// The thread is dead now... m_pDccSendData is safe.
			output(KVI_OUT_DCCINFO,
				__tr("Connection terminated (%s:%d)"), m_pDccSendData->szAddress.ptr(), m_pDccSendData->uPort
			);
			if( g_pOptions->m_bNotifyDccSendCompletionInConsole ) {
				m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
					__tr("DCC file transfer with %s completed [%s: %u bytes]"),
					m_pDccSendData->nick.ptr(), m_pDccSendData->originalFileName.ptr(), m_pDccSendData->fileLength
				);
			}
			if( g_pEventManager->eventEnabled(
				m_pDccSendData->bGetSession ? KviEvent_OnDccGetTransferComplete : KviEvent_OnDccSendTransferComplete)
			) {
				KviStr tmp(KviStr::Format, "%s %s %s %s %s %u",
					m_pDccSendData->nick.ptr(),
					m_pDccSendData->username.ptr(),
					m_pDccSendData->host.ptr(),
					m_pDccSendData->szAddress.ptr(),
					m_pDccSendData->originalFileName.ptr(),
					m_pDccSendData->fileLength
				);
				m_pFrm->m_pUserParser->callEvent(
					m_pDccSendData->bGetSession ? KviEvent_OnDccGetTransferComplete : KviEvent_OnDccSendTransferComplete,
					this, tmp
				);
			}
			destroyDccSendData();
			if( g_pOptions->m_bAutoCloseDccSendOnSuccess )
				close(); // TODO: is this safe?
			break;
		case KVI_DCC_EVENT_LISTENING:
			m_pFrm->m_pSocket->sendData(ev->m_dataString.ptr(), ev->m_dataString.len());
			outputNoFmt(KVI_OUT_DCCINFO, __tr("Sent DCC send request, waiting for reply..."));
			break;
		default:
			return KviWindow::event(e);
		break;
	}
	return true;
}

/**
 * ================ resizeEvent ===============
 */
void KviDccSend::resizeEvent(QResizeEvent *)
{
	int hght = m_pFileLabel->sizeHint().height();
	m_pFileLabel->setGeometry(0, 0, width(), hght);
	m_pProgress->setGeometry(0, hght, width(), hght);
	m_pStatLabel->setGeometry(0, hght << 1, width(), hght);
	m_pSplitter->setGeometry(0, hght * 3, width(), height() - (hght * 3));
}

void KviDccSend::acceptDccSendRequest(KviDccSendRequestData *data)
{
	QFile *f = new QFile(data->filePath.ptr());
	if( data->resumeValue != 0 ) {
		if( !f->open(IO_WriteOnly | IO_Append) ) {
			delete f;
			output(KVI_OUT_DCCERROR,
				__tr("Cannot start a RESUME session: the file %s does not exist"),
				data->filePath.ptr()
			);
			return;
		}
	} else {
		if( !f->open(IO_WriteOnly | IO_Truncate) ) {
			delete f;
			output(KVI_OUT_DCCERROR,
				__tr("Cannot open file %s for writing. DCC get failed"),
				data->filePath.ptr()
			);
			return;
		}
	}
	m_pDccSendData = new KviDccSendData();
	m_pDccSendData->parent           = this;
	m_pDccSendData->nick             = data->nick;
	m_pDccSendData->username         = data->username;
	m_pDccSendData->host             = data->host;
	m_pDccSendData->filePath         = data->filePath;
	m_pDccSendData->fileLength       = data->fileLength;
	m_pDccSendData->uAddress         = data->uAddress;
	m_pDccSendData->uPort            = data->uPort;
	m_pDccSendData->receivedBytes    = data->resumeValue;
	m_pDccSendData->file             = f;
	m_pDccSendData->originalFileName = data->originalFileName;
	m_pDccSendData->bListening       = false;
	m_pDccSendData->bGetSession      = true;
	m_pDccSendData->bSendZeroAck     = g_pOptions->m_bDccGetSendZeroAck;
	m_pDccSendData->resumeMutex      = new KviMutex();

	KviStr tmp(KviStr::Format,
		__tr("File: %s [%u bytes]"), m_pDccSendData->filePath.ptr(), m_pDccSendData->fileLength
	);
	m_pFileLabel->setText(tmp.ptr());

	if( m_pDccSendData->receivedBytes != 0 ) {
		// Need to resume
		m_pFrm->m_pSocket->sendFmtData(
			"PRIVMSG %s :%cDCC RESUME %s %u %u%c",
			m_pDccSendData->nick.ptr(), 0x01,
			m_pDccSendData->originalFileName.ptr(),
			m_pDccSendData->uPort,
			m_pDccSendData->receivedBytes, 0x01
		);
		output(KVI_OUT_DCCINFO,
			__tr("Sent DCC RESUME request to %s [%s: %u of %u bytes], waiting for reply..."),
			m_pDccSendData->nick.ptr(), m_pDccSendData->originalFileName.ptr(),
			m_pDccSendData->receivedBytes, m_pDccSendData->fileLength
		);
	} else
		initiateGet();
}

void KviDccSend::initiateGet()
{
	// This is called either by a direct DCC Get request
	// or a resume request from KviDccManager
	__range_valid(m_pDccSendData);

	if( m_thread ) {
		debug("WARNING: initiateGet was called twice!");
		return;
	}

	struct in_addr addr;
	addr.s_addr = m_pDccSendData->uAddress;
	kvi_binaryIpToString(addr, m_pDccSendData->szAddress);
	// Create the slave thread...

	m_thread = new KviDccSendAcceptThread(m_pDccSendData);
	if( m_thread )
		m_thread->start();
	else
		outputNoFmt(KVI_OUT_DCCERROR, __tr("Cannot create slave thread. DCC get failed"));
}

void KviDccSend::requestDccSend(KviDccSendRequestData *data)
{
	QFile *f = new QFile(data->filePath.ptr());
	if( !f->open(IO_ReadOnly) ) {
		delete f;
		output(KVI_OUT_DCCERROR, __tr("Cannot open file %s for reading. DCC send failed"), data->filePath.ptr());
		return;
	}
	QFileInfo fi(*f);

	m_pDccSendData = new KviDccSendData();
	m_pDccSendData->parent             = this;
	m_pDccSendData->nick               = data->nick;
	m_pDccSendData->username           = data->username;
	m_pDccSendData->host               = data->host;
	m_pDccSendData->filePath           = data->filePath;
	m_pDccSendData->fileLength         = fi.size();
	m_pDccSendData->uPort              = 0;
	m_pDccSendData->sentBytes          = 0;
	m_pDccSendData->realSentBytesCount = 0;
	m_pDccSendData->file               = f;
	m_pDccSendData->originalFileName   = fi.fileName();
	m_pDccSendData->bListening         = false;
	m_pDccSendData->bFastSend          = g_pOptions->m_bUseFastDccSend;
	m_pDccSendData->ackedBytes         = 0;
	m_pDccSendData->stalledCount       = 0;
	m_pDccSendData->blockSize          = g_pOptions->m_iDccSendBlockSize;
	m_pDccSendData->bGetSession        = false;
	m_pDccSendData->uPortToListenOn    = m_pFrm->m_pDccManager->getDccSendListenPort();
	m_pDccSendData->resumeMutex        = new KviMutex();

	if( m_pDccSendData->blockSize > KVI_DCC_SEND_MAX_BLOCK_SIZE ) {
		m_pDccSendData->blockSize = KVI_DCC_SEND_MAX_BLOCK_SIZE;
		g_pOptions->m_iDccSendBlockSize = KVI_DCC_SEND_MAX_BLOCK_SIZE;
	}

	if( m_pDccSendData->blockSize <= 0 ) {
		g_pOptions->m_iDccSendBlockSize = 512;
		m_pDccSendData->blockSize       = 512;
	}
	if( g_pOptions->m_bReplaceSpacesInDccSendFileNames )
		m_pDccSendData->originalFileName.replaceAll(' ', "_"); // Replace all spaces with underscores
	m_pDccSendData->uAddress = m_pFrm->m_pSocket->getSockAddress();

	if( g_pOptions->m_bUseUserDefinedIpForDccRequests ) {
		struct in_addr addr;
		if( kvi_stringIpToBinaryIp(g_pOptions->m_szDccLocalIpAddress.ptr(), &addr) ) {
			output(KVI_OUT_DCCINFO, __tr("Using %s as local host address"), g_pOptions->m_szDccLocalIpAddress.ptr());
			m_pDccSendData->uAddress = addr.s_addr;
		} else {
			output(KVI_OUT_DCCERROR,
				__tr("IP address %s is invalid: using default IP address"), g_pOptions->m_szDccLocalIpAddress.ptr()
			);
			g_pOptions->m_bUseUserDefinedIpForDccRequests = false;
		}
	}

	if( m_pDccSendData->uAddress == 0 ) {
		outputNoFmt(KVI_OUT_DCCERROR, __tr("Cannot resolve local host. DCC send failed"));
		return;
	}

	m_pDccSendData->szAddress = __tr("unknown");

	KviStr tmp(KviStr::Format,
		__tr("File: %s [%u bytes]"),
		m_pDccSendData->filePath.ptr(), m_pDccSendData->fileLength
	);
	m_pFileLabel->setText(tmp.ptr());

	// Now start listening
	m_thread = new KviDccSendRequestThread(m_pDccSendData);
	if( m_thread )
		m_thread->start();
	else
		outputNoFmt(KVI_OUT_DCCERROR, __tr("Cannot create slave thread. DCC send failed"));
}

bool KviDccSend::youAreListening(unsigned short uPort, const char *nick)
{
	if( !m_thread       ) return false;
	if( !m_pDccSendData ) return false;
	bool bOk = false;
	m_pDccSendData->resumeMutex->lock();
	if( m_pDccSendData->bListening ) { // This is protected by the mutex
		if( uPort == m_pDccSendData->uPort )  {
			bOk = kvi_strEqualCI(nick, m_pDccSendData->nick.ptr());
		}
	}
	m_pDccSendData->resumeMutex->unlock();
	return bOk;
}

bool KviDccSend::resumeForCurrentSend(unsigned long uResumePos)
{
	if( !m_thread       ) return false;
	if( !m_pDccSendData ) return false;
	bool bOk = false;
	m_pDccSendData->resumeMutex->lock();
	if( m_pDccSendData->bListening ) { // This is protected by the mutex
		if( uResumePos <= m_pDccSendData->fileLength ) {
			bOk = m_pDccSendData->file->at(uResumePos);
			if( bOk ) {
				output(KVI_OUT_DCCINFO, __tr("Acknowledged RESUME. Transfer will initiate from position %u"), uResumePos);
				m_pDccSendData->sentBytes = uResumePos;
			}
		}
	}
	m_pDccSendData->resumeMutex->unlock();
	return bOk;
}

#include "m_kvi_dcc_send.moc"
