// =============================================================================
//
//      --- kvi_dcc_send_thread.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccSendThread"

#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include <qfile.h>

#include "kvi_application.h"
#include "kvi_dcc_event.h"
#include "kvi_dcc_send.h"
#include "kvi_dcc_send_thread.h"
#include "kvi_defines.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_mutex.h"
#include "kvi_netutils.h"

// Time to sleep if no data has to be read or sent
#define KVI_DCC_IDLE_TIME_USECS 200

// TODO: Make optional in order to stop flashing in the DCC send progress?
#define KVI_DCC_FAILED_WRITES_LIMIT_FOR_STALLED 10

KviDccSendThread::KviDccSendThread(KviDccSendData *data)
	: KviDccThread()
{
	m_pData = data;
}

KviDccSendThread::~KviDccSendThread()
{
	// Nothing here
}

////////////////////////////////////////////////////////////////////////////////

KviDccSendAcceptThread::KviDccSendAcceptThread(KviDccSendData *data)
	: KviDccSendThread(data)
{
	// Nothing here
}

KviDccSendAcceptThread::~KviDccSendAcceptThread()
{
	// Nothing here
}

/**
 * Specific routine for accepting DCC Send requests
 * from users on IRC.
 * A user sent us a CTCP DCC with a port and an address.
 * Now we try to contact that host and initiate a DCC Send...
 */
void KviDccSendAcceptThread::run()
{
	// Prepare the target data
	struct in_addr inAddress;
	inAddress.s_addr = m_pData->uAddress;
	struct sockaddr_in hostSockAddr;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(m_pData->uPort);
	hostSockAddr.sin_addr   = inAddress;

	// Let's go...
	m_socket = socket(PF_INET, SOCK_STREAM, 0);

	if( m_socket < 0 ) {
		abort(__tr("Unable to create a stream socket. DCC get failed"), m_pData->parent);
		return;
	}

	if( fcntl(m_socket, F_SETFL, O_NONBLOCK) < 0 ) {
		abort(__tr("Unable to create a non-blocking stream socket. DCC get failed"), m_pData->parent);
		return;
	}

	if( m_bStopRequested ) return;

	// Ok... now try to connect
	int iError = connect((struct sockaddr *) (&hostSockAddr), sizeof(hostSockAddr));

	if( iError != KVI_ERROR_Success ) {
		m_pData->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"), kvi_getErrorString(iError));
		abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
		return;
	}

	// Now wait for connection...
	KviStr tmp(KviStr::Format, __tr("Connecting to %s on port %u"), m_pData->szAddress.ptr(), m_pData->uPort);
	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_MSG, tmp.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	iError = waitForOutgoingConnection();

	if( iError != KVI_ERROR_Success ) {
		m_pData->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"), kvi_getErrorString(iError));
		abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
		return;
	}

	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, __tr("Connected: receiving file"));
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->startTime = QTime::currentTime();
	m_pData->realBytesCount = 0;

	getPostStatusEvent();
	eventLoop();
}

void KviDccSendAcceptThread::eventLoop()
{
	if( m_pData->bSendZeroAck )
		sendAck(); // Send a zero ack... some clients need that...

	for( ; !m_bStopRequested; ) {
		if( readData() ) {
			sendAck();
		}
	}
}

void KviDccSendAcceptThread::sendAck()
{
	unsigned int toAckBytes = htonl(m_pData->receivedBytes);
	write(m_socket, &toAckBytes, sizeof(unsigned int));
	if( m_pData->receivedBytes >= m_pData->fileLength ) {
		KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_FINISHED, __tr("File transfer completed"));
		KviApplication::postEvent(m_pData->parent, ev);
		stop();
	}
}

/**
 * Check if there is data available on sock,
 * read it, split into messages and post data events
 * to the parent DCC window class.
 */
bool KviDccSendAcceptThread::readData()
{
	if( !selectForRead(m_socket) ) {
		usleep(KVI_DCC_IDLE_TIME_USECS);
		return false;
	} // else: there is data to read...

	if( m_bStopRequested ) return false;

	// Read data
	char buffer[1024];
	int readLength = read(m_socket, buffer, 1024); // read() is not atomic

	if( readLength <= 0 ) {
		// Oops?
		if( readLength == 0 ) {
			// Plain close. No problems.
			// Post the close event to the parent.
			// It is a KVI_DCC_EVENT_ERROR just because I need make the parent close the connection
			abort(__tr("Remote end has closed the connection"), m_pData->parent);
			return false;
		} else {
			// Error?
			if( (errno == EINTR) || (errno == EAGAIN) )
				return false;
			// Yes... error :(
			int iError;
			switch( errno ) {
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error; pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// Post the error event to the parent and exit
			m_pData->tmpBuffer.sprintf(__tr("READ ERROR: %s"), kvi_getErrorString(iError));
			abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
			return false;
		}
	}
	// Successfully read.
	m_pData->receivedBytes  += readLength;
	m_pData->realBytesCount += readLength;
	int writtenBytes = m_pData->file->writeBlock(buffer, readLength);
	if( writtenBytes < readLength ) {
		abort(__tr("Unable to write to file. DCC get failed"), m_pData->parent);
		return false;
	}

	getPostStatusEvent();
	return true;
}

void KviDccSendAcceptThread::getPostStatusEvent()
{
	// Some calculations.
	// TODO: If the user starts the DCC before midnight and current time is after midnight we will have
	// negative elapsedSecs here.
	int elapsedSecs = m_pData->startTime.secsTo(QTime::currentTime());
	if( elapsedSecs >= 0 ) {
		int speed = (elapsedSecs ? (m_pData->realBytesCount / elapsedSecs) : 0);
		int remainingSecs = m_pData->fileLength - m_pData->receivedBytes; // Size

		if( speed > 0 ) {
			remainingSecs = remainingSecs / speed;
			m_pData->tmpBuffer.sprintf(
				__tr("Received %u bytes in %d min %d sec (%d bytes/sec): %d min %d sec remaining"),
				m_pData->realBytesCount, elapsedSecs / 60, elapsedSecs % 60, speed, remainingSecs / 60, remainingSecs % 60
			);
		} else {
			m_pData->tmpBuffer.sprintf(
				__tr("Received %u bytes in %d min %d sec (%d bytes/sec): ? min ? sec remaining"),
				m_pData->realBytesCount, elapsedSecs / 60, elapsedSecs % 60, speed
			);
		}
	} else m_pData->tmpBuffer = __tr("Yaaawn... midnight passed. I am too tired to display more info...");

	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_DCCSENDSTATUS, m_pData->tmpBuffer.ptr());
	if( m_pData->fileLength > 0 )
		ev->m_param = (100 * m_pData->receivedBytes) / m_pData->fileLength;
	else
		ev->m_param = 100;

	KviApplication::postEvent(m_pData->parent, ev);
}

////////////////////////////////////////////////////////////////////////////////

KviDccSendRequestThread::KviDccSendRequestThread(KviDccSendData *data)
	: KviDccSendThread(data)
{
	// Nothing here
}

KviDccSendRequestThread::~KviDccSendRequestThread()
{
	// Nothing here
}

void KviDccSendRequestThread::run()
{
	KviDccEvent *ev = 0;
	int newsock = -1;

	struct sockaddr_in sockAddress;

	sockAddress.sin_family      = AF_INET;
	sockAddress.sin_port        = htons(m_pData->uPortToListenOn);
	sockAddress.sin_addr.s_addr = INADDR_ANY;

	// Let's go...
	m_socket = socket(PF_INET, SOCK_STREAM, 0);

	if( m_socket < 0 ) {
		abort(__tr("Unable to create a listening socket. DCC send failed"), m_pData->parent);
		return;
	}

	if( m_bStopRequested ) return;

	if( (bind(m_socket, (struct sockaddr *) &sockAddress, sizeof(sockAddress)) < 0) || (listen(m_socket, 100) < 0) ) {
		abort(__tr("Unable to set up a listening socket. DCC send failed"), m_pData->parent);
		return;
	}

	socklen_t iSize = sizeof(sockAddress);
	getsockname(m_socket, (struct sockaddr *) &sockAddress, &iSize);

	m_pData->uPort = ntohs(sockAddress.sin_port);

	m_pData->tmpBuffer.sprintf(__tr("Listening on port %u"), m_pData->uPort);
	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->tmpBuffer.sprintf(
		"PRIVMSG %s :%cDCC SEND %s %u %u %u%c",
		m_pData->nick.ptr(), 0x01, m_pData->originalFileName.ptr(), ntohl(m_pData->uAddress),
		m_pData->uPort, m_pData->fileLength, 0x01
	);

	ev = new KviDccEvent(KVI_DCC_EVENT_LISTENING, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	// Here we could also receive a RESUME message
	m_pData->resumeMutex->lock();
	m_pData->bListening = true;
	m_pData->resumeMutex->unlock();

	newsock = waitForIncomingConnection(&(m_pData->uAddress), &(m_pData->uPort), &(m_pData->szAddress));

	m_pData->resumeMutex->lock();
	m_pData->bListening = false;
	m_pData->resumeMutex->unlock();

	// If we have received a resume here, the file is already at the right position
	// and sentBytes is already updated.
	close(m_socket); // Close the old socket...
	m_socket = newsock;

	m_pData->tmpBuffer.sprintf(__tr("Connected to %s on port %u"), m_pData->szAddress.ptr(), m_pData->uPort);
	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->startTime = QTime::currentTime();
	eventLoop();
}

void KviDccSendRequestThread::eventLoop()
{
	for( ; !m_bStopRequested; ) {
		writeData();
		receiveAck();
	}
}

void KviDccSendRequestThread::writeData()
{
	char buffer[KVI_DCC_SEND_MAX_BLOCK_SIZE];

	if( ((m_pData->bFastSend) || (m_pData->realSentBytesCount == 0) || (m_pData->ackedBytes >= m_pData->sentBytes)) &&
		 (m_pData->sentBytes < m_pData->fileLength)
	) {
		fd_set out_fd_set;
		struct timeval tv;
		// First check for write ability
		FD_ZERO(&out_fd_set);
		FD_SET(m_socket, &out_fd_set);
		tv.tv_sec  = 0;
		tv.tv_usec = KVI_DCC_IDLE_TIME_USECS;

		if( m_bStopRequested ) return;

		int retval = select(m_socket + 1, 0, &out_fd_set, 0, &tv);
		if( retval != 1 ) {
			// TODO: maybe check for select() errors here if retval == -1?
			m_pData->stalledCount++;
			if( m_pData->stalledCount == KVI_DCC_FAILED_WRITES_LIMIT_FOR_STALLED )
				sendNotifyStalled();
			usleep(KVI_DCC_IDLE_TIME_USECS); // Relax
			return;
		} // else can write

		m_pData->stalledCount = 0;

		int toWriteBytes = m_pData->fileLength - m_pData->sentBytes;
		if( toWriteBytes > m_pData->blockSize )
			toWriteBytes = m_pData->blockSize;
		toWriteBytes = m_pData->file->readBlock(buffer, toWriteBytes);

		int wroteBytes = write(m_socket, buffer, toWriteBytes);

		if( wroteBytes < toWriteBytes ) {
			// Oops
			if( wroteBytes == -1 ) {
				// Error
				if( (errno != EINTR ) && (errno != EAGAIN) ) {
					int iError;
					switch( errno ) {
						case EBADF:        iError = KVI_ERROR_BadFileDescriptor;     break;
						case EFAULT:       iError = KVI_ERROR_OutOfAddressSpace;     break;
						case EINVAL:       iError = KVI_ERROR_BadFileDescriptor;     break;
						case ENOSPC:       iError = KVI_ERROR_BrokenPipe;            break;
						case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
						case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
						// Unhandled error; pass errno to the strerror function
						default:           iError = -errno;                          break;
					}
					m_pData->tmpBuffer.sprintf(__tr("WRITE ERROR: %s"), kvi_getErrorString(iError));
					abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
					return;
				} else m_pData->file->at(m_pData->file->at() - toWriteBytes);
			} else {
				// Seek back in the file
				int goBack = toWriteBytes - wroteBytes;
				m_pData->file->at(m_pData->file->at() - goBack);
			}
		}

		if( wroteBytes > 0 ) {
			m_pData->realSentBytesCount += wroteBytes;
			m_pData->sentBytes          += wroteBytes;
		}
		sendPostStatusEvent();
	}
	usleep(KVI_DCC_IDLE_TIME_USECS); // Relax... always..
}

/**
 * Check if there is data available on m_socket,
 * read it, split into messages and post data events
 * to the parent DCC window class.
 */
void KviDccSendRequestThread::receiveAck()
{
	if( !selectForRead(m_socket) ) {
		usleep(KVI_DCC_IDLE_TIME_USECS);
		return;
	} // else there is data to read

	if( m_bStopRequested ) return;

	// Read data
	unsigned int ack;
	int readLength = read(m_socket, &ack, sizeof(unsigned int)); // read() is not atomic

	if( readLength < (int) sizeof(unsigned int) ) {
		// Oops?
		if( readLength == 0 ) {
			abort(__tr("Remote end has closed the connection"), m_pData->parent);
			return;
		} else if( readLength < 0 ) {
			// Error?
			if( (errno == EINTR) || (errno == EAGAIN) )
				return;
			// Yes... error :(
			int iError;
			switch( errno ) {
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error; pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// Post the error event to the parent and exit
			m_pData->tmpBuffer.sprintf(__tr("READ ERROR: %s"), kvi_getErrorString(iError));
			abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
			return;
		}
		abort(__tr("Unrecoverable read error: fragmented ack from peer"), m_pData->parent);
		return;
	}

	// Successfully read.
	m_pData->ackedBytes = ntohl(ack);
	if( m_pData->ackedBytes >= m_pData->fileLength ) {
		KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_FINISHED, __tr("File transfer completed"));
		KviApplication::postEvent(m_pData->parent, ev);
	}
	sendPostStatusEvent();
}

/**
 * Send is stalled - send notification
 */
void KviDccSendRequestThread::sendNotifyStalled()
{
	int elapsedSecs = m_pData->startTime.secsTo(QTime::currentTime());
	if( elapsedSecs >= 0 ) {
		int speed = (elapsedSecs ? (m_pData->realSentBytesCount / elapsedSecs) : 0);
		m_pData->tmpBuffer.sprintf(
			__tr("Sent %u bytes in %d min %d sec (%d bytes/sec): received ack for %d bytes (stalled)"),
			m_pData->realSentBytesCount, elapsedSecs / 60, elapsedSecs % 60, speed, m_pData->ackedBytes
		);
	} else m_pData->tmpBuffer = __tr("Stalled: since it is past midnight, maybe the remote side fell asleep?");

	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_DCCSENDSTATUS, m_pData->tmpBuffer.ptr());
	if( m_pData->fileLength > 0 )
		ev->m_param = (100 * m_pData->ackedBytes) / m_pData->fileLength;
	else
		ev->m_param = 100;
	KviApplication::postEvent(m_pData->parent, ev);
}

void KviDccSendRequestThread::sendPostStatusEvent()
{
	int elapsedSecs = m_pData->startTime.secsTo(QTime::currentTime());
	if( elapsedSecs >= 0 ) {
		int speed = (elapsedSecs ? (m_pData->realSentBytesCount / elapsedSecs) : 0);
		m_pData->tmpBuffer.sprintf(
			__tr("Sent %u bytes in %d min %d sec (%d bytes/sec): received ack for %d bytes"),
			m_pData->realSentBytesCount, elapsedSecs / 60, elapsedSecs % 60, speed, m_pData->ackedBytes
		);
	} else {
		m_pData->tmpBuffer = __tr("Yaaawn... midnight passed: we are closed. Info will be displayed again tomorrow morning.");
	}

	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_DCCSENDSTATUS, m_pData->tmpBuffer.ptr());
	if( m_pData->fileLength > 0 )
		ev->m_param = (100 * m_pData->ackedBytes) / m_pData->fileLength;
	else
		ev->m_param = 100;

	KviApplication::postEvent(m_pData->parent, ev);
}
