// =============================================================================
//
//      --- kvi_dirview.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDirView"

#include <qfileinfo.h>

#include "kvi_app.h"
#include "kvi_defines.h"
#include "kvi_dirview.h"
#include "kvi_dirview_item.h"
#include "kvi_fileutils.h"
#include "kvi_frame.h"
#include "kvi_iconloader.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"

static const QString g_iMultipleFiles("kmultiple.png");

KviDirectoryView::KviDirectoryView(QWidget *parent, KviFrame *lpFrm)
	: KviIconView(parent)
{
	m_pFrm = lpFrm;
	setGridX(70);
	setSelectionMode(QIconView::Extended);
	setAutoArrange(true);
	setWordWrapIconText(true);
	setResizeMode(QIconView::Adjust);
	setAcceptDrops(true);
	viewport()->setAcceptDrops(true);
	parent->setAcceptDrops(true);
	setItemTextPos(QIconView::Bottom);

	setSorting(true);
	m_szDirPath = "/";
	connect(
		this, SIGNAL(dropped(QDropEvent *, const QValueList<QIconDragItem> &)),
		this, SLOT(itemDropped(QDropEvent *, const QValueList<QIconDragItem> &))
	);
	connect(this, SIGNAL(moved()), this, SLOT(emitNeedUpdate()));
	connect(
		this, SIGNAL(rightButtonPressed(QIconViewItem *, const QPoint &)),
		this, SLOT(showContextMenu(QIconViewItem *, const QPoint &))
	);
	connect(
		this, SIGNAL(itemRenamed(QIconViewItem *, const QString &)),
		this, SLOT(dirItemRenamed(QIconViewItem *, const QString &))
	);
	m_pContextPopup = new KviPopupMenu();
}

void KviDirectoryView::applyOptions()
{
	emit needUpdate();
}

KviDirectoryView::~KviDirectoryView()
{
	delete m_pContextPopup;
	m_pContextPopup = 0;
}

void KviDirectoryView::dirItemRenamed(QIconViewItem *i, const QString &text)
{
	KviStr tmp = text;
	KviDirectoryViewItem *it = (KviDirectoryViewItem *) i;
	if( tmp.findFirstIdx('/') != -1 ) {
		g_pApp->warningBox(
			__tr("Unable to rename file %s.\nPlease use drag-and-drop to move\nthe file across directories."),
			it->m_data->szFileName.ptr()
		);
		it->setText(it->m_data->szFileName.ptr());
		return;
	}
	QFileInfo fi(text);
	if( fi.exists() ) {
		g_pApp->warningBox(
			__tr("Unable to rename file %s.\nThe file %s already exists."),
			it->m_data->szFileName.ptr(), tmp.ptr()
		);
		it->setText(it->m_data->szFileName.ptr());
	} else {
		KviStr newPath = it->m_data->szDirPath;
		newPath.ensureLastCharIs('/');
		newPath.append(text);
		if( !kvi_renameFile(it->m_data->szAbsFilePath.ptr(), newPath.ptr()) ) {
			g_pApp->warningBox(
				__tr("Unable to rename file %s.\nThe destination may already exist."),
				it->m_data->szFileName.ptr()
			);
			// Failed
			it->setText(it->m_data->szFileName.ptr());
		} else {
			it->m_data->szAbsFilePath = newPath;
			it->m_data->szFileName    = text;
		}
	}
}

QDragObject *KviDirectoryView::dragObject()
{
	if( !currentItem() ) return 0;

	QStringList fnamelist;
	KviDirectoryViewItem *item;
	for( item = (KviDirectoryViewItem *) firstItem(); item; item = (KviDirectoryViewItem *) item->nextItem() ) {
		if( item->isSelected() )
			fnamelist.append(_CHAR_2_QSTRING(item->m_data->szAbsFilePath.ptr()));
	}

	QUriDrag *drag = new QUriDrag(viewport());
	drag->setFilenames(fnamelist);

	if( fnamelist.count() == 1 ) {
		drag->setPixmap(
			QPixmap(*(currentItem()->pixmap())),
			QPoint(currentItem()->pixmapRect().width() >> 1, currentItem()->pixmapRect().height() >> 1 )
		);
	} else {
		drag->setPixmap(
			KviIconLoader::loadMimeIcon(g_iMultipleFiles),
			QPoint(currentItem()->pixmapRect().width() >> 1, currentItem()->pixmapRect().height() >> 1)
		);
	}

	return drag;
}

void KviDirectoryView::itemDropped(QDropEvent *e, const QValueList<QIconDragItem> &lst)
{
	QStringList list;
	bool bNeedReload = false;
	if( QUriDrag::decodeLocalFiles(e, list) ) {
		if( !list.isEmpty() ) {
			QStringList::ConstIterator it = list.begin();
			for( ; it != list.end(); ++it ) {
				KviStr filename = *it;
				KviStr newPath  = filename;
				newPath.cutToLast('/', true);
				KviStr dirPath = m_szDirPath;
				dirPath.ensureLastCharIs('/');
				newPath.prepend(dirPath);
				if( !kvi_renameFile(filename.ptr(), newPath.ptr()) ) {
					// Failed
					g_pApp->warningBox(__tr("Unable to move file %s.\nThe destination may already exist."), filename.ptr());
				} else {
					bNeedReload = true;
				}
			}
		}
	}
	if( bNeedReload ) emit needUpdate();
}

void KviDirectoryView::emitNeedUpdate()
{
	emit needUpdate();
}

void KviDirectoryView::showContextMenu(QIconViewItem *i, const QPoint &pos)
{
	m_pContextPopup->clear();
	if( i ) {
		// Some items are selected
		if( g_pOptions->m_szDirBrowserOpenWithEditorCmd.hasData() && (((KviDirectoryViewItem *) i)->m_data->bIsFile) ) {
			m_pContextPopup->insertItem(__tr("&Open with Editor"), this, SLOT(openWithEditorSelectedFiles()));
			m_pContextPopup->insertSeparator();
		}
		m_pContextPopup->insertItem(__tr("Move to &Trash"), this, SLOT(trashSelectedFiles()));
	}
	m_pContextPopup->popup(pos);
}

void KviDirectoryView::trashSelectedFiles()
{
	bool bNeedUpdate = false;
	KviDirectoryViewItem *item;
	for( item = (KviDirectoryViewItem *) firstItem(); item; item = (KviDirectoryViewItem *) item->nextItem() ) {
		if( item->isSelected() ) {
			if( !(item->m_data->bIsDirectory &&
				(kvi_strEqualCI(item->m_data->szFileName.ptr(), "..") || kvi_strEqualCI(item->m_data->szFileName.ptr(), "."))
				)
			) {
				if( !g_pApp->trashFile(item->m_data->szAbsFilePath.ptr()) ) {
					g_pApp->warningBox(__tr("Unable to move file %s to trash.\n"), item->m_data->szAbsFilePath.ptr());
				} else bNeedUpdate = true;
			}
		}
	}
	if( bNeedUpdate ) emit needUpdate();
}

void KviDirectoryView::openWithEditorSelectedFiles()
{
	g_pOptions->m_szDirBrowserOpenWithEditorCmd.stripWhiteSpace();
	KviDirectoryViewItem *item;
	for( item = (KviDirectoryViewItem *) firstItem(); item; item = (KviDirectoryViewItem *) item->nextItem() ) {
		if( item->isSelected() && item->m_data->bIsFile ) {
			g_pApp->executeFileWithCommand(
				g_pOptions->m_szDirBrowserOpenWithEditorCmd.ptr(), item->m_data->szAbsFilePath.ptr(), m_pFrm
			);
		}
	}
}

#include "m_kvi_dirview.moc"
