// =============================================================================
//
//      --- kvi_app_script.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1998-1999 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviAppScript"

#include <qfile.h>

#include "kvi_alias.h"
#include "kvi_app.h"
#include "kvi_config.h"
#include "kvi_event.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_process.h"
#include "kvi_rawevent.h"
#include "kvi_script_wizard.h"
#include "kvi_userpopupmenu.h"
#include "kvi_usertoolbartemplate.h"
#include "kvi_window.h"

extern KviAliasManager    *g_pAliasManager;
extern KviEventManager    *g_pEventManager;
extern KviRawEventManager *g_pRawEventManager;

bool KviApp::saveScript(KviSaveScriptStruct *s)
{
	if( !s->pWnd ) return false;

	KviStr date = QDateTime::currentDateTime().toString();

	s->pWnd->output(KVI_OUT_INTERNAL, __tr("Creating script %s"), s->szName.ptr());

	// Make the subdir
	KviStr tmp;
	getLocalKVIrcDirectory(tmp, Scripts, s->szSubdir.ptr());

	if( kvi_directoryExists(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Directory %s exists: moving to trash..."), tmp.ptr());
		if( !g_pApp->trashFile(tmp.ptr()) ) {
			s->pWnd->output(KVI_OUT_ERROR, __tr("Failed to trash the directory %s: writing inside the existing one."), tmp.ptr());
		}
	}

	if( !kvi_directoryExists(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Creating directory %s"), tmp.ptr());
		if( !kvi_makeDir(tmp.ptr()) ) {
			s->pWnd->output(KVI_OUT_ERROR, __tr("Could not make directory %s"), tmp.ptr());
			return false;
		}
	}
	tmp.ensureLastCharIs('/');

	KviStr buffer(KviStr::Format, "# KVIrc script\n# Automatically generated at %s\n\necho %s %s\necho By ",
		date.ptr(), s->szName.ptr(), s->szVersion.ptr());
	if( s->szAuthor.hasData() )
		buffer.append(s->szAuthor);
	else
		buffer.append("unknown");
	buffer.append("\n\n");

	// Create this directory name
	buffer.append("# Extract this script directory name\n");

	buffer.append("if(\"$strleft(1, $0)\" == \"/\") %thisFilePath = $strlefttolast(/, $0)\n");
	buffer.append("else %thisFilePath = $KVIrcScriptDir/");
	buffer.append(s->szSubdir.ptr());
	buffer.append("\n\n");
	buffer.append("option szLastScriptPath %thisFilePath\n");
	buffer.append("\n\n");

	s->szDescription.append("\n\nDetails:\n\n");

	if( s->bSaveColors            ) script_appendColorSettingsBuffer(buffer, s);
	if( s->bSaveFonts             ) script_appendFontSettingsBuffer(buffer, s);
	if( s->bSaveTextColors        ) script_appendTextColorSettingsBuffer(buffer, s);
	if( s->bSaveBackgrounds       ) script_saveBackgrounds(buffer, s);
	if( s->bSaveMessages          ) script_appendMessageSettingsBuffer(buffer, s);
	if( s->bSaveOutputRedirection ) script_appendOutputRedirectionSettingsBuffer(buffer, s);
	if( s->bSaveCTCPOptions       ) script_appendCTCPOptionsSettingsBuffer(buffer, s);
	if( s->bSaveIrcEngineOptions  ) script_appendIrcEngineOptionsSettingsBuffer(buffer, s);
	if( s->bSaveAliases           ) script_saveAliases(buffer, tmp, s);
	if( s->bSaveEvents            ) script_saveEvents(buffer, tmp, s);
	if( s->bSaveRawEvents         ) script_saveRawEvents(buffer, tmp, s);
	if( s->bSavePopups            ) script_savePopups(buffer, tmp, s);
	if( s->bSaveToolbar           ) script_saveToolbar(buffer, tmp, s);
	s->szEpilogue.stripWhiteSpace();
	if( s->szEpilogue.hasData()   ) buffer.append(s->szEpilogue);
	if( s->pExtraFilesList        ) script_copyExtraFiles(buffer, tmp, s);

	KviStr scriptFile(KviStr::Format, "%smain.kvs", tmp.ptr());
	if( kvi_writeFile(scriptFile.ptr(), buffer) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Saved main script file to %s"), scriptFile.ptr());
	} else {
		s->pWnd->output(KVI_OUT_ERROR, __tr("Could not write file %s"), scriptFile.ptr());
		return false;
	}

	KviStr def(KviStr::Format,
		"[KVIrc_Script_Definition]\nNAME=%s\nVERSION=%s\nGENERATOR=%s\nDATE=%s\nLANGUAGETAG=%s\nAUTHOR=%s\nEMAIL=%s\n",
		s->szName.ptr(), s->szVersion.ptr(), KVI_VERSION, date.ptr(), KVI_SCRIPT_LANGUAGE_TAG,
		s->szAuthor.ptr(), s->szEMail.ptr()
	);

	KviStr defFile(KviStr::Format, "%sscript.def", tmp.ptr());
	if( kvi_writeFile(defFile.ptr(), def) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Saved script definition file to %s"), defFile.ptr());
	} else {
		s->pWnd->output(KVI_OUT_ERROR, __tr("Could not write file %s"), defFile.ptr());
		return false;
	}

	KviStr desFile(KviStr::Format, "%sscript.des", tmp.ptr());
	if( kvi_writeFile(desFile.ptr(), s->szDescription) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Saved script description file to %s"), desFile.ptr());
	} else {
		s->pWnd->output(KVI_OUT_ERROR, __tr("Could not write file %s"), desFile.ptr());
		return false;
	}

	if( s->bCreateTarball ) {
		KviProcess *proc = new KviProcess();
		KviStr tmp2;
		connect(proc, SIGNAL(processExited(KviProcess *)), this, SLOT(processExited(KviProcess *)));
		getLocalKVIrcDirectory(tmp2, Scripts);
		KviStr tarGzFile(KviStr::Format, "%s/%s.tar.gz", tmp2.ptr(), s->szSubdir.ptr());
		if( kvi_fileExists(tarGzFile.ptr()) ) {
			// Trash it first
			s->pWnd->output(KVI_OUT_INTERNAL, __tr("File %s exists: moving to trash..."), tarGzFile.ptr());
			if( !g_pApp->trashFile(tarGzFile.ptr()) ) {
				s->pWnd->output(KVI_OUT_ERROR, __tr("Failed to trash the file %s: running tar & gzip anyway... you may obtain strange results..."), tarGzFile.ptr());
			}
		}
		KviStr tarballCmd(KviStr::Format, "kvi_make_scriptdist.sh %s %s", tmp2.ptr(), s->szSubdir.ptr());
		if( proc->run(tarballCmd.ptr()) ) {
			s->pWnd->output(KVI_OUT_INTERNAL, __tr("Creating tar.gz package %s"), tarGzFile.ptr());
		} else {
			delete proc;
			s->pWnd->output(KVI_OUT_INTERNAL, __tr("Failed to make the tar.gz package: subprocess failed to run"));
		}
	}

	return true;
}

void KviApp::processExited(KviProcess *proc)
{
	if( proc ) delete proc;
}

void KviApp::script_appendFontSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the font options\n");
	buffer.append("echo Setting fonts...\n");
	for( int i = 0; g_pOptions->optFontsTable[i].optName; i++ ) {
		KviStr tmp;
		KviConfig::getFontProperties(tmp, ((QFont *) (g_pOptions->optFontsTable[i].optValue)));
		buffer.append("option ");
		buffer.append(g_pOptions->optFontsTable[i].optName);
		buffer.append(' ');
		buffer.append(tmp);
		buffer.append('\n');
	}
	buffer.append('\n');

	s->szDescription.append("Contains font settings\n");
}

void KviApp::script_appendColorSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the color options\n");
	buffer.append("echo Setting colors...\n");
	for( int i = 0; g_pOptions->optColorsTable[i].optName; i++ ) {
		buffer.append("option ");
		buffer.append(g_pOptions->optColorsTable[i].optName);
		buffer.append(' ');
		if( (g_pOptions->optColorsTable[i].optType & KVI_OPT_TYPE_MASK) == KVI_OPT_CLR ) {
			KviStr tmp(KviStr::Format, "%d,%d,%d",
				((QColor *) (g_pOptions->optColorsTable[i].optValue))->red(),
				((QColor *) (g_pOptions->optColorsTable[i].optValue))->green(),
				((QColor *) (g_pOptions->optColorsTable[i].optValue))->blue());
			buffer.append(tmp);
		} else if( (g_pOptions->optColorsTable[i].optType & KVI_OPT_TYPE_MASK) == KVI_OPT_PCLR ) {
			KviStr tmp(KviStr::Format, "%d,%d,%d",
				(*((QColor **) (g_pOptions->optColorsTable[i].optValue)))->red(),
				(*((QColor **) (g_pOptions->optColorsTable[i].optValue)))->green(),
				(*((QColor **) (g_pOptions->optColorsTable[i].optValue)))->blue());
			buffer.append(tmp);
		}
		buffer.append('\n');
	}
	buffer.append("# Set the boolean values associated with colors\n");
	for( int i = 0; g_pOptions->optBoolTable[i].optName; i++ ) {
		if( (g_pOptions->optBoolTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_GUI ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optBoolTable[i].optName);
			buffer.append(' ');
			KviStr tmp = ((*((bool *) (g_pOptions->optBoolTable[i].optValue))) ? "1" : "0" );
			buffer.append(tmp);
			buffer.append('\n');
		}
	}
	buffer.append("option szApplicationGUIStyle ");
	buffer.append(g_pOptions->m_szApplicationGUIStyle);
	buffer.append("\n\n");
	s->szDescription.append("Contains color settings\n");
}

void KviApp::script_appendTextColorSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the text color values\n");
	buffer.append("echo Setting text colors...\n");
	for( int i = 0; g_pOptions->optTextTable[i].optName; i++ ) {
		buffer.append("option ");
		buffer.append(g_pOptions->optTextTable[i].optName);
		buffer.append(' ');
		KviStr tmp;
		tmp.setNum((*((char *) (g_pOptions->optTextTable[i].optValue))));
		buffer.append(tmp);
		buffer.append('\n');
	}
	buffer.append('\n');

	s->szDescription.append("Contains text color settings\n");
}

void KviApp::script_saveBackground(
	KviStr &buffer, KviStr &szDir, const char *fName, const char *szPixName, QPixmap *pix, KviSaveScriptStruct *s)
{
	if( !pix->isNull() ) {
		KviStr szFile = szDir;
		szFile.append(fName);
		if( pix->save(szFile.ptr(), "PNG") ) {
			buffer.append("option ");
			buffer.append(szPixName);
			buffer.append(" %thisFilePath/");
			buffer.append(fName);
			buffer.append('\n');
			s->pWnd->output(KVI_OUT_INTERNAL, __tr("Wrote file %s"), szFile.ptr());
			s->szDescription.append("Contains reference to external image ");
			s->szDescription.append(fName);
			s->szDescription.append(" used for option ");
			s->szDescription.append(szPixName);
			s->szDescription.append('\n');
			return;
		} else s->pWnd->output(KVI_OUT_INTERNAL, __tr("Could not save background file %s, disabling it..."), szFile.ptr());
	}

	buffer.append("option ");
	buffer.append(szPixName);
	buffer.append('\n');
	s->szDescription.append("Clears the ");
	s->szDescription.append(szPixName);
	s->szDescription.append(" option\n");
}

void KviApp::script_saveBackgrounds(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the backgrounds...\n");
	buffer.append("echo Setting backgrounds...\n");

	s->szDescription.append("Contains background image settings\n");

	KviStr tmp;
	getLocalKVIrcDirectory(tmp, Scripts, s->szSubdir.ptr());
	tmp.ensureLastCharIs('/');

	script_saveBackground(buffer, tmp, "mdi.png", "szMdiBack", g_pOptions->m_pixMdiBack, s);
	script_saveBackground(buffer, tmp, "listbox.png", "szListBoxBack", g_pOptions->m_pixListBoxBack, s);
	script_saveBackground(buffer, tmp, "input.png", "szInputBack", g_pOptions->m_pixInputBack, s);
	script_saveBackground(buffer, tmp, "labels.png", "szLabelsBack", g_pOptions->m_pixLabelsBack, s);
	script_saveBackground(buffer, tmp, "view.png", "szViewBack", g_pOptions->m_pixViewBack, s);

	buffer.append('\n');
}

void KviApp::script_saveAliases(KviStr &buffer, KviStr &path, KviSaveScriptStruct *s)
{
	KviStr tmp = path;
	tmp.append("aliases.conf");
	if( g_pAliasManager->save(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Aliases saved to file %s"), tmp.ptr());

		s->szDescription.append("Contains aliases in file aliases.conf\n");

		buffer.append("# Load aliases...\n");
		buffer.append("echo Loading aliases from %thisFilePath/aliases.conf\n");
		buffer.append("loadconf aliases %thisFilePath/aliases.conf\n\n");
	} else s->pWnd->output(KVI_OUT_ERROR, __tr("Could not save aliases to file %s"), tmp.ptr());
}

void KviApp::script_saveEvents(KviStr &buffer, KviStr &path, KviSaveScriptStruct *s)
{
	KviStr tmp = path;
	tmp.append("events.conf");
	if( g_pEventManager->save(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Event handlers saved to file %s"), tmp.ptr());

		s->szDescription.append("Contains events in file events.conf\n");

		buffer.append("# Load events...\n");
		buffer.append("echo Loading events from %thisFilePath/events.conf\n");
		buffer.append("loadconf events %thisFilePath/events.conf\n\n");
	} else s->pWnd->output(KVI_OUT_ERROR, __tr("Could not save event handlers to file %s"), tmp.ptr());
}

void KviApp::script_saveRawEvents(KviStr &buffer, KviStr &path, KviSaveScriptStruct *s)
{
	KviStr tmp = path;
	tmp.append("rawevents.conf");
	if( g_pRawEventManager->save(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("RAW Event handlers saved to file %s"), tmp.ptr());

		s->szDescription.append("Contains raw events in file rawevents.conf\n");

		buffer.append("# Load raw events...\n");
		buffer.append("echo Loading raw events from %thisFilePath/rawevents.conf\n");
		buffer.append("loadconf rawevents %thisFilePath/rawevents.conf\n\n");
	} else s->pWnd->output(KVI_OUT_ERROR, __tr("Could not save raw event handlers to file %s"), tmp.ptr());
}

void KviApp::script_savePopups(KviStr &buffer, KviStr &path, KviSaveScriptStruct *s)
{
	KviStr tmp = path;
	tmp.append("popups.conf");
	if( KviUserPopupMenu::saveAllPopups(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Popups saved to file %s"), tmp.ptr());

		s->szDescription.append("Contains popups in file popups.conf\n");

		buffer.append("# Load popups...\n");
		buffer.append("echo Loading popups from %thisFilePath/popups.conf\n");
		buffer.append("loadconf popups %thisFilePath/popups.conf\n\n");
	} else s->pWnd->output(KVI_OUT_ERROR, __tr("Could not save popups to file %s"), tmp.ptr());
}

void KviApp::script_saveToolbar(KviStr &buffer, KviStr &path, KviSaveScriptStruct *s)
{
	KviStr tmp = path;
	tmp.append("toolbar.conf");

	if( KviUserToolBarTemplate::saveToolBarTemplate(tmp.ptr()) ) {
		s->pWnd->output(KVI_OUT_INTERNAL, __tr("Toolbar saved to file %s"), tmp.ptr());

		s->szDescription.append("Contains toolbar in file toolbar.conf\n");

		buffer.append("# Load toolbar...\n");
		buffer.append("echo Loading toolbar from %thisFilePath/toolbar.conf\n");
		buffer.append("loadconf toolbar %thisFilePath/toolbar.conf\n\n");
	} else s->pWnd->output(KVI_OUT_ERROR, __tr("Could not save toolbar to file %s"), tmp.ptr());
}

void KviApp::script_appendMessageSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the boolean values associated with colors\n");
	buffer.append("echo Setting messages...\n");

	s->szDescription.append("Contains default messages\n");

	for( int i = 0; g_pOptions->optStringsTable[i].optName; i++ ) {
		if( ((g_pOptions->optStringsTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_MSG)
		 || ((g_pOptions->optStringsTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_CAPTION) ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optStringsTable[i].optName);
			buffer.append(" \"");
			KviStr tmp = (*((KviStr *) (g_pOptions->optStringsTable[i].optValue)));
			tmp.replaceAll('\\', "\\\\");
			tmp.replaceAll(';', "\\;");
			tmp.replaceAll('\n', "\\\n");
			tmp.replaceAll('$', "\\$");
			tmp.replaceAll('%', "\\%");
			tmp.replaceAll('\"', "\\\"");
			buffer.append(tmp);
			buffer.append("\"\n");
		}
	}
	buffer.append('\n');
}

void KviApp::script_appendOutputRedirectionSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the output redirection options\n");
	buffer.append("echo Setting output redirection...\n");

	s->szDescription.append("Contains output redirection settings\n");

	for( int i = 0; g_pOptions->optBoolTable[i].optName; i++ ) {
		if( (g_pOptions->optBoolTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_REDIR ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optBoolTable[i].optName);
			buffer.append(' ');
			buffer.append(((*((bool *) (g_pOptions->optBoolTable[i].optValue))) ? '1' : '0'));
			buffer.append('\n');
		}
	}
	buffer.append('\n');
}

void KviApp::script_appendCTCPOptionsSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the CTCP options\n");
	buffer.append("echo Setting CTCP options...\n");

	s->szDescription.append("Contains CTCP options\n");

	for( int i = 0; g_pOptions->optBoolTable[i].optName; i++ ) {
		if( (g_pOptions->optBoolTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_CTCP ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optBoolTable[i].optName);
			buffer.append(' ');
			buffer.append(((*((bool *) (g_pOptions->optBoolTable[i].optValue))) ? '1' : '0'));
			buffer.append('\n');
		}
	}
	for( int i = 0; g_pOptions->optIntTable[i].optName; i++ ) {
		if( (g_pOptions->optIntTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_CTCP ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optIntTable[i].optName);
			buffer.append(' ');
			KviStr tmp(KviStr::Format, "%d", (*((int *) (g_pOptions->optIntTable[i].optValue))));
			buffer.append(tmp);
			buffer.append('\n');
		}
	}
	buffer.append('\n');
}

void KviApp::script_appendIrcEngineOptionsSettingsBuffer(KviStr &buffer, KviSaveScriptStruct *s)
{
	buffer.append("# Set the IRC engine options\n");
	buffer.append("echo Setting IRC engine options...\n");

	s->szDescription.append("Contains IRC engine options\n");

	for( int i = 0; g_pOptions->optBoolTable[i].optName; i++ ) {
		if( (g_pOptions->optBoolTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_IRCENG ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optBoolTable[i].optName);
			buffer.append(' ');
			buffer.append(((*((bool *) (g_pOptions->optBoolTable[i].optValue))) ? '1' : '0'));
			buffer.append('\n');
		}
	}
	for( int i = 0; g_pOptions->optStringsTable[i].optName; i++ ) {
		if( (g_pOptions->optStringsTable[i].optType & KVI_OPT_FLAG_MASK) == KVI_OPT_IRCENG ) {
			buffer.append("option ");
			buffer.append(g_pOptions->optStringsTable[i].optName);
			buffer.append(" \"");
			KviStr tmp = (*((KviStr *) (g_pOptions->optStringsTable[i].optValue)));
			tmp.replaceAll('\\', "\\\\");
			tmp.replaceAll(';', "\\;");
			tmp.replaceAll('\n', "\\\n");
			tmp.replaceAll('$', "\\$");
			tmp.replaceAll('%', "\\%");
			tmp.replaceAll('\"', "\\\"");
			buffer.append(tmp);
			buffer.append("\"\n");
		}
	}

	buffer.append('\n');
}

void KviApp::script_copyExtraFiles(KviStr &buffer, KviStr &path, KviSaveScriptStruct *s)
{
	s->szDescription.append("Contains extra files:\n");

	for( KviStr *str = s->pExtraFilesList->first(); str; str = s->pExtraFilesList->next() ) {
		KviStr target = path;

		int lastSlash = str->findLastIdx('/');
		if( (lastSlash > -1) && ((lastSlash + 1) < str->len()) && (str->len() > 0) ) {
			KviStr fName = str->right(str->len() - (lastSlash + 1));
			target.append(fName);

			QFile f(target.ptr());
			if( f.exists() ) {
				s->pWnd->output(KVI_OUT_INTERNAL, __tr("Destination file %s exists, removing it."), target.ptr());
				if( !f.remove() )
					s->pWnd->output(KVI_OUT_ERROR, __tr("Could not remove file %s"), target.ptr());
			}
			if( kvi_copyFile(str->ptr(), target.ptr()) ) {
				s->szDescription.append("-- ");
				s->szDescription.append(fName);
				s->szDescription.append("\n");
				s->pWnd->output(KVI_OUT_INTERNAL, __tr("Copied file %s to %s"), str->ptr(), target.ptr());
			} else s->pWnd->output(KVI_OUT_ERROR, __tr("Failed to copy file %s to %s"), str->ptr(), target.ptr());
		} else s->pWnd->output(KVI_OUT_ERROR, __tr("Failed to copy file %s: invalid file name or path"), str->ptr());
	}
}
