// =============================================================================
//
//      --- kvi_channel.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviChannel"

#include <qcursor.h>
#include <qfile.h>
#include <qsplitter.h>
#include <qtooltip.h>

#include "kvi_banmask.h"
#include "kvi_chanlabel.h"
#include "kvi_channel.h"
#include "kvi_debug.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_input.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_user.h"
#include "kvi_irc_view.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_maskeditor.h"
#include "kvi_modeeditor.h"
#include "kvi_options.h"
#include "kvi_topiceditor.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"
#include "kvi_userpopupmenu.h"
#include "kvi_winproperties.h"

/**
	@quickhelp: KviChannel
	@widget: Channel window
		This window is used to represent an IRC channel.<br>
		The channel name is displayed in the window's titlebar.<br>
		For scripting purposes, the channel name is also the window name
		reported by the <a href="s_window.kvihelp">$window</a> identifier.<br>
		Just under the channel name you can see three labels showing:
		<b>the channel topic, </b><br>
		If you are a channel operator, you can double-click on this label
		to edit the channel topic.<br>
		<b>current channel mode, </b><br>
		Double-clicking on this label displays the channel mode control center.<br>
		If you are a channel operator, you will be able to change the channel modes.<br>
		<b>user and mode statistics</a></b>
		Right clicking on this label displays a menu that
		lets you open a mask editor.<br>
		You will be allowed to set the bans, ban exceptions and invite exceptions
		for the channel (if the server supports such modes).<br>
		Bu double-clicking on this label, you will open the last accessed mask editor.<br>
		The main part of the window is the <a href="qh_kviircview.kvihelp">channel view</a>
		and on the right of that is the <a href="qh_kviuserlistbox.kvihelp">channel userlist</a>.<br>
		The bottom portion of the window is the <a href="qh_kviinput.kvihelp">text input box</a>.<br>
		To part the channel, press the close button in the upper right corner or use the <a href="part.kvihelp">part</a> command.<br>
*/

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap          *g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern KviUserPopupMenu *g_pChannelPopup;
extern KviUserPopupMenu *g_pUserlistPopup;
extern KviEventManager  *g_pEventManager;

/**
 * ============ KviChannel ============
 */
KviChannel::KviChannel(KviFrame *lpFrm, const char *name)
	: KviWindow(name, KVI_WND_TYPE_CHANNEL, lpFrm)
{
	m_pBanList              = new QPtrList<KviBanMask>;
	m_pBanList->setAutoDelete(true);
	m_iBanCount             = 0;
	m_pExceptionList        = new QPtrList<KviBanMask>;
	m_pExceptionList->setAutoDelete(true);
	m_iExceptionCount       = 0;
	m_pInviteExceptionList  = new QPtrList<KviBanMask>;
	m_pInviteExceptionList->setAutoDelete(true);
	m_iInviteExceptionCount = 0;

	m_pTopicEditor          = 0; // Create it later... when requested
	m_pModeEditor           = 0; // Same as above
	m_pBanEditor            = 0;
	m_pBanExceptEditor      = 0;
	m_pInviteExceptEditor   = 0;

	m_bTopicChangeEnabled   = true;
	m_bModeChangeEnabled    = true;

	m_pMasksEditPopup       = 0;
	m_iLastMasksEdited      = -1;

	m_bOnChannel            = true;
	m_joinTime              = QTime::currentTime();
	m_iChanSyncMask         = KVI_CHANSYNC_NOTINSYNC;

	m_pTopSplitter = new QSplitter(QSplitter::Horizontal, this, "top_chan_splitter");
	m_pTopSplitter->setOpaqueResize();

	m_pTopicLabel  = new KviChanLabel(
		__tr("[ Channel topic unknown (No topic message received) ]"),
		m_pTopSplitter, "topic_label"
	);
	setTopicToolTip();
	connect(m_pTopicLabel, SIGNAL(doubleClicked()), this, SLOT(doEditTopic()));

	m_pModeLabel   = new KviChanLabel("", m_pTopSplitter, "mode_label");
	setModeToolTip();
	connect(m_pModeLabel, SIGNAL(doubleClicked()), this, SLOT(doEditModes()));

	m_pUsersLabel  = new KviChanLabel("0 [o:0 v:0 b:0]", m_pTopSplitter, "users_label");
	connect(m_pUsersLabel, SIGNAL(doubleClicked()), this, SLOT(doEditMasks()));
	connect(m_pUsersLabel, SIGNAL(rightClicked()), this, SLOT(usersLabelRightClicked()));

	m_pSplitter    = new QSplitter(QSplitter::Horizontal, this, "central_chan_splitter");
	m_pSplitter->setOpaqueResize();
	m_pView        = new KviIrcView(m_pSplitter, lpFrm, this);
	m_pListBox     = new KviUserListBox(m_pSplitter, this, lpFrm->m_pUserParser, lpFrm);
	connect(m_pListBox, SIGNAL(rightClicked()), this, SLOT(listBoxRightClicked()));

	m_pInput       = new KviInput(this, lpFrm->m_pUserParser, m_pListBox);
	// Reasonable default for the splitters
	QValueList<int> l;
	l.append(80);
	l.append(20);
	m_pSplitter->setSizes(l);

	QValueList<int> l2;
	l2.append(60);
	l2.append(20);
	l2.append(20);
	m_pTopSplitter->setSizes(l2);

	applyLabelsOptions();

	connect(m_pView, SIGNAL(contextPopupRequested(KviIrcView *)), this, SLOT(viewRightClicked(KviIrcView *)));
}

/**
 * ============ ~KviChannel ============
 */
KviChannel::~KviChannel()
{
	if( m_pBanList             ) { delete m_pBanList;             m_pBanList             = 0; }
	if( m_pExceptionList       ) { delete m_pExceptionList;       m_pExceptionList       = 0; }
	if( m_pInviteExceptionList ) { delete m_pInviteExceptionList; m_pInviteExceptionList = 0; }
}

void KviChannel::saveProperties()
{
	KviWindowProperty p;
	p.rect = externalGeometry();
	p.isDocked = isAttached();
	QValueList<int> l(m_pSplitter->sizes());
	p.splitWidth1 = 0;
	for( int i = 0; i < ((int) (l.count() - 1)); i++ ) {
		p.splitWidth1 += *(l.at(i));
	}
	p.splitWidth2 = (l.count() >= 2) ? *(l.at(l.count() - 1)) : 0;
	QValueList<int> l2(m_pTopSplitter->sizes());
	p.topSplitWidth1 = (l2.count() >= 1) ? *(l2.at(0)) : 0;
	p.topSplitWidth2 = (l2.count() >= 2) ? *(l2.at(1)) : 0;
	p.topSplitWidth3 = (l2.count() >= 3) ? *(l2.at(2)) : 0;
	p.timestamp = m_pView->timestamp();
	p.imagesVisible = m_pView->imagesVisible();
	KviWindow *w = m_pFrm->activeWindow();
	p.isMaximized = isAttached() && w ? w->isMaximized() : isMaximized();
	g_pOptions->m_pWinPropertiesList->setProperty(caption(), &p);
}

void KviChannel::setProperties(KviWindowProperty *p)
{
	QValueList<int> l;
	l.append(p->splitWidth1);
	l.append(p->splitWidth2);
	m_pSplitter->setSizes(l);
	QValueList<int> l2;
	l2.append(p->topSplitWidth1);
	l2.append(p->topSplitWidth2);
	l2.append(p->topSplitWidth3);
	m_pTopSplitter->setSizes(l2);
	m_pView->setTimestamp(p->timestamp);
	m_pView->setShowImages(p->imagesVisible);
}

void KviChannel::doEditTopic()
{
	setTopicChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp() || (!isModeActive('t')));
	if( !m_bTopicChangeEnabled ) return;

	if( !m_pTopicEditor ) {
		m_pTopicEditor = new KviTopicEditor(this);
		m_pTopicEditor->hide();
		m_pTopicEditor->connectMe(this, SLOT(topicEditCompleted()));
	}

	QPoint p = mapFromGlobal(m_pTopicLabel->mapToGlobal(QPoint(0, 0)));
	m_pTopicEditor->setFont(m_pTopicLabel->font());
	m_pTopicEditor->setGeometry(p.x(), p.y(), m_pTopicLabel->width(), m_pTopicLabel->height());
	m_pTopicEditor->setText(m_pTopicLabel->text());
	m_pTopicEditor->show();
	m_pTopicEditor->setFocus();
}

void KviChannel::topicEditCompleted()
{
	m_pTopicEditor->hide();
	KviStr tmp1(m_pTopicEditor->text());
	KviStr tmp2(m_pTopicLabel->text());
	if( !kvi_strEqualCS(tmp1.ptr(), tmp2.ptr()) ) {
		m_pFrm->m_pSocket->sendFmtData("TOPIC %s :%s", caption(), tmp1.ptr());
	}
	m_pInput->setFocus();
}

void KviChannel::doEditModes()
{
	setModeChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp());
	if( m_pModeEditor ) return;

	m_pModeEditor = new KviModeEditor(m_pSplitter);
	m_pSplitter->moveToFirst(m_pModeEditor);
	m_pModeEditor->setCanCommit(m_bModeChangeEnabled);
	m_pModeEditor->doEdit(
		m_szChanMode, m_szUsersLimit, m_szChanKey, this, SLOT(modeEditCompleted(const char *, const char *))
	);
}

void KviChannel::modeEditCompleted(const char *modeDiff, const char *additionalModeDiff)
{
	if( modeDiff ) {
		if( *modeDiff ) {
			m_pFrm->m_pSocket->sendFmtData("MODE %s %s", caption(), modeDiff);
			if( additionalModeDiff ) {
				if( *additionalModeDiff ) {
					m_pFrm->m_pSocket->sendFmtData("MODE %s %s", caption(), additionalModeDiff);
				}
			}
		}
	}
	if( m_pModeEditor ) {
		delete m_pModeEditor;
		m_pModeEditor = 0;
	}
	m_pInput->setFocus();
}

void KviChannel::usersLabelRightClicked()
{
	if( !m_pMasksEditPopup )
		m_pMasksEditPopup = new KviPopupMenu(this);
	m_pMasksEditPopup->clear();
	m_pMasksEditPopup->insertItem(__tr("&Ban List"),              this, SLOT(doEditBans()));
	m_pMasksEditPopup->insertItem(__tr("Ban &Exception List"),    this, SLOT(doEditBanExceptions()));
	m_pMasksEditPopup->insertItem(__tr("&Invite Exception List"), this, SLOT(doEditInviteExceptions()));
	m_pMasksEditPopup->popup(QCursor::pos());
}

void KviChannel::doEditBans()
{
	setModeChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp());
	if( m_pBanEditor ) return;

	m_pBanEditor = new KviMaskEditor(m_pSplitter);
	m_pSplitter->moveToFirst(m_pBanEditor);
	m_pBanEditor->doEdit(this, SLOT(banEditCompleted(const char *)), m_pBanList, __tr("Ban Mask"), m_bModeChangeEnabled);
	m_iLastMasksEdited = KVI_MASKEDIT_BANS;
}

void KviChannel::doEditBanExceptions()
{
	setModeChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp());
	if( m_pBanExceptEditor ) return;

	m_pBanExceptEditor = new KviMaskEditor(m_pSplitter);
	m_pSplitter->moveToFirst(m_pBanExceptEditor);
	m_pBanExceptEditor->doEdit(
		this, SLOT(banExceptEditCompleted(const char *)),
		m_pExceptionList, __tr("Ban Exception Mask"), m_bModeChangeEnabled
	);
	m_iLastMasksEdited = KVI_MASKEDIT_BANEXCEPTIONS;
}

void KviChannel::doEditInviteExceptions()
{
	setModeChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp());
	if( m_pInviteExceptEditor ) return;

	m_pInviteExceptEditor = new KviMaskEditor(m_pSplitter);
	m_pSplitter->moveToFirst(m_pInviteExceptEditor);
	m_pInviteExceptEditor->doEdit(
		this, SLOT(inviteExceptEditCompleted(const char *)),
			m_pInviteExceptionList, __tr("Invite Exception Mask"), m_bModeChangeEnabled);
	m_iLastMasksEdited = KVI_MASKEDIT_INVITEEXCEPTIONS;
}

void KviChannel::doEditMasks()
{
	switch( m_iLastMasksEdited ) {
		case KVI_MASKEDIT_BANS:             doEditBans();              break;
		case KVI_MASKEDIT_BANEXCEPTIONS:    doEditBanExceptions();     break;
		case KVI_MASKEDIT_INVITEEXCEPTIONS: doEditInviteExceptions();  break;
		default:                            doEditBans();              break;
	}
}

void KviChannel::banEditCompleted(const char *masksToRemove)
{
	maskEditCompleted('b', masksToRemove);
	if( m_pBanEditor ) {
		delete m_pBanEditor;
		m_pBanEditor = 0;
	}
}

void KviChannel::banExceptEditCompleted(const char *masksToRemove)
{
	maskEditCompleted('e', masksToRemove);
	if( m_pBanExceptEditor ) {
		delete m_pBanExceptEditor;
		m_pBanExceptEditor = 0;
	}
}

void KviChannel::inviteExceptEditCompleted(const char *masksToRemove)
{
	maskEditCompleted('I', masksToRemove);
	if( m_pInviteExceptEditor ) {
		delete m_pInviteExceptEditor;
		m_pInviteExceptEditor = 0;
	}
}

void KviChannel::maskEditCompleted(char aFlag, const char *masksToRemove)
{
	// Unban/unexc 3 masks per time...
	if( *masksToRemove ) {
		KviStr tmp;
		KviStr masks;
		KviStr flags;
		int i = 0;
		while( *masksToRemove ) {
			masksToRemove = kvi_extractToken(tmp, masksToRemove, ' ');
			if( masks.hasData() ) masks += " ";
			masks.append(tmp);
			flags.append(aFlag);
			i++;
			if( i == 3 ) {
				m_pFrm->m_pSocket->sendFmtData("MODE %s -%s %s", caption(), flags.ptr(), masks.ptr());
				masks = "";
				flags = "";
				i = 0;
			}
		}
		if( i != 0 ) m_pFrm->m_pSocket->sendFmtData("MODE %s -%s %s", caption(), flags.ptr(), masks.ptr());
	}
}

void KviChannel::viewRightClicked(KviIrcView *)
{
	g_pChannelPopup->doPopup(this, SLOT(viewPopupClicked(const KviStr &)));
}

void KviChannel::viewPopupClicked(const KviStr &dataBuffer)
{
	m_pFrm->m_pUserParser->parseCommand(dataBuffer.ptr(), this);
}

void KviChannel::listBoxRightClicked()
{
	g_pUserlistPopup->doPopup(this, SLOT(userlistPopupClicked(const KviStr &)));
}

void KviChannel::userlistPopupClicked(const KviStr &dataBuffer)
{
	m_pFrm->m_pUserParser->parseCommand(dataBuffer.ptr(), this);
}


/**
 * =============== applyOptions ================
 */
void KviChannel::applyLabelsOptions()
{
	m_pUsersLabel->setTextColor(g_pOptions->m_clrLabelsActiveFore);
	m_pUsersLabel->setBackColor(g_pOptions->m_clrLabelsBack);

	setTopicChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp() || (!isModeActive('t')));
	setModeChangeEnabled(isMeOwner() || isMeOp() || isMeHalfOp());

	m_pTopicLabel->setFont(g_pOptions->m_fntLabels);
	m_pModeLabel->setFont(g_pOptions->m_fntLabels);
	m_pUsersLabel->setFont(g_pOptions->m_fntLabels);

	if( !g_pOptions->m_pixLabelsBack->isNull() ) {
		m_pTopicLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
		 m_pModeLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
		m_pUsersLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
	} else {
		m_pTopicLabel->setBackgroundPixmapPointer(0);
		 m_pModeLabel->setBackgroundPixmapPointer(0);
		m_pUsersLabel->setBackgroundPixmapPointer(0);
	}
}

void KviChannel::setTopicToolTip()
{
	QToolTip::remove(m_pTopicLabel);
	const char *str = m_pTopicLabel->strippedText();
	KviStr tmp(str);
	kvi_free(str);
	QString s;
	if( tmp.hasData() ) {
		s = _CHAR_2_QSTRING(__tr("Topic: "));
		s.append(_CHAR_2_QSTRING(tmp.ptr()));
	} else s = _CHAR_2_QSTRING(__tr("No topic is set"));
	s.append("\n");
	s.append(m_bTopicChangeEnabled ? _CHAR_2_QSTRING(__tr("Double-click to edit")) : _CHAR_2_QSTRING(__tr("You cannot edit it")));
	QToolTip::add(m_pTopicLabel, s);
}

void KviChannel::setTopic(const char *text)
{
	QString *pHist = new QString(text);
	__range_valid(pHist);
	m_pTopicLabel->setText(text);
	if( m_pTopicEditor ) {
		m_pTopicEditor->m_pHistory->insert(0, pHist);
		if( m_pTopicEditor->m_pHistory->count() > 10 )
			m_pTopicEditor->m_pHistory->removeLast();
	}
	if( pHist ) delete pHist;
	setTopicToolTip();
}

void KviChannel::setTopicChangeEnabled(bool bEnabled)
{
	m_bTopicChangeEnabled = bEnabled;
	m_pTopicLabel->setTextColor(bEnabled ? g_pOptions->m_clrLabelsActiveFore : g_pOptions->m_clrLabelsInactiveFore);
	m_pTopicLabel->setBackColor(g_pOptions->m_clrLabelsBack);
	setTopicToolTip();
}

void KviChannel::setModeToolTip()
{
	QToolTip::remove(m_pModeLabel);
	QString s = __tr("Current channel mode");
	if( m_bModeChangeEnabled )
		s.append(__tr("\nDouble-click to edit"));
	QToolTip::add(m_pModeLabel, s);
}

void KviChannel::setModeChangeEnabled(bool bEnabled)
{
	m_bModeChangeEnabled = bEnabled;
	m_pModeLabel->setTextColor(bEnabled ? g_pOptions->m_clrLabelsActiveFore : g_pOptions->m_clrLabelsInactiveFore);
	m_pModeLabel->setBackColor(g_pOptions->m_clrLabelsBack);

	if( m_pModeEditor         )         m_pModeEditor->setCanCommit(m_bModeChangeEnabled);
	if( m_pBanEditor          )          m_pBanEditor->setCanCommit(m_bModeChangeEnabled);
	if( m_pBanExceptEditor    )    m_pBanExceptEditor->setCanCommit(m_bModeChangeEnabled);
	if( m_pInviteExceptEditor ) m_pInviteExceptEditor->setCanCommit(m_bModeChangeEnabled);

	setModeToolTip();
}

bool KviChannel::isMeOwner()
{
	return m_pListBox->isOwner(m_pFrm->m_global.szCurrentNick.ptr());
}

bool KviChannel::isMeOp()
{
	return m_pListBox->isOp(m_pFrm->m_global.szCurrentNick.ptr());
}

bool KviChannel::isMeHalfOp()
{
	return m_pListBox->isHalfOp(m_pFrm->m_global.szCurrentNick.ptr());
}

bool KviChannel::isMeVoice()
{
	return m_pListBox->isVoice(m_pFrm->m_global.szCurrentNick.ptr());
}

bool KviChannel::isMeUserOp()
{
	return m_pListBox->isUserOp(m_pFrm->m_global.szCurrentNick.ptr());
}

bool KviChannel::isOwner(const char *nick)
{
	return m_pListBox->isOwner(nick);
}

bool KviChannel::isOp(const char *nick)
{
	return m_pListBox->isOp(nick);
}

bool KviChannel::isHalfOp(const char *nick)
{
	return m_pListBox->isHalfOp(nick);
}

bool KviChannel::isVoice(const char *nick)
{
	return m_pListBox->isVoice(nick);
}

bool KviChannel::isUserOp(const char *nick)
{
	return m_pListBox->isUserOp(nick);
}

bool KviChannel::isModeActive(char modeFlag)
{
	QString txt = m_pModeLabel->text();
	return (txt.find(modeFlag) != -1);
}

void KviChannel::setModeFlag(char modeFlag, bool bSet, const char *param)
{
	switch( modeFlag ) {
		case 'k':
			if( bSet ) {
				m_szChanKey = param;
				m_szChanKey.cutFromFirst(' ');
				if( m_szChanKey.isEmpty() )
					m_szChanKey = "?";
			} else m_szChanKey = "";
			break;
		case 'l':
			if( bSet ) {
				m_szUsersLimit = param;
				m_szUsersLimit.cutFromFirst(' ');
				if( m_szUsersLimit.isEmpty() )
					m_szUsersLimit = "?";
			} else m_szUsersLimit = "";
			break;
		default:
			if( bSet ) {
				if( m_szChanMode.findFirstIdx(modeFlag) == -1 )
					m_szChanMode.append(modeFlag);
			} else {
				int idx = m_szChanMode.findFirstIdx(modeFlag);
				if( idx != -1 ) m_szChanMode.cut(idx, 1);
			}
		break;
	}
	KviStr tmp = m_szChanMode;
	if( m_szChanKey.hasData() ) {
		tmp.append(" k:");
		tmp.append(m_szChanKey);
	}
	if( m_szUsersLimit.hasData() ) {
		tmp.append(" l:");
		tmp.append(m_szUsersLimit);
	}
	m_pModeLabel->setText(tmp.ptr());
	if( modeFlag == 't' )
		setTopicChangeEnabled(!isModeActive('t'));
}

void KviChannel::unsetModeFlag(char modeFlag)
{
	setModeFlag(modeFlag, false);
}

void KviChannel::endOfNamesList()
{
	if( !(m_iChanSyncMask & KVI_CHANSYNC_HAVENAMES) ) {
		m_iChanSyncMask |= KVI_CHANSYNC_HAVENAMES;
		if( m_pListBox->count() == 1 ) {
			// I am alone!
			// Theoretically, no topic is set
			// And most servers will not send it
			m_iChanSyncMask |= KVI_CHANSYNC_HAVETOPIC;
		}
		checkSyncComplete();
	}
	repaintListBox();
}

void KviChannel::endOfWho()
{
	if( !(m_iChanSyncMask & KVI_CHANSYNC_HAVEWHO) ) {
		m_iChanSyncMask |= KVI_CHANSYNC_HAVEWHO;
		checkSyncComplete();
	}
}

void KviChannel::gotModeReply()
{
	if( !(m_iChanSyncMask & KVI_CHANSYNC_HAVEMODE) ) {
		m_iChanSyncMask |= KVI_CHANSYNC_HAVEMODE;
		checkSyncComplete();
	}
}

void KviChannel::gotTopicReply()
{
	if( !(m_iChanSyncMask & KVI_CHANSYNC_HAVETOPIC) ) {
		m_iChanSyncMask |= KVI_CHANSYNC_HAVETOPIC;
		checkSyncComplete();
	}
}

void KviChannel::gotBanlistReply()
{
	if( !(m_iChanSyncMask & KVI_CHANSYNC_HAVEBANS) ) {
		m_iChanSyncMask |= KVI_CHANSYNC_HAVEBANS;
		checkSyncComplete();
	}
}

void KviChannel::checkSyncComplete()
{
	bool bInSync = false;
	if( g_pOptions->m_bRequestWhoAfterEndOfNames ) {
		bInSync = (m_iChanSyncMask == (KVI_CHANSYNC_HAVEWHO | KVI_CHANSYNC_HAVEMODE | KVI_CHANSYNC_HAVETOPIC | KVI_CHANSYNC_HAVENAMES | KVI_CHANSYNC_HAVEBANS));
	} else {
		bInSync = (m_iChanSyncMask == (KVI_CHANSYNC_HAVEMODE | KVI_CHANSYNC_HAVETOPIC | KVI_CHANSYNC_HAVENAMES | KVI_CHANSYNC_HAVEBANS));
	}
	if( bInSync ) {
		int msecs = m_joinTime.msecsTo(QTime::currentTime());
		int secs  = msecs / 1000;
		msecs     = msecs % 1000;

		bool bHalt = false;
		if( g_pEventManager->eventEnabled(KviEvent_OnChannelSync) ) {
			KviStr tmp(KviStr::Format, "%s %d.%d", caption(), secs, msecs);
			bHalt = m_pFrm->m_pUserParser->callEvent(KviEvent_OnChannelSync, this, tmp);
		}

		if( g_pOptions->m_bShowChannelSyncTime && (!bHalt) ) {
			output(KVI_OUT_INTERNAL, __tr("Channel synchronized in %d.%d seconds"), secs, msecs);
		}
	}
}

bool KviChannel::hasAllNames()
{
	return (m_iChanSyncMask & KVI_CHANSYNC_HAVENAMES);
}

bool KviChannel::whoRequestDone()
{
	return (m_iChanSyncMask & KVI_CHANSYNC_HAVEWHO);
}

void KviChannel::repaintListBox()
{
	m_pListBox->doRepaint();
	updateUsersLabel();
}

void KviChannel::join(const KviIrcUser &user, char bOp, char bVoice, char bHalfOp, char bUserOp, char bOwner, bool bRepaint)
{
	m_pListBox->join(user, bOp, bVoice, bHalfOp, bUserOp, bOwner, bRepaint);
	if( bRepaint ) updateUsersLabel();
}

void KviChannel::join(const char *nick, char bOp, char bVoice, char bHalfOp, char bUserOp, char bOwner, bool bRepaint)
{
	KviIrcUser u(nick);
	m_pListBox->join(u, bOp, bVoice, bHalfOp, bUserOp, bOwner, bRepaint);
	if( bRepaint ) updateUsersLabel();
}

void KviChannel::part(const char *nick)
{
	m_pListBox->part(nick);
	updateUsersLabel();
}

void KviChannel::part(const KviIrcUser &user)
{
	m_pListBox->part(user);
	updateUsersLabel();
}

bool KviChannel::nickChange(const KviIrcUser &nicker, const char *newNick)
{
	bool bRet = m_pListBox->nickChange(nicker, newNick);
	if( bRet ) updateUsersLabel();
	return bRet;
}

bool KviChannel::owner(const char *nick, bool bOwner)
{
	bool bRet = m_pListBox->owner(nick, bOwner ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::owner(const KviIrcUser &user, bool bOwner)
{
	bool bRet = m_pListBox->owner(user, bOwner ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::op(const char *nick, bool bOp)
{
	bool bRet = m_pListBox->op(nick, bOp ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::op(const KviIrcUser &user, bool bOp)
{
	bool bRet = m_pListBox->op(user, bOp ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::halfop(const char *nick, bool bHalfOp)
{
	bool bRet = m_pListBox->halfop(nick, bHalfOp ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::halfop(const KviIrcUser &user, bool bHalfOp)
{
	bool bRet = m_pListBox->halfop(user, bHalfOp ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::voice(const char *nick, bool bVoice)
{
	bool bRet = m_pListBox->voice(nick, bVoice ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::voice(const KviIrcUser &user, bool bVoice)
{
	bool bRet = m_pListBox->voice(user, bVoice ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::userop(const char *nick, bool bUserOp)
{
	bool bRet = m_pListBox->userop(nick, bUserOp ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::userop(const KviIrcUser &user, bool bUserOp)
{
	bool bRet = m_pListBox->userop(user, bUserOp ? 1 : 0);
	updateUsersLabel();
	return bRet;
}

bool KviChannel::isOnChannel(const KviIrcUser &user)
{
	return (m_pListBox->findUser(user) != 0);
}

bool KviChannel::isOnChannel(const char *nick)
{
	return (m_pListBox->findUser(nick) != 0);
}

void KviChannel::banMask(const char *banMask, const char *setBy, const char *setAt, bool bAdd)
{
	if( bAdd ) {
		// Make sure that it is not there yet...
		for(KviBanMask *s = m_pBanList->first(); s; s = m_pBanList->next() ) {
			if( kvi_strEqualCI(s->mask.ptr(), banMask) ) return;
		}
		KviBanMask *m = new KviBanMask();
		m->mask = banMask;
		m->setBy = setBy;
		if( !setAt )
			m->setAt = QDateTime::currentDateTime().toString();
		else
			m->setAt = setAt;
		m_pBanList->append(m);
		m_iBanCount++;
		updateUsersLabel();
	} else {
		for( KviBanMask *s = m_pBanList->first(); s; s = m_pBanList->next() ) {
			if( kvi_strEqualCI(s->mask.ptr(), banMask) ) {
				m_pBanList->removeRef(s);
				m_iBanCount--;
				__range_valid(m_iBanCount >= 0);
				updateUsersLabel();
				return;
			}
		}
	}
}

void KviChannel::banExceptionMask(const char *banExcMask, const char *setBy, const char *setAt, bool bAdd)
{
	if( bAdd ) {
		// Make sure that it is not there yet...
		for( KviBanMask *s = m_pExceptionList->first(); s; s = m_pExceptionList->next() ) {
			if( kvi_strEqualCI(s->mask.ptr(), banExcMask) ) return;
		}
		KviBanMask *m = new KviBanMask();
		m->mask = banExcMask;
		m->setBy = setBy;
		if( !setAt )
			m->setAt = QDateTime::currentDateTime().toString();
		else
			m->setAt = setAt;
		m_pExceptionList->append(m);
		m_iExceptionCount++;
		updateUsersLabel();
	} else {
		for( KviBanMask *s = m_pExceptionList->first(); s; s = m_pExceptionList->next() ) {
			if( kvi_strEqualCI(s->mask.ptr(), banExcMask) ) {
				m_pExceptionList->removeRef(s);
				m_iExceptionCount--;
				__range_valid(m_iExceptionCount >= 0);
				updateUsersLabel();
				return;
			}
		}
	}
}

void KviChannel::inviteExceptionMask(const char *inviteExcMask, const char *setBy, const char *setAt, bool bAdd)
{
	if( bAdd ) {
		// Make sure that it is not there yet...
		for( KviBanMask *s = m_pInviteExceptionList->first(); s; s = m_pInviteExceptionList->next() ) {
			if( kvi_strEqualCI(s->mask.ptr(), inviteExcMask) ) return;
		}
		KviBanMask *m = new KviBanMask();
		m->mask = inviteExcMask;
		m->setBy = setBy;
		if( !setAt )
			m->setAt = QDateTime::currentDateTime().toString();
		else
			m->setAt = setAt;
		m_pInviteExceptionList->append(m);
		m_iInviteExceptionCount++;
		updateUsersLabel();
	} else {
		for( KviBanMask *s = m_pInviteExceptionList->first(); s; s = m_pInviteExceptionList->next() ) {
			if( kvi_strEqualCI(s->mask.ptr(), inviteExcMask) ) {
				m_pInviteExceptionList->removeRef(s);
				m_iInviteExceptionCount--;
				__range_valid(m_iInviteExceptionCount >= 0);
				updateUsersLabel();
				return;
			}
		}
	}
}

KviIrcUserChanList *KviChannel::userList()
{
	return m_pListBox->userList();
}

void KviChannel::appendTopic(KviStr &buffer)
{
	buffer.append(m_pTopicLabel->text());
}

void KviChannel::updateUsersLabel()
{
	KviStr str(
		KviStr::Format, "%d [o:%d v:%d b:%d", m_pListBox->count(),
		m_pListBox->opCount(), m_pListBox->voiceCount(), m_iBanCount
	);
	if( m_iExceptionCount       ) str.append(" e:%d", m_iExceptionCount);
	if( m_iInviteExceptionCount ) str.append(" i:%d", m_iInviteExceptionCount);
	str.append(']');
	m_pUsersLabel->setText(str.ptr());
}

void KviChannel::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages, false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
	applyLabelsOptions();

	m_pInput->applyOptions();
	KviWindow::applyOptions();

	// Trick
	resize(width() - 1, height() - 1);
	resize(width() + 1, height() + 1);
}

/**
 * ================ myIconPtr =================
 */
QPixmap *KviChannel::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_CHANNEL];
}

/**
 * ================ resizeEvent ===============
 */
void KviChannel::resizeEvent(QResizeEvent *)
{
	int inputSize = m_pInput->heightHint();
	int labelSize = m_pUsersLabel->sizeHint().height();

	m_pTopSplitter->setGeometry(0, 0, width(), labelSize);
	m_pSplitter->setGeometry(0, labelSize, width(), height() - (inputSize + labelSize));
	m_pInput->setGeometry(0, height() - inputSize, width(), inputSize);
}

bool KviChannel::kickDumpChanStatus(
	const char *fName, const char *kicked, const char *kicker, const char *kicker_mask, const char *reason)
{
	QFile f(fName);
	if( !f.open(IO_WriteOnly) )
		return false;

	KviStr tmp(KviStr::Format, __tr("Kick log for %s on channel %s\n"), kicked, caption());
	f.writeBlock(tmp.ptr(), tmp.len());
	QDateTime tm(QDateTime::currentDateTime());
	KviStr d = tm.toString();
	tmp.sprintf(__tr("Written at %s\n\n"), d.ptr());
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp.sprintf(__tr("You were kicked from %s by %s [%s]\n"), caption(), kicker, kicker_mask);
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp.sprintf(__tr("Kick reason: %s\n\n"), reason);
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp = __tr("Channel topic:\n");
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp = m_pTopicLabel->text();
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp = __tr("\nChannel mode:\n");
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp = m_pModeLabel->text();
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp = __tr("\nChannel text buffer:\n");
	f.writeBlock(tmp.ptr(), tmp.len());
	m_pView->getTextBuffer(tmp);
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp.sprintf(__tr("You were kicked from %s by %s [%s]: %s\n\n"), caption(), kicker, kicker_mask, reason);
	f.writeBlock(tmp.ptr(), tmp.len());
	tmp = __tr("Channel user list:\n");
	f.writeBlock(tmp.ptr(), tmp.len());
	unsigned int num = 0;
	for( KviIrcUser *pU = m_pListBox->firstUser(); pU; pU = m_pListBox->nextUser() ) {
		KviStr mask;
		pU->mask(mask);
		tmp.sprintf("%d: %s [%s]\n", num, pU->nick(), mask.ptr());
		f.writeBlock(tmp.ptr(), tmp.len());
		num++;
	}
	f.close();
	return true;
}

#include "m_kvi_channel.moc"
