// =============================================================================
//
//      --- kvi_regusersdb.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviRegisteredUsersDb"

#include <qfile.h>

#include "kvi_defines.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_regusersdb.h"

// TODO: Command to set/get the pass

KviRegisteredUsersDb::KviRegisteredUsersDb()
{
	m_pRegList = new QPtrList<KviRegisteredUser>;
	m_pRegList->setAutoDelete(true);
}

KviRegisteredUsersDb::~KviRegisteredUsersDb()
{
	delete m_pRegList;
	m_pRegList = 0;
}

void KviRegisteredUsersDb::registerUser(KviRegisteredUser *user, const char *flags)
{
	if( flags )
		setFlags(user, flags);
	m_pRegList->append(user);
}

void KviRegisteredUsersDb::deleteFlags(KviRegisteredUser *user, const char *flags)
{
	KviStr *tempFlags   = new KviStr();
	KviStr *deleteFlags = new KviStr(flags);
	char   *oldFlags    = user->allFlags.ptr();

	while( *oldFlags ) {
		if( !deleteFlags->contains(*oldFlags) ) tempFlags->append(*oldFlags);
		if(  deleteFlags->contains("n")       ) user->bNotify = 0;
		if(  deleteFlags->contains("i")       ) user->bIgnore = 0;
		oldFlags++;
	}
	user->allFlags = tempFlags->ptr();
}

void KviRegisteredUsersDb::addFlags(KviRegisteredUser *user, const char *flags)
{
	while( *flags ) {
		if( *flags == 'n' )
			user->bNotify = 1;
		else if( *flags == 'i' )
			user->bIgnore = 1;
		if( !(user->allFlags.contains(*flags, true)) ) {
			if( ((*flags >= 'A') && (*flags <= 'Z')) ||
				((*flags >= 'a') && (*flags <= 'z')) ||
				((*flags >= '0') && (*flags <= '9'))
			) {
				user->allFlags.append(*flags);
			}
		}
		flags++;
	}
}

void KviRegisteredUsersDb::setFlags(KviRegisteredUser *user, const char *flags)
{
	user->allFlags.setLen(0);
	user->bNotify = 0;
	user->bIgnore = 0;
	addFlags(user, flags);
}

void KviRegisteredUsersDb::setPasswd(KviRegisteredUser *user, const char *passwd)
{
	user->passwd = passwd;
}

void KviRegisteredUsersDb::getPasswd(KviRegisteredUser *user, KviStr &buf)
{
	buf = user->passwd;
}

void KviRegisteredUsersDb::setComment(KviRegisteredUser *user, const char *comment)
{
	user->comment = comment;
}

void KviRegisteredUsersDb::getComment(KviRegisteredUser *user, KviStr &buf)
{
	buf = user->comment;
}

void KviRegisteredUsersDb::unregisterUser(KviRegisteredUser *user)
{
	m_pRegList->removeRef(user);
}

void KviRegisteredUsersDb::clear()
{
	delete m_pRegList;
	m_pRegList = new QPtrList<KviRegisteredUser>;
	m_pRegList->setAutoDelete(true);
}

KviRegisteredUser *KviRegisteredUsersDb::findUserByMask(const char *mask)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		if( r->user.matches(mask) )
			return r;
	}
	return 0;
}

KviRegisteredUser *KviRegisteredUsersDb::findUserWithFlagByMask(const char *mask, char flag)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		if( r->user.matches(mask) ) {
			if( r->allFlags.findFirstIdx(flag) >= 0 )
				return r;
		}
	}
	return 0;
}

/**
 * Speed up the lookups
 */
KviRegisteredUser *KviRegisteredUsersDb::findIgnoredUserByMask(const char *mask)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		if( r->user.matches(mask) ) {
			if( r->bIgnore == 1 )
				return r;
		}
	}
	return 0;
}

KviRegisteredUser *KviRegisteredUsersDb::findExactMatch(const char *mask)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		KviStr m;
		r->user.mask(m);
		if( kvi_strEqualCI(m.ptr(), mask) )
			return r;
	}
	return 0;
}

void KviRegisteredUsersDb::getFlags(const char *mask, KviStr &buf)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		if( r->user.matches(mask) ) {
			if( !buf.hasData() )
				buf = r->allFlags;
			else {
				char *flags = r->allFlags.ptr();
				while( *flags ) {
					if( !(buf.contains(*flags, true)) )
						buf.append(*flags);
					flags++;
				}
			}
		}
	}
}

void KviRegisteredUsersDb::getFlagsExact(const char *mask, KviStr &buf)
{
	KviRegisteredUser *r = findExactMatch(mask);
	if( r )
		buf = r->allFlags;
}

void KviRegisteredUsersDb::getWatchString(KviStr &buf)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		if( r->bNotify != 0 ) {
			KviStr tmp = r->user.nick();
			if( tmp.findFirstIdx('*')==-1 ) {
				if( buf.hasData() )
					buf.append(' ');
				buf.append('+');
				buf.append(r->user.nick());
			}
		}
	}
}

void KviRegisteredUsersDb::getNotifyString(KviStr &buf)
{
	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		if( r->bNotify != 0 ) {
			KviStr tmp = r->user.nick();
			if( tmp.findFirstIdx('*') == -1 ) {
				if( buf.hasData() )
					buf.append(' ');
				buf.append(r->user.nick());
			}
		}
	}
}

bool KviRegisteredUsersDb::load(const char *filename, KviStr &error)
{
	KviStr buffer;
	if( !kvi_loadFile(filename, buffer) ) {
		error = __tr("Unable to load file");
		return false;
	}
	if( !kvi_strEqualCSN(KVI_MAGIC_STRING, buffer.ptr(), KVI_MAGIC_STRING_LENGTH) ) {
		error = __tr("Invalid magic");
		return false; // Not a KVIrc file
	}
	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);
	buffer.stripLeftWhiteSpace(); // Be flexible... allow some spaces... for manual editing by experts :)

	if( buffer.isEmpty() )
		return true; // Done
	while( buffer.hasData() ) {
		KviStr line;
		buffer.getLine(line);
		if( line.isEmpty() )
			continue;
		KviRegisteredUser *user = new KviRegisteredUser();

		KviStr tmp, aux;
		tmp = user->user.setMask(line.ptr(), ' ');
		tmp.stripLeftWhiteSpace();
		// Get flags
		tmp.getToken(aux, ' ');
		registerUser(user, aux.ptr());

		// Get passwd
		tmp.getToken(aux, ' ');
		if( aux.hasData() )
			setPasswd(user, aux.ptr());

		// Get passwd
		if( tmp.hasData() )
			setComment(user, tmp.ptr());
	}
	return true;
}

bool KviRegisteredUsersDb::save(const char *filename)
{
	QFile f(filename);
	if( !f.open(IO_WriteOnly | IO_Truncate) )
		return false;

	f.writeBlock(KVI_MAGIC_STRING, KVI_MAGIC_STRING_LENGTH);

	for( KviRegisteredUser *r = m_pRegList->first(); r; r = m_pRegList->next() ) {
		KviStr tmp;
		r->user.mask(tmp);

		// Has the user any flags... not only n, i
		if( (r->allFlags.hasData()) ) {
			tmp.append(" ");
			tmp.append(r->allFlags);
		} else
			tmp.append(" -");	// Placeholder if we have no flags

		if( (r->passwd.hasData()) ) {
			tmp.append(" ");
			tmp.append(r->passwd);
		} else
			tmp.append(" -");	// Placeholder if we have no password

		if( (r->comment.hasData()) ) {
			tmp.append(" ");
			tmp.append(r->comment);
		}
		f.writeBlock(tmp.ptr(), tmp.len());
		f.writeBlock("\n", 1);
	}

	f.close();
	return true;
}
