// =============================================================================
//
//      --- kvi_serverdialog.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviServerDialog"

#include <qcursor.h>
#include <qheader.h>
#include <qlayout.h>
#include <qtooltip.h>

#include "kvi_app.h"
#include "kvi_boolselector.h"
#include "kvi_combobox.h"
#include "kvi_debug.h"
#include "kvi_fieldeditor.h"
#include "kvi_fileutils.h"
#include "kvi_frame.h"
#include "kvi_iconloader.h"
#include "kvi_integerselector.h"
#include "kvi_irc_network.h"
#include "kvi_irc_proxy.h"
#include "kvi_irc_proxylist.h"
#include "kvi_irc_server.h"
#include "kvi_irc_serverlist.h"
#include "kvi_label.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_maintoolbar.h"
#include "kvi_options.h"
#include "kvi_pushbutton.h"
#include "kvi_serverdialog.h"
#include "kvi_settings.h"
#include "kvi_stringselector.h"

/**
 * The server options dialog
 */
KviServerDialog::KviServerDialog(QWidget *parent)
	: KviTabDialog(parent, "dlgserver", false, _i18n_("KVIrc: Server Options"))
{
	setIcon(KviIconLoader::loadIcon(tb_configServer));

	// Local copy of data
	m_pManager      = new KviIrcServerManager();
	m_pProxyManager = new KviIrcProxyManager();
	     m_pManager->copyFrom(g_pOptions->m_pServerManager);
	m_pProxyManager->copyFrom(g_pOptions->m_pProxyManager);

	m_curEditedServerItem = 0;
	m_curEditedProxyItem  = 0;

	//
	// Server tab...
	//

	//   0           1                        2
	// 0 +++++++++++ ++++++++++++++++++++++++ +++++++++++++++++
	//   +  Label  + +      m_pNetCombo     + +   New Network +
	//   +++++++++++ ++++++++++++++++++++++++ +++++++++++++++++
	// 1 ++++++++++++++++++++++++++++++++++++ +++++++++++++++++
	//   +                                  + +   m_pDelNet   +
	//   +                  m_pServerList   + +++++++++++++++++
	// 2 +                                  + +++++++++++++++++
	//   +                                  + +  m_pNewServer +
	//   +                                  + +++++++++++++++++
	// 3 +                                  + +++++++++++++++++
	//   +                                  + + m_pEditServer +
	//   +                                  + +++++++++++++++++
	// 4 +                                  + +++++++++++++++++
	//   +                                  + +  m_pDelServer +
	//   +                                  + +++++++++++++++++
	// 5 +                                  + +++++++++++++++++
	//   +                                  + +    m_pSort    +
	//   +                                  + +++++++++++++++++
	// 6 +                                  + +++++++++++++++++
	//   +                                  + + import *.ini  +
	//   ++++++++++++++++++++++++++++++++++++ +++++++++++++++++
	//
	QFrame *tab = addPage(_CHAR_2_QSTRING(_i18n_("&Servers")));
	// Tab, rows, cols
	QGridLayout *g = new QGridLayout(tab, 7, 3, 10, 4);

	KviLabel *l = new KviLabel(__tr("Network:"), tab);
	g->addWidget(l, 0, 0);

	m_pNetCombo = new KviComboBox(true, tab);
	// Row, column
	g->addWidget(m_pNetCombo, 0, 1);
	connect(m_pNetCombo, SIGNAL(highlighted(const QString &)), this, SLOT(currentNetworkChanged(const QString &)));

	m_pServerList = new KviListView(tab, "serverlist", false);

	m_pServerList->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pServerList->addColumn(_i18n_("Server"));
	m_pServerList->addColumn(_i18n_("Port"));
	m_pServerList->addColumn(_i18n_("Description"));
	m_pServerList->addColumn(_i18n_("Password"));
	m_pServerList->addColumn(_i18n_("IP Address"));
#ifdef COMPILE_NEED_IPV6
	m_pServerList->addColumn(_i18n_("IPv6"));
#endif
	m_pServerList->setMultiSelection(false);
	m_pServerList->setSorting(-1);
	m_pServerList->header()->setClickEnabled(false, -1); // Disable click
	m_pServerList->header()->setMovingEnabled(false);
	m_pServerList->setFocusPolicy(StrongFocus);
	QToolTip::add(m_pServerList, __tr("Double-click to edit"));

	m_pEditor = new KviFieldEditor(m_pServerList->viewport());
	m_pServerList->setFieldEditor(m_pEditor);
	connect(m_pEditor, SIGNAL(editFinished(const QString &)), this, SLOT(serverEditFinished(const QString &)));

	// fromRow, toRow, fromCol, toCol
	g->addMultiCellWidget(m_pServerList, 1, 6, 0, 1);
	connect(m_pServerList, SIGNAL(currentChanged(KviListViewItem *)), this, SLOT(currentServerChanged(KviListViewItem *)));
	connect(m_pServerList, SIGNAL(doubleClicked(KviListViewItem *)),  this, SLOT(serverDoubleClicked(KviListViewItem *)));

	KviPushButton *newNet = new KviPushButton(_i18n_("Add &Network"), tab);
	g->addWidget(newNet, 0, 2);
	connect(newNet, SIGNAL(clicked()), this, SLOT(newNetwork()));

	m_pDelNet = new KviPushButton(_i18n_("Re&move Network"), tab);
	g->addWidget(m_pDelNet, 1, 2);
	connect(m_pDelNet, SIGNAL(clicked()), this, SLOT(deleteNetwork()));

	m_pNewServer = new KviPushButton(_i18n_("&Add Server"), tab);
	g->addWidget(m_pNewServer, 2, 2);
	connect(m_pNewServer, SIGNAL(clicked()), this, SLOT(newServer()));

	m_pEditServer = new KviPushButton(_i18n_("&Edit Server"), tab);
	g->addWidget(m_pEditServer, 3, 2);
	connect(m_pEditServer, SIGNAL(clicked()), this, SLOT(editServer()));

	m_pDelServer = new KviPushButton(_i18n_("&Remove Server"), tab);
	g->addWidget(m_pDelServer, 4, 2);
	connect(m_pDelServer, SIGNAL(clicked()), this, SLOT(deleteServer()));

	m_pSort = new KviPushButton(_i18n_("&Sort by Domain"), tab);
	g->addWidget(m_pSort, 5, 2);
	connect(m_pSort, SIGNAL(clicked()), this, SLOT(sortServers()));

	KviPushButton *import = new KviPushButton(_i18n_("&Import from *.ini"), tab);
	g->addWidget(import, 6, 2);
	connect(import, SIGNAL(clicked()), this, SLOT(importFromIni()));

	g->setColStretch(1, 1);
	g->activate();

	//
	// Other misc options tab
	//
	tab = addPage(_CHAR_2_QSTRING(_i18n_("&Connection")));
	// Tab, rows, cols,
	g = new QGridLayout(tab, 10, 1, 10, 4);

	KviStringSelector *ss = new KviStringSelector(tab,
		__tr("Local host IP address:"), &(g_pOptions->m_szLocalHostIp), true
	);
	g->addWidget(ss, 0, 0);


	KviBoolSelector *bs = new KviBoolSelector(tab,
		__tr("Bind IRC sockets to the local host IP address"), &(g_pOptions->m_bBindIrcSocketToSpecificInterface)
	);
	g->addWidget(bs, 1, 0);

	KviIntegerSelector *is = new KviIntegerSelector(tab,
		__tr("Connection timeout:"),
		(void *) &(g_pOptions->m_iSocketTimeout),
		KviIntegerSelector::Int, 5, 600, 60, true, 100, 0,
		__tr(" second(s)")
	);
	g->addWidget(is, 2, 0);

	bs = new KviBoolSelector(tab,
		__tr("Automatically reconnect on disconnect"), &(g_pOptions->m_bAutoReconnectOnDisconnect)
	);
	g->addWidget(bs, 3, 0);

	bs = new KviBoolSelector(tab,
		__tr("Close channel and query windows on disconnect"), &(g_pOptions->m_bCloseWindowsOnDisconnect)
	);
	g->addWidget(bs, 4, 0);

	bs = new KviBoolSelector(tab, __tr("Rejoin channels on reconnect"), &(g_pOptions->m_bRejoinChannelsOnReconnect));
	g->addWidget(bs, 5, 0);

	bs = new KviBoolSelector(tab,
		__tr("Try next server in the network if connect fails"), &(g_pOptions->m_bTryNextServerOnConnectFailed)
	);
	g->addWidget(bs, 6, 0);

	QFrame *r = new QFrame(tab);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(r, 7, 0);

	bs = new KviBoolSelector(tab, __tr("Force synchronous DNS calls"), &(g_pOptions->m_bForceSyncDns));
	g->addWidget(bs, 8, 0);

	r = new QFrame(tab);
	g->addWidget(r, 9, 0);
	g->setRowStretch(9, 1);

	g->activate();

	//
	// Other misc options tab
	//
	tab = addPage(_CHAR_2_QSTRING(_i18n_("&Proxy Servers")));
	// Tab, rows, cols
	g = new QGridLayout(tab, 7, 2, 10, 4);

	bs = new KviBoolSelector(tab,
		__tr("Suppress auto-login with nickname and username"), &(g_pOptions->m_bUseTelnetProxy)
	);
	g->addMultiCellWidget(bs, 0, 0, 0, 1);

	KviBoolSelector *bs1 = new KviBoolSelector(tab,
		__tr("Use HTTP/1.0 proxy server"), &(g_pOptions->m_bProxyProtoHttp)
	);
	g->addMultiCellWidget(bs1, 1, 1, 0, 1);

	KviBoolSelector *bs2 = new KviBoolSelector(tab, __tr("Use SOCKS proxy server"), &(g_pOptions->m_bUseProxy));
	g->addMultiCellWidget(bs2, 2, 2, 0, 1);
	connect(bs2, SIGNAL(toggled(bool)), bs1, SLOT(setDisabled(bool)));
	connect(bs1, SIGNAL(toggled(bool)), bs2, SLOT(setDisabled(bool)));

	bs = new KviBoolSelector(tab,
		__tr("Use SOCKS protocol version 5 instead of version 4"),
		&(g_pOptions->m_bProxyProtoV5), g_pOptions->m_bUseProxy
	);
	connect(bs2, SIGNAL(toggled(bool)), bs, SLOT(setEnabled(bool)));
	g->addMultiCellWidget(bs, 3, 3, 0, 1);

	m_pProxyList = new KviListView(tab, "proxylist", false);

	m_pProxyList->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pProxyList->addColumn(_i18n_("Proxy"));
	m_pProxyList->addColumn(_i18n_("Port"));
	m_pProxyList->addColumn(_i18n_("Username"));
	m_pProxyList->addColumn(_i18n_("Password"));
	m_pProxyList->addColumn(_i18n_("IP Address"));
	m_pProxyList->setMultiSelection(false);
	m_pProxyList->setSorting(-1);
	m_pProxyList->header()->setClickEnabled(false, -1); // Disable click
	m_pProxyList->header()->setMovingEnabled(false);
	m_pProxyList->setFocusPolicy(StrongFocus);
	QToolTip::add(m_pProxyList, __tr("Double-click to edit"));

	m_pProxyEditor = new KviFieldEditor(m_pProxyList->viewport());
	m_pProxyList->setFieldEditor(m_pProxyEditor);
	connect(m_pProxyEditor, SIGNAL(editFinished(const QString &)), this, SLOT(proxyEditFinished(const QString &)));

	// fromRow, toRow, fromCol, toCol
	g->addMultiCellWidget(m_pProxyList, 4, 6, 0, 0);
	connect(m_pProxyList, SIGNAL(currentChanged(KviListViewItem *)), this, SLOT(currentProxyChanged(KviListViewItem *)));
	connect(m_pProxyList, SIGNAL(doubleClicked(KviListViewItem *)),  this, SLOT(proxyDoubleClicked(KviListViewItem *)));

	KviPushButton *nProxy = new KviPushButton(_i18n_("&Add Proxy"), tab);
	g->addWidget(nProxy, 4, 1);
	connect(nProxy, SIGNAL(clicked()), this, SLOT(newProxy()));

	m_pEditProxy = new KviPushButton(_i18n_("&Edit Proxy"), tab);
	g->addWidget(m_pEditProxy, 5, 1);
	connect(m_pEditProxy, SIGNAL(clicked()), this, SLOT(editProxy()));

	m_pDelProxy = new KviPushButton(_i18n_("&Remove Proxy"), tab);
	g->addWidget(m_pDelProxy, 6, 1);
	connect(m_pDelProxy, SIGNAL(clicked()), this, SLOT(deleteProxy()));

	g->setRowStretch(4, 1);
	g->activate();

	//
	// Init stuff
	//
	setButtonHelpText(__tr("What is &This?"));
	connect(this, SIGNAL(helpClicked()), g_pApp, SLOT(slot_whatIsThisRequest()));
	setButtonOKText(__tr("&OK"));
	setButtonCancelText(__tr("&Cancel"));

	installEventFilter(this);

	fillNetCombo();
	fillProxyList();
}

/**
 *
 * Destruction
 *
 */
KviServerDialog::~KviServerDialog()
{
	delete m_pManager;      m_pManager      = 0;
	delete m_pProxyManager; m_pProxyManager = 0;
}

void KviServerDialog::done(int r)
{
	KviTabDialog::done(r);
	// Commit the changes
	if( r == Accepted ) {
		updateNetworkName();
		KviIntegerSelector::commitAll(this);
		KviBoolSelector::commitAll(this);
		KviStringSelector::commitAll(this);
		g_pOptions->m_pServerManager->copyFrom(m_pManager);
		g_pOptions->m_pProxyManager->copyFrom(m_pProxyManager);
	}
	emit finished((r == Accepted));
}

void KviServerDialog::closeEvent(QCloseEvent *)
{
	emit finished(false);
}

/**
 *
 * Servers tab stuff
 *
 */
void KviServerDialog::fillNetCombo()
{
	// This will clear the combo, fill it with the network list,
	// make sure that a current network exists, and select it.
	// If the network list is empty, it will disable the comboBox and the deleteNet button
	m_pNetCombo->clear();
	__range_valid(m_pManager->networkList());
	m_pNetCombo->setEnabled(!m_pManager->networkList()->isEmpty());
	m_pDelNet->setEnabled(!m_pManager->networkList()->isEmpty());
	m_pNewServer->setEnabled(!m_pManager->networkList()->isEmpty());
	if( m_pManager->networkList()->isEmpty() ) {
		m_pServerList->clear();
		enableServerButtons(false);
		return;
	}
	int idx = 0;
	int cur = -1;
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
	// Fill the list
	for( KviIrcNetwork *n = m_pManager->networkList()->first(); n; n = m_pManager->networkList()->next() ) {
		m_pNetCombo->insertItem(n->name()); // Append it
		if( net == n )
			cur = idx; // This is the current...
		idx++;
	}
	__range_valid(cur >= 0); // Now must be set!
	// Select it
	m_pNetCombo->setCurrentItem(cur);
	fillServerList(net);
}

void KviServerDialog::deleteNetwork()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pManager->removeNetwork(net));
#else
	m_pManager->removeNetwork(net);
#endif
	fillNetCombo();
}

void KviServerDialog::deleteServer()
{
	KviIrcServerListItem *it = (KviIrcServerListItem *) m_pServerList->currentItem();
	if( !it ) return;

	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(net->removeServer(it->m_ptr));
#else
	net->removeServer(it->m_ptr);
#endif
	fillServerList(net);
}

void KviServerDialog::editServer()
{
	KviIrcServerListItem *it = (KviIrcServerListItem *) m_pServerList->currentItem();
	if( it )
		editServerField(it, Host);
}

void KviServerDialog::newServer()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	if( !net ) return;

	KviIrcServer *srv = new KviIrcServer();
	srv->addresses.clear();
	srv->szHost        = "irc.unknown.net";
	srv->szDescription = __tr("None");
	srv->szPort        = "6667";
	srv->szPassword    = "";
#ifdef COMPILE_NEED_IPV6
	srv->bIPv6         = false;
#endif
	net->appendServer(srv);
	fillServerList(net);
}

void KviServerDialog::serverEditFinished(const QString &text)
{
	__range_valid(m_curEditedServerItem);
	KviStr szT(text);

	switch( m_curEditedServerField ) {
		case Host:
			szT.stripWhiteSpace();
			if( szT.findFirstIdx(' ') == -1 ) {
				// Must contain at least one dot and no spaces
				m_curEditedServerItem->m_ptr->szHost = szT.ptr();
				m_curEditedServerItem->setText(0, szT.ptr());
			}
			break;
		case Port:
			szT.stripWhiteSpace();
			if( !szT.isUnsignedNum() )
				szT = "6667"; // Reasonable default
			m_curEditedServerItem->m_ptr->szPort = szT.ptr();
			m_curEditedServerItem->setText(1, szT.ptr());
			break;
		case Description:
			m_curEditedServerItem->m_ptr->szDescription = szT.ptr();
			m_curEditedServerItem->setText(2, szT.ptr());
			break;
		case Password:
			m_curEditedServerItem->m_ptr->szPassword = szT.ptr();
			m_curEditedServerItem->setText(3, szT.ptr());
			break;
		case IP: {
			szT.stripWhiteSpace();
			QHostAddress addr;
			m_curEditedServerItem->m_ptr->addresses.clear();
			if( !addr.setAddress(szT.ptr()) )
				m_curEditedServerItem->setText(4, "");
			else {
				m_curEditedServerItem->m_ptr->addresses.append(addr);
				m_curEditedServerItem->setText(4, szT.ptr());
			}
			break;
		}
		case IPv6:
#ifdef COMPILE_NEED_IPV6
			szT.stripWhiteSpace();
			m_curEditedServerItem->m_ptr->bIPv6 = (
				kvi_strEqualCI(szT.ptr(), "yes") || kvi_strEqualCI(szT.ptr(), "1") || kvi_strEqualCI(szT.ptr(), "true")
			);
			m_curEditedServerItem->setText(5, szT.ptr());
			break;
#endif
		default:
			return;
			break;
	}
	m_pServerList->setFocus();
}

void KviServerDialog::importFromIni()
{
	KviStr szFName(kvi_askForOpenFileName(0, "*.ini"));
	if( !szFName.isEmpty() ) {
		if( !m_pManager->importFromIni(szFName.ptr()) )
			g_pApp->warningBox(_i18n_("Cannot open file %s\n"), szFName.ptr());
		fillNetCombo();
	}
}

void KviServerDialog::editServerField(KviIrcServerListItem *it, KviOptDlgServerField field)
{
	__range_valid(it);
	__range_invalid(m_pEditor->isVisible());
	QRect rct = m_pServerList->itemRect(it);
	int fieldWidth = m_pServerList->columnWidth(field);
	int fieldX     = -m_pServerList->contentsX();
	for( int i = 0; i < field; i++ )
		fieldX += m_pServerList->columnWidth(i);
	m_pEditor->move(fieldX, rct.top());
	m_pEditor->resize(fieldWidth, rct.height());
	QString field_ptr("");
	switch( field ) {
#ifdef COMPILE_NEED_IPV6
		case IPv6:        field_ptr = (it->m_ptr->bIPv6 ? (char *) "yes" : (char *) "no"); break;
#endif
		case Host:        field_ptr = it->m_ptr->szHost.ptr();        break;
		case Port:        field_ptr = it->m_ptr->szPort.ptr();        break;
		case Description: field_ptr = it->m_ptr->szDescription.ptr(); break;
		case Password:    field_ptr = it->m_ptr->szPassword.ptr();    break;
		case IP: {
			// TODO: Only showing the first IP address for now. Adding a
			// dropdown for all the aliases might clutter the interface
			// overly much
			QHostAddress addr = it->m_ptr->addresses.first();
			if( !addr.isNull() )
				field_ptr = addr.toString().ascii();
			break;
		}
		default:
			return;
			break;
	}
	m_curEditedServerField = field;
	m_curEditedServerItem  = it;
	m_pEditor->edit(field_ptr.ascii());
}

void KviServerDialog::serverDoubleClicked(KviListViewItem *i)
{
	KviIrcServerListItem *it = (KviIrcServerListItem *) i;
	QPoint pnt = QCursor::pos();
	pnt = m_pServerList->viewport()->mapFromGlobal(pnt);
	int xLeft = m_pServerList->columnWidth(0) - m_pServerList->contentsX();
	if( pnt.x() < xLeft )
		editServerField(it, Host);
	else {
		xLeft += m_pServerList->columnWidth(1); // Port
		if( pnt.x() < xLeft )
			editServerField(it, Port);
		else {
			xLeft += m_pServerList->columnWidth(2); // Description
			if( pnt.x() < xLeft )
				editServerField(it, Description);
			else {
				xLeft += m_pServerList->columnWidth(3); // Password
				if( pnt.x() < xLeft )
					editServerField(it, Password);
				else {
					xLeft += m_pServerList->columnWidth(4); // IP
					if( pnt.x() < xLeft )
						editServerField(it, IP);
#ifdef COMPILE_NEED_IPV6
					else {
						xLeft += m_pServerList->columnWidth(5); // IPv6
						if( pnt.x() < xLeft )
							editServerField(it, IPv6);
					}
#endif
				}
			}
		}
	}
}

void KviServerDialog::sortServers()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
	net->sortServers();
	fillNetCombo();
}

void KviServerDialog::newNetwork()
{
	KviStr newNet = _i18n_("New Network");
	// Unique name please!
	while( m_pManager->getNetworkByName(newNet.ptr()) )
		newNet += "!";
	KviIrcNetwork *net = new KviIrcNetwork(newNet.ptr());
	m_pManager->insertNetwork(net);
	net->sortServers();
	fillNetCombo();
}

void KviServerDialog::currentNetworkChanged(const QString &name)
{
	KviStr text = name; // Mmmmh, I do not like it, but it is here...
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pManager->setCurrentNetwork(text.ptr()));
#else
	m_pManager->setCurrentNetwork(text.ptr());
#endif
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(stricmp(text.ptr(), net->name()) == 0);
	fillServerList(net);
}

void KviServerDialog::currentServerChanged(KviListViewItem *it)
{
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pManager->setCurrentServer(((KviIrcServerListItem *) it)->m_ptr));
#else
	m_pManager->setCurrentServer(((KviIrcServerListItem *) it)->m_ptr);
#endif
}

void KviServerDialog::enableServerButtons(bool bEnable)
{
	m_pSort->setEnabled(bEnable);
	m_pEditServer->setEnabled(bEnable);
	m_pDelServer->setEnabled(bEnable);
}

void KviServerDialog::fillServerList(KviIrcNetwork *net)
{
	__range_valid(net);
	m_pServerList->clear();
	__range_valid(net->serverList());
	if( net->serverList()->isEmpty() ) {
		// Disable everything
		enableServerButtons(false);
		return;
	}
	// Fill the server list
	KviIrcServer *srv = net->currentServer();
	KviIrcServerListItem *cur = 0;
	__range_valid(srv); // MUST BE SET!!!! the list is not empty!
	m_pServerList->setUpdatesEnabled(false);
	int idx = 0;
	int curIdx = -1;
	for( KviIrcServer *s = net->serverList()->first(); s; s = net->serverList()->next() ) {
		KviIrcServerListItem *it = new KviIrcServerListItem(m_pServerList, s);
		if( srv == s ) {
			m_pServerList->setCurrentItem(it);
			m_pServerList->setSelected(it, true);
			cur    = it;
			curIdx = idx;
		}
		idx++;
	}
	m_pServerList->setUpdatesEnabled(true);
	__range_valid(cur); // Must be set now!!!
	m_pServerList->ensureItemVisible(cur);
	__range_valid(curIdx >= 0);
	enableServerButtons(true);
}

bool KviServerDialog::eventFilter(QObject *, QEvent *e)
{
	if( (e->type() != QEvent::Accel) && (e->type() != QEvent::KeyPress) )
		return false; // We're only interested in key press events, really

	QKeyEvent *ev = (QKeyEvent *) e;
	if( m_pEditor->isVisible() ) {
		if( ev->key() != Qt::Key_Tab ) {
			m_pEditor->keyPressEvent(ev);
		} else {
			// Loop through fields
			KviOptDlgServerField field;
			switch( m_curEditedServerField ) {
				case Host:        field = Port;        break;
				case Port:        field = Description; break;
				case Description: field = Password;    break;
				case Password:    field = IP;          break;
#ifndef COMPILE_NEED_IPV6
				case IP:          field = Host;        break;
#else
				case IP:          field = IPv6;        break;
				case IPv6:        field = Host;        break;
#endif
				default:          field = Host;        break;
			}
			// Accept the previous changes...
			m_pEditor->terminateEdit(true);
			// And edit the next field
			editServerField(m_curEditedServerItem, field);
		}
		ev->accept();
		return true;
	}

	if( m_pProxyEditor->isVisible() ) {
		if( ev->key() != Qt::Key_Tab ) {
			m_pProxyEditor->keyPressEvent(ev);
		} else {
			// Loop through fields
			KviOptDlgProxyField field;
			switch( m_curEditedProxyField ) {
				case ProxyHost:     field = ProxyPort;     break;
				case ProxyPort:     field = ProxyUsername; break;
				case ProxyUsername: field = ProxyPassword; break;
				case ProxyPassword: field = ProxyIP;       break;
				case ProxyIP:       field = ProxyHost;     break;
				default:            field = ProxyHost;     break;
			}
			// Accept the previous changes...
			m_pProxyEditor->terminateEdit(true);
			// And edit the next field
			editProxyField(m_curEditedProxyItem, field);
		}
		ev->accept();
		return true;
	}

	if( ev->key() == Qt::Key_Escape )
		return false;
	if( m_pServerList->hasFocus() && (ev->key() == Qt::Key_Space) ) {
		editServer();
		ev->accept();
	} else if( m_pProxyList->hasFocus() && (ev->key() == Qt::Key_Space) ) {
		editProxy();
		ev->accept();
	} else if( m_pNetCombo->hasFocus() ) {
		if( ev->key() == Qt::Key_Return )
			ev->accept(); // Always block RETURN key event
		if( ev->key() == Qt::Key_Return || ev->key() == Qt::Key_Tab ) {
			// Name of the current network changed?
			if( updateNetworkName() ) fillNetCombo();
		}
	}
	return true;
}

bool KviServerDialog::updateNetworkName()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
	if( !net ) return false; // Should not happen

	KviStr szNet = m_pNetCombo->currentText();
	if( !szNet.isEmpty() ) {
		KviIrcNetwork *newNet = m_pManager->getNetworkByName(szNet.ptr());
		// net != 0 now
		// newNet now can be == 0   --> new name for the current network
		//                   == net --> new name for the current network (changed just case)
		//           != 0 && != net --> another existing network name selected
		if( newNet && (newNet != net) ) {
			// Another network name
			m_pManager->setCurrentNetwork(newNet);
		} else {
			// Renamed the current network
			newNet = new KviIrcNetwork(net);
			newNet->setName(szNet.ptr());
			m_pManager->removeNetwork(net);
			m_pManager->insertNetwork(newNet);
			return true;
		}
	}
	return false;
}

/**
 *
 * Proxy tab stuff
 *
 */
void KviServerDialog::fillProxyList()
{
	m_pProxyList->clear();
	if( m_pProxyManager->proxyList()->isEmpty() ) {
		// Disable everything
		enableProxyButtons(false);
		return;
	}
	// Fill the server list
	KviIrcProxy *prx = m_pProxyManager->currentProxy();
	KviIrcProxyListItem *cur = 0;
	__range_valid(prx); // MUST BE SET!!!! the list is not empty!
	m_pProxyList->setUpdatesEnabled(false);
	int idx = 0;
	int curIdx = -1;
	for( KviIrcProxy *s = m_pProxyManager->proxyList()->first(); s; s = m_pProxyManager->proxyList()->next() ) {
		KviIrcProxyListItem *it = new KviIrcProxyListItem(m_pProxyList, s);
		if( prx == s ) {
			m_pProxyList->setCurrentItem(it);
			m_pProxyList->setSelected(it, true);
			cur = it;
			curIdx = idx;
		}
		idx++;
	}
	m_pProxyList->setUpdatesEnabled(true);
	__range_valid(cur); // Must be set now!!!
	m_pProxyList->ensureItemVisible(cur);
	__range_valid(curIdx >= 0);
	enableProxyButtons(true);
}

void KviServerDialog::enableProxyButtons(bool bEnable)
{
	m_pEditProxy->setEnabled(bEnable);
	m_pDelProxy->setEnabled(bEnable);
}

void KviServerDialog::deleteProxy()
{
	KviIrcProxyListItem *it = (KviIrcProxyListItem *) m_pProxyList->currentItem();
	if( !it ) return;

#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pProxyManager->removeProxy(it->m_ptr));
#else
	m_pProxyManager->removeProxy(it->m_ptr);
#endif
	fillProxyList();
}

void KviServerDialog::editProxy()
{
	KviIrcProxyListItem *it = (KviIrcProxyListItem *) m_pProxyList->currentItem();
	if( it )
		editProxyField(it, ProxyHost);
}

void KviServerDialog::newProxy()
{
	KviIrcProxy *prx = new KviIrcProxy();
	prx->szHost     = "proxy.localdomain";
	prx->szUsername = "";
	prx->szPort     = "1080";
	prx->szPassword = "";
	m_pProxyManager->appendProxy(prx);
	fillProxyList();
}

void KviServerDialog::editProxyField(KviIrcProxyListItem *it, KviOptDlgProxyField field)
{
	__range_valid(it);
	__range_invalid(m_pProxyEditor->isVisible());
	QRect rct      = m_pProxyList->itemRect(it);
	int fieldWidth = m_pProxyList->columnWidth(field);
	int fieldX     = -m_pProxyList->contentsX();
	for( int i = 0; i < field; i++ )
		fieldX += m_pProxyList->columnWidth(i);
	m_pProxyEditor->move(fieldX, rct.top());
	m_pProxyEditor->resize(fieldWidth, rct.height());
	QString field_ptr("");
	switch( field ) {
		case ProxyHost:     field_ptr = it->m_ptr->szHost.ptr();     break;
		case ProxyPort:     field_ptr = it->m_ptr->szPort.ptr();     break;
		case ProxyUsername: field_ptr = it->m_ptr->szUsername.ptr(); break;
		case ProxyPassword: field_ptr = it->m_ptr->szPassword.ptr(); break;
		case ProxyIP: {
			// TODO: Only showing the first IP address for now. Adding a
			// dropdown for all the aliases might clutter the interface
			// overly much
			QHostAddress addr = it->m_ptr->addresses.first();
			if( !addr.isNull() )
				field_ptr = addr.toString().ascii();
			break;
		}
		default:
			return;
			break;
	}
	m_curEditedProxyField = field;
	m_curEditedProxyItem  = it;
	m_pProxyEditor->edit(field_ptr.ascii());
}

void KviServerDialog::proxyEditFinished(const QString &text)
{
	KviStr szT(text);

	__range_valid(m_curEditedProxyItem);
	switch( m_curEditedProxyField ) {
		case ProxyHost:
			szT.stripWhiteSpace();
			if( (szT.findFirstIdx('.') != -1) && (szT.findFirstIdx(' ') == -1) ) {
				// Must contain at least one dot and no spaces
				m_curEditedProxyItem->m_ptr->szHost = szT.ptr();
				m_curEditedProxyItem->setText(0, szT.ptr());
			}
			break;
		case ProxyPort:
			szT.stripWhiteSpace();
			if( !szT.isUnsignedNum() )
				szT = "1080"; // Reasonable default
			m_curEditedProxyItem->m_ptr->szPort = szT.ptr();
			m_curEditedProxyItem->setText(1, szT.ptr());
			break;
		case ProxyUsername:
			m_curEditedProxyItem->m_ptr->szUsername = szT.ptr();
			m_curEditedProxyItem->setText(2, szT.ptr());
			break;
		case ProxyPassword:
			m_curEditedProxyItem->m_ptr->szPassword = szT.ptr();
			m_curEditedProxyItem->setText(3, szT.ptr());
			break;
		case ProxyIP: {
			szT.stripWhiteSpace();
			QHostAddress addr;
			m_curEditedProxyItem->m_ptr->addresses.clear();
			if( !addr.setAddress(szT.ptr()) )
				m_curEditedProxyItem->setText(4, "");
			else {
				m_curEditedProxyItem->m_ptr->addresses.append(addr);
				m_curEditedProxyItem->setText(4, szT.ptr());
			}
			break;
		}
		default:
			return;
			break;
	}
	m_pProxyList->setFocus();
}

void KviServerDialog::currentProxyChanged(KviListViewItem *it)
{
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pProxyManager->setCurrentProxy(((KviIrcProxyListItem *) it)->m_ptr));
#else
	m_pProxyManager->setCurrentProxy(((KviIrcProxyListItem *) it)->m_ptr);
#endif
}

void KviServerDialog::proxyDoubleClicked(KviListViewItem *i)
{
	KviIrcProxyListItem *it = (KviIrcProxyListItem *) i;
	QPoint pnt = QCursor::pos();
	pnt = m_pProxyList->viewport()->mapFromGlobal(pnt);
	int xLeft =m_pProxyList->columnWidth(0) - m_pServerList->contentsX();
	if( pnt.x() < xLeft )
		editProxyField(it, ProxyHost);
	else {
		xLeft += m_pProxyList->columnWidth(1); // Port
		if( pnt.x() < xLeft )
			editProxyField(it, ProxyPort);
		else {
			xLeft += m_pProxyList->columnWidth(2); // Username
			if( pnt.x() < xLeft )
				editProxyField(it, ProxyUsername);
			else {
				xLeft += m_pProxyList->columnWidth(3); // Password
				if( pnt.x() < xLeft )
					editProxyField(it, ProxyPassword);
				else {
					xLeft += m_pProxyList->columnWidth(4); // IP
					if( pnt.x() < xLeft )
						editProxyField(it, ProxyIP);
				}
			}
		}
	}
}

#include "m_kvi_serverdialog.moc"
