// =============================================================================
//
//      --- kvi_userlistbox.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserListBox"

#define _KVI_USERLISTBOX_CPP_

#include <qdragobject.h>
#include <qpainter.h>
#include <qscrollbar.h>

#include "kvi_app.h"
#include "kvi_debug.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_irc_user.h"
#include "kvi_irc_userchanlist.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_statusbar.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"

// Declared in kvi_app.cpp
extern QPixmap *g_pListBoxOpPixmap;
extern QPixmap *g_pListBoxVoicePixmap;
extern QPixmap *g_pListBoxHelpOpPixmap;
extern QPixmap *g_pListBoxUserOpPixmap;
extern QPixmap *g_pListBoxOwnerPixmap;
extern KviApp  *g_pApp;

extern KviEventManager *g_pEventManager;

KviUserListBox::KviUserListBox(QWidget *parent, KviWindow *parentWnd, KviUserParser *parser, KviFrame *pFrm)
	: QWidget(parent, "userlist_box")
{
	m_pMemBuffer = new QPixmap(width(), height());
	m_pScrollBar = new QScrollBar(this, "userlist_scrollbar");
	connect(m_pScrollBar, SIGNAL(valueChanged(int)), this, SLOT(scrollBarMoved(int)));
	m_pScrollBar->hide();
	m_pFrm        = pFrm;
	m_pClientList = new KviIrcUserChanList(pFrm->m_pUserList);
	m_iCurTopItem = 0;
	m_pParent     = parentWnd;
	m_pUserParser = parser;
	setAcceptDrops(true);
	QSize s(KVI_USERLISTBOX_MINIMUM_WIDTH, KVI_USERLISTBOX_MINIMUM_HEIGHT);
	setMinimumSize(s);
}

KviUserListBox::~KviUserListBox()
{
	delete m_pMemBuffer;  m_pMemBuffer  = 0;
	delete m_pClientList; m_pClientList = 0;
}

void KviUserListBox::wheelEvent(QWheelEvent *e)
{
	g_pApp->sendEvent(m_pScrollBar, e);
}

QPtrList<KviStr> *KviUserListBox::createNickStringList()
{
	QPtrList<KviStr> *ret = new QPtrList<KviStr>;
	ret->setAutoDelete(true);
	for( KviIrcUser *u = m_pClientList->firstUser(); u; u = m_pClientList->nextUser() )
		ret->append(new KviStr(u->nick()));
	return ret;
}

KviIrcUser *KviUserListBox::firstOp()
{
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->oFlag )
			return data->pNode->pUser;
	}
	return 0;
}

void KviUserListBox::join(const KviIrcUser &user, char bOp, char bVoice, char bHalfOp, char bUserOp, char bOwner, bool bRepaint)
{
	m_pClientList->join(user, bOp, bVoice, bHalfOp, bUserOp, bOwner);
	if( bRepaint ) {
		updateScrollBar();
		paintEvent(0);
	}
}

bool KviUserListBox::nickChange(const KviIrcUser &nicker, const char *nick)
{
	bool bRet = m_pClientList->nickChange(nicker, nick);
	if( bRet ) {
		updateScrollBar();
		paintEvent(0);
	}
	return bRet;
}

void KviUserListBox::part(const char *nick)
{
	m_pClientList->part(nick);
	updateScrollBar();
	paintEvent(0);
}

void KviUserListBox::part(const KviIrcUser &user)
{
	m_pClientList->part(user);
	updateScrollBar();
	paintEvent(0);
}

void KviUserListBox::partAll()
{
	m_pClientList->clearList();
	updateScrollBar();
	paintEvent(0);
}

void KviUserListBox::select(const char *nick)
{
	if( !m_pClientList->select(nick) )
		return;
	KviIrcUserChanData *data = m_pClientList->findData(nick);
	KviStr szMask;
	data->pNode->pUser->mask(szMask);
	m_pFrm->m_pStatusBar->tempText(szMask.ptr(), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME);
	paintEvent(0);
}

void KviUserListBox::select(const KviIrcUser &user)
{
	if( !m_pClientList->select(user) )
		return;
	KviIrcUserChanData *data = m_pClientList->findData(user.nick());
	KviStr szMask;
	data->pNode->pUser->mask(szMask);
	m_pFrm->m_pStatusBar->tempText(szMask.ptr(), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME);
	paintEvent(0);
}

void KviUserListBox::selectAll()
{
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() )
		data->extFlag = 1;
	paintEvent(0);
}

void KviUserListBox::deselect(const char *nick)
{
	m_pClientList->deselect(nick);
	paintEvent(0);
}

void KviUserListBox::deselect(const KviIrcUser &user)
{
	if( !m_pClientList->deselect(user) )
		return;
	paintEvent(0);
}

void KviUserListBox::deselectAll()
{
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() )
		data->extFlag = 0;
	paintEvent(0);
}

bool KviUserListBox::owner(const KviIrcUser &user, char bOwner)
{
	bool bRet = m_pClientList->owner(user, bOwner);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::owner(const char *nick, char bOwner)
{
	bool bRet = m_pClientList->op(nick, bOwner);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::op(const KviIrcUser &user, char bOp)
{
	bool bRet = m_pClientList->op(user, bOp);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::op(const char *nick, char bOp)
{
	bool bRet = m_pClientList->op(nick, bOp);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::halfop(const KviIrcUser &user, char bHalfOp)
{
	bool bRet = m_pClientList->halfop(user, bHalfOp);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::halfop(const char *nick, char bHalfOp)
{
	bool bRet = m_pClientList->halfop(nick, bHalfOp);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::voice(const KviIrcUser &user, char bVoice)
{
	bool bRet = m_pClientList->voice(user, bVoice);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::voice(const char *nick, char bVoice)
{
	bool bRet = m_pClientList->voice(nick, bVoice);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::userop(const KviIrcUser &user, char bUserOp)
{
	bool bRet = m_pClientList->userop(user, bUserOp);
	paintEvent(0);
	return bRet;
}

bool KviUserListBox::userop(const char *nick, char bUserOp)
{
	bool bRet = m_pClientList->userop(nick, bUserOp);
	paintEvent(0);
	return bRet;
}

void KviUserListBox::scrollBarMoved(int value)
{
	m_iCurTopItem = value;
	if( m_iCurTopItem > (m_pClientList->count() - 1) )
		m_iCurTopItem = m_pClientList->count() - 1;
	if( m_iCurTopItem < 0 )
		m_iCurTopItem = 0;
	paintEvent(0);
}

bool KviUserListBox::isOwner(const KviIrcUser &user)
{
	return m_pClientList->isOwner(user);
}

bool KviUserListBox::isOwner(const char *nick)
{
	return m_pClientList->isOwner(nick);
}

bool KviUserListBox::isOp(const KviIrcUser &user)
{
	return m_pClientList->isOp(user);
}

bool KviUserListBox::isOp(const char *nick)
{
	return m_pClientList->isOp(nick);
}

bool KviUserListBox::isHalfOp(const KviIrcUser &user)
{
	return m_pClientList->isHalfOp(user);
}

bool KviUserListBox::isHalfOp(const char *nick)
{
	return m_pClientList->isHalfOp(nick);
}

bool KviUserListBox::isVoice(const KviIrcUser &user)
{
	return m_pClientList->isVoice(user);
}

bool KviUserListBox::isVoice(const char *nick)
{
	return m_pClientList->isVoice(nick);
}

bool KviUserListBox::isUserOp(const KviIrcUser &user)
{
	return m_pClientList->isUserOp(user);
}

bool KviUserListBox::isUserOp(const char *nick)
{
	return m_pClientList->isUserOp(nick);
}

int KviUserListBox::count()
{
	return m_pClientList->count();
}

int KviUserListBox::ownerCount()
{
	return m_pClientList->ownerCount();
}

int KviUserListBox::opCount()
{
	return m_pClientList->opCount();
}

int KviUserListBox::halfopCount()
{
	return m_pClientList->halfopCount();
}

int KviUserListBox::voiceCount()
{
	return m_pClientList->voiceCount();
}

int KviUserListBox::useropCount()
{
	return m_pClientList->useropCount();
}

int KviUserListBox::findUserPosition(const char *nick)
{
	return m_pClientList->findUserPosition(nick);
}

KviIrcUser *KviUserListBox::findUser(const KviIrcUser &user)
{
	return m_pClientList->findUser(user);
}

KviIrcUser *KviUserListBox::findUser(const char *nick)
{
	return m_pClientList->findUser(nick);
}

KviIrcUser *KviUserListBox::firstUser()
{
	return m_pClientList->firstUser();
}

KviIrcUser *KviUserListBox::lastUser()
{
	return m_pClientList->lastUser();
}

KviIrcUser *KviUserListBox::nextUser()
{
	return m_pClientList->nextUser();
}

KviIrcUser *KviUserListBox::prevUser()
{
	return m_pClientList->prevUser();
}

KviIrcUserChanList *KviUserListBox::userList()
{
	return m_pClientList;
}

void KviUserListBox::doRepaint()
{
	updateScrollBar();
	paintEvent(0);
}

void KviUserListBox::paintEvent(QPaintEvent *)
{
	if( !isVisible() )
		return;
	int widgetWidth  = m_pScrollBar->isVisible() ? width() - KVI_USERLISTBOX_SCROLLBAR_WIDTH : width();
	int widgetHeight = height();
	QPainter pa(m_pMemBuffer);

	// Draw the background
	if( g_pOptions->m_pixListBoxBack->isNull() )
		pa.fillRect(0, 0, widgetWidth, widgetHeight, g_pOptions->m_clrListBoxBack);
	else
		pa.drawTiledPixmap(0, 0, widgetWidth, widgetHeight, *(g_pOptions->m_pixListBoxBack));

	// Bottom of the cell to paint
	int yCurCellBottom =
		KVI_USERLISTBOX_VERTICAL_BORDER + g_pOptions->m_iListBoxCellHeight - g_pOptions->m_iListBoxBaseLineOffset;
	// Maximum y coordinate for yCurCellBottom
	int yBottomBorder = widgetHeight - KVI_USERLISTBOX_VERTICAL_BORDER;
	// x coordinate of the text
	int xLeft = KVI_USERLISTBOX_HORIZONTAL_BORDER;
	// Increase it if images are enabled, and if there are ops or voiced users inside
	if( g_pOptions->m_bListBoxShowImages &&
	    (m_pClientList->opCount() || m_pClientList->voiceCount() || m_pClientList->halfopCount() ||
	     m_pClientList->useropCount() || m_pClientList->ownerCount())
	) {
		xLeft += 18;
	}
	// Available space for text
	int availableXSpace = widgetWidth - (KVI_USERLISTBOX_HORIZONTAL_BORDER + xLeft);

	if( m_pClientList->count() &&
		(widgetHeight > (g_pOptions->m_iListBoxCellHeight + (KVI_USERLISTBOX_VERTICAL_BORDER << 1))) &&
		(widgetWidth  > (xLeft + KVI_USERLISTBOX_HORIZONTAL_BORDER))
	) {
		// Have items inside and have vertical and horizontal space for at least one item
		KviIrcUserChanData *data = m_pClientList->firstData();

		if( m_iCurTopItem > 0 ) {
			// Skip the first items that are not visible
			__range_valid(m_iCurTopItem < m_pClientList->count());
			__range_valid(m_pScrollBar->isVisible());
			for( int i = 0; data && (i < m_iCurTopItem); i++ ) {
				data = m_pClientList->nextData();
			}
		}

		if( data ) {
			pa.setFont(g_pOptions->m_fntListBox);
			// Now we point to the first item to paint
			for( ; data && (yCurCellBottom < yBottomBorder); data = m_pClientList->nextData() ) {
				if( g_pOptions->m_bListBoxShowImages &&
				   ((data->oFlag != 0) || (data->vFlag != 0) || (data->hFlag != 0) ||
				    (data->uFlag != 0) || (data->qFlag != 0))
				) {
					// Paint the pixmap first
					// Calculate the position of the image
					int imageYPos =
						yCurCellBottom + g_pOptions->m_iListBoxBaseLineOffset
						- ((g_pOptions->m_iListBoxCellHeight + 16) >> 1);
					// Set the mask if needed
					QPixmap *pix;
					if( data->qFlag != 0 )      pix = g_pListBoxOwnerPixmap;
					else if( data->oFlag != 0 ) pix = g_pListBoxOpPixmap;
					else if( data->hFlag != 0 ) pix = g_pListBoxHelpOpPixmap;
					else if( data->vFlag != 0 ) pix = g_pListBoxVoicePixmap;
					else pix = g_pListBoxUserOpPixmap;
					if( pix->mask() ) {
						pa.setClipping(true);
						pa.setClipRect(KVI_USERLISTBOX_HORIZONTAL_BORDER, imageYPos, widgetWidth, widgetHeight);
					}
					// Draw the pixmap
					bitBlt(m_pMemBuffer, KVI_USERLISTBOX_HORIZONTAL_BORDER, imageYPos, pix, 0, 0, 16, 16);
					pa.setClipping(false);
				}
				// Now the text
				if( data->extFlag != 0 ) {
					// Selected item
					pa.fillRect(
						xLeft,
						(yCurCellBottom + g_pOptions->m_iListBoxBaseLineOffset) - g_pOptions->m_iListBoxCellHeight,
						availableXSpace,
						g_pOptions->m_iListBoxCellHeight,
						g_pOptions->m_clrListBoxSeleBack
					);
					pa.setPen(g_pOptions->m_clrListBoxSeleFore);
					const char *str = data->pNode->pUser->nick();
					pa.drawText(xLeft, yCurCellBottom, str, strlen(str));
				} else {
					if( data->qFlag != 0 )      pa.setPen(g_pOptions->m_clrListBoxOwner);
					else if( data->oFlag != 0 ) pa.setPen(g_pOptions->m_clrListBoxOp);
					else if( data->hFlag != 0 ) pa.setPen(g_pOptions->m_clrListBoxHalfOp);
					else if( data->vFlag != 0 ) pa.setPen(g_pOptions->m_clrListBoxVoice);
					else if( data->uFlag != 0 ) pa.setPen(g_pOptions->m_clrListBoxUserOp);
					else pa.setPen(g_pOptions->m_clrListBoxNormal);
					const char *str = data->pNode->pUser->nick();
					pa.drawText(xLeft, yCurCellBottom, str, strlen(str));
				}
				yCurCellBottom += g_pOptions->m_iListBoxCellHeight;
			}
		}
	}

	// Need to draw the sunken rectangle around the view now...
	pa.setPen(QPen(colorGroup().dark(), 1));
	pa.drawLine(0, 0, widgetWidth, 0);
	pa.drawLine(0, 0, 0, widgetHeight);
	pa.setPen(colorGroup().light());
	pa.drawLine(1, widgetHeight - 1, widgetWidth - 1, widgetHeight - 1);
	pa.drawLine(widgetWidth - 1, 1, widgetWidth - 1, widgetHeight);

	// Copy to the display
	bitBlt(this, 0, 0, m_pMemBuffer, 0, 0, widgetWidth, widgetHeight);
}

void KviUserListBox::resizeEvent(QResizeEvent *)
{
	m_pMemBuffer->resize(width(), height());
	updateScrollBar();
}

void KviUserListBox::updateScrollBar()
{
	// Update the scrollbar
	int maxItems = (height() - (KVI_USERLISTBOX_VERTICAL_BORDER << 1)) / g_pOptions->m_iListBoxCellHeight;
	if( maxItems < 0 )
		maxItems = 0;
	if( maxItems < m_pClientList->count() ) {
		m_pScrollBar->setGeometry(width() - KVI_USERLISTBOX_SCROLLBAR_WIDTH, 0, KVI_USERLISTBOX_SCROLLBAR_WIDTH, height());
		m_pScrollBar->setRange(0, m_pClientList->count() - maxItems);
		if( !m_pScrollBar->isVisible() ) {
			m_iCurTopItem = 0;
			// TODO: check the width for the scrollbar?
			m_pScrollBar->show();
		} else {
			// TODO: check this: scrollbar scrolled to the end, wait for some nicks to leave and check the results...
			if( m_iCurTopItem > (m_pClientList->count() - maxItems) )
				m_iCurTopItem = m_pClientList->count() - maxItems;
		}
	} else {
		if( m_pScrollBar->isVisible() )
			m_pScrollBar->hide();
		m_iCurTopItem = 0;
	}
}

void KviUserListBox::mousePressEvent(QMouseEvent *e)
{
	if( e->button() & (LeftButton | RightButton) ) {
		// Selection handling
		int item = itemFromPos(e->pos());
		if( item >= 0 ) {
			m_iLastSelectedItem = item;
			if( !(e->state() & ShiftButton) ) {
				// Deselect all
				for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
					data->extFlag = 0;
				}
			}
			// Select-invert this item
			KviIrcUserChanData *data = m_pClientList->firstData();
			for( int i = 0; data && (i < item); i++ )
				data = m_pClientList->nextData();
			if( data ) {
				if( data->extFlag == 0 )
					data->extFlag = 1;
				else
					data->extFlag = (e->button() & LeftButton) ? 0 : 1; // Right button does not unselect!
				KviStr szMask;
				data->pNode->pUser->mask(szMask);
				m_pFrm->m_pStatusBar->tempText(szMask.ptr(), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME);
			}
			paintEvent(0);
		}
		m_iLastSelectedItem = item;
	}
	// Pop up if it was the right button and we clicked a visible item...
	if( (e->button() & RightButton) && (m_iLastSelectedItem >= 0) )
		emit rightClicked();
	// TODO: and middleButtonPress?
}

void KviUserListBox::mouseDoubleClickEvent(QMouseEvent *e)
{
	if( e->button() & (LeftButton) ) {
		// Selection handling
		int item = itemFromPos(e->pos());
		if( item >= 0 ) {
			m_iLastSelectedItem = item;
			if( !(e->state() & ShiftButton) ) {
				// Deselect all
				for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
					data->extFlag = 0;
				}
			}
			// Select-invert this item
			KviIrcUserChanData *data = m_pClientList->firstData();
			for( int i = 0; data && (i < item); i++ )
				data = m_pClientList->nextData();
			if( data ) {
				if( data->extFlag == 0 )
					data->extFlag = 1;
				KviStr szMask;
				data->pNode->pUser->mask(szMask);
				m_pFrm->m_pStatusBar->tempText(szMask.ptr(), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME);
				bool bHalted = false;
				if( g_pEventManager->eventEnabled(KviEvent_OnNickDoubleClicked) ) {
					KviStr tmp = data->pNode->pUser->nick();
					bHalted = m_pUserParser->callEvent(KviEvent_OnNickDoubleClicked, m_pParent, tmp);
				}
				if( !bHalted )
					m_pFrm->raiseOrCreateQuery(data->pNode->pUser->nick());
			}
			paintEvent(0);
		}
		m_iLastSelectedItem = item;
	}
}

void KviUserListBox::mouseMoveEvent(QMouseEvent *e)
{
	if( e->state() & LeftButton ) {
		// Selection handling
		int item = itemFromPos(e->pos());
		if( item >=0 && (item != m_iLastSelectedItem) ) { // Do not do it more than once for each item
			if( !(e->state() & ShiftButton) ) {
				// Deselect all
				for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
					data->extFlag = 0;
				}
			}
			// Select-invert this item
			KviIrcUserChanData *data = m_pClientList->firstData();
			for( int i = 0; data && (i < item); i++ )
				data = m_pClientList->nextData();
			if( data ) {
				if( data->extFlag == 0 )
					data->extFlag = 1;
				KviStr szMask;
				data->pNode->pUser->mask(szMask);
				m_pFrm->m_pStatusBar->tempText(szMask.ptr(), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME);
			}
			paintEvent(0);
		}
		m_iLastSelectedItem = item;
	}
}

void KviUserListBox::dragEnterEvent(QDragEnterEvent *e)
{
	e->accept(m_pClientList->count() && QUriDrag::canDecode(e));
}

void KviUserListBox::dragMoveEvent(QDragMoveEvent *e)
{
	if( m_pClientList->count() == 0 )
		return;
	int item = itemFromPos(e->pos());
	if( item != m_iLastSelectedItem ) { // Do not do it more than once for each item
		// Deselect all
		for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
			data->extFlag = 0;
		}
		if( item >=0 ) {
			// Select-invert this item
			KviIrcUserChanData *data = m_pClientList->firstData();
			for( int i = 0; data && (i < item); i++ )
				data = m_pClientList->nextData();
			if( data ) {
				data->extFlag = 1;
				KviStr szMask;
				data->pNode->pUser->mask(szMask);
				szMask.prepend(__tr("Drop the file to DCC SEND it to "));
				m_pFrm->m_pStatusBar->tempText(szMask.ptr(), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME);
			}
		} else {
			m_pFrm->m_pStatusBar->tempText(
				__tr("Drop the file over a nickname to start the DCC SEND"), KVI_USERLISTBOX_USERMASK_STATUSBAR_SHOW_TIME
			);
		}
		paintEvent(0);
	}
	m_iLastSelectedItem = item;
}

void KviUserListBox::dropEvent(QDropEvent *e)
{
	if( m_pClientList->count() == 0 )
		return;
	int item = itemFromPos(e->pos());
	// Deselect all
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		data->extFlag = 0;
	}
	if( item >=0 ) {
		// Select-invert this item
		KviIrcUserChanData *data = m_pClientList->firstData();
		for( int i = 0; data && (i < item); i++ )
			data = m_pClientList->nextData();
		if( data ) {
			data->extFlag = 1;
			QStringList list;
			if( QUriDrag::decodeLocalFiles(e, list) ) {
				if( !list.isEmpty() ) {
					QStringList::ConstIterator it = list.begin();
					for( ; it != list.end(); ++it ) {
						QString tmp = *it;
						tmp.prepend(" ");
						tmp.prepend(data->pNode->pUser->nick());
						tmp.prepend("/DCC SEND ");
						m_pUserParser->parseUserCommand(tmp, m_pParent);
					}
				}
			}
		}
	}
	paintEvent(0);
	m_iLastSelectedItem = item;
}

void KviUserListBox::mouseReleaseEvent(QMouseEvent *e)
{
	m_iLastSelectedItem = -1;
}

int KviUserListBox::itemFromPos(QPoint pos)
{
	pos.setY(pos.y() - KVI_USERLISTBOX_VERTICAL_BORDER);
	int item = (pos.y() / g_pOptions->m_iListBoxCellHeight) + m_iCurTopItem;
	if( (item < 0) || (item >= m_pClientList->count()) )
		item = -1;
	return item;
}

void KviUserListBox::appendNickList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( bFirstDone )
			buffer.append(',');
		else
			bFirstDone = true;
		buffer.append(data->pNode->pUser->nick());
	}
}

void KviUserListBox::appendOwnerList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->qFlag != 0 ) { // Owner
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendOpList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->oFlag != 0 ) { // Op
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendNOpList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->oFlag == 0 ) { // Non-Op
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendHalfOpList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->hFlag != 0 ) { // Halfop
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendVoiceList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->vFlag != 0 ) { // Voice
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendUserOpList(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->uFlag != 0 ) { // Userop
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendSelectedNicknames(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->extFlag != 0 ) { // Selected
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
		}
	}
}

void KviUserListBox::appendSelectedMasks(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->extFlag != 0 ) { // Selected
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->nick());
			buffer.append('!');
			buffer.append(data->pNode->pUser->username());
			buffer.append('@');
			buffer.append(data->pNode->pUser->host());
		}
	}
}

void KviUserListBox::appendSelectedHosts(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->extFlag != 0 ) { // Selected
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->host());
		}
	}
}

void KviUserListBox::appendSelectedUsernames(KviStr &buffer)
{
	bool bFirstDone = false;
	for( KviIrcUserChanData *data = m_pClientList->firstData(); data; data = m_pClientList->nextData() ) {
		if( data->extFlag != 0 ) { // Selected
			if( bFirstDone )
				buffer.append(',');
			else
				bFirstDone = true;
			buffer.append(data->pNode->pUser->username());
		}
	}
}

#include "m_kvi_userlistbox.moc"
