// =============================================================================
//
//      --- kvi_usertoolbartemplate.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserToolbarTemplate"

#include <qfile.h>

#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_string.h"
#include "kvi_userpopupmenu.h"
#include "kvi_usertoolbar.h"
#include "kvi_usertoolbartemplate.h"
#include "kvi_usertoolbutton.h"

// From kvi_options.cpp:
extern KviUserToolBarTemplate *g_pUserToolBarTemplate;

bool KviUserToolBarTemplate::loadToolBarTemplate(const char *filename, KviStr &error)
{
	KviStr buffer;
	if( !kvi_loadFile(filename, buffer) ) {
		error = _i18n_("Unable to load file");
		return false;
	}

	if( !kvi_strEqualCSN(KVI_MAGIC_STRING, buffer.ptr(), KVI_MAGIC_STRING_LENGTH) ) {
		error = _i18n_("Invalid magic in KVIrc configuration file");
		return false; // Not a KVIrc file
	}

	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);
	while( buffer.hasData() ) {
		buffer.stripLeftWhiteSpace();
		if( !kvi_strEqualCSN("[TOOLBAR", buffer.ptr(), 8) ) {
			error = _i18n_("Invalid TOOLBAR label");
			return false;
		}

		buffer.cutLeft(8);
		buffer.stripLeftWhiteSpace();

		KviUserToolBarTemplate *target = g_pUserToolBarTemplate;
		if( !kvi_strEqualNoLocaleCIN("bytes ", buffer.ptr(), 6) ) {
			error = __tr("Syntax error in [TOOLBAR] label");
			return false;
		}

		buffer.cutLeft(6);
		buffer.stripLeftWhiteSpace();

		if( !isdigit(*(buffer.ptr())) ) {
			error = _i18n_("Syntax error in [TOOLBAR] label (digit expected)");
			return false;
		}

		KviStr numBytes;
		if( !buffer.getToken(numBytes, ']') ) {
			error = _i18n_("Syntax error in [TOOLBAR] label (cannot extract toolbar length)");
			__debug(buffer.ptr());
			return false;
		}

		bool bOk  = false;
		int bytes = numBytes.toInt(&bOk);
		if( !bOk ) { // Syntax error; someone messed with the file
			error = _i18n_("Unrecogized token ");
			error += numBytes;
			__debug(buffer.ptr());
			return false;
		}

		buffer.cutLeft(1); // Cut the newline
		if( buffer.len() < bytes ) {
			error = _i18n_("Toolbar buffer smaller than declared length (");
			error += numBytes;
			error += ")";
			__debug(buffer.ptr());
			return false;
		}

		KviStr toolbarBuf = buffer.left(bytes);
		buffer.cutLeft(bytes);

		buffer.stripLeftWhiteSpace();

		if( !kvi_strEqualCSN("[ENDTOOLBAR]\n", buffer.ptr(), 13) ) {
			error = __tr("Something went wrong with the toolbar buffer length");
			__debug(buffer.ptr());
			return false;
		}
		buffer.cutLeft(13);

		if( !target->loadFromDataBuffer(toolbarBuf, error) ) {
			__debug(toolbarBuf.ptr());
			return false;
		}
	}
	return true;
}

bool KviUserToolBarTemplate::loadFromDataBuffer(KviStr &buffer, KviStr &error)
{
	clearToolBar();
	buffer.stripLeftWhiteSpace();

	while( buffer.hasData() ) {
		if( kvi_strEqualNoLocaleCIN("[separator]\n", buffer.ptr(), 12) ) {
			buffer.cutLeft(12);
			insertSeparatorItem();
		} else if( kvi_strEqualNoLocaleCIN("[button]\n", buffer.ptr(), 9) ) {
			buffer.cutLeft(9);
			KviStr image, name, sbar, bytes;
			if( !buffer.getLine(image) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if( !buffer.getLine(name) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if( !buffer.getLine(sbar) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if( !buffer.getLine(bytes) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}

			if( !kvi_strEqualNoLocaleCIN("image=", image.ptr(), 6) ) {
				error = __tr("Unrecognized item image label");
				return false;
			}
			image.cutLeft(6);
			image.stripWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("name=", name.ptr(), 5) ) {
				error = __tr("Unrecognized item name label");
				return false;
			}
			name.cutLeft(5);
			name.stripWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("statusbar=", sbar.ptr(), 10) ) {
				error = __tr("Unrecognized item statusbar label");
				return false;
			}
			sbar.cutLeft(10);
			sbar.stripWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("bytes=", bytes.ptr(), 6) ) {
				error = __tr("Unrecognized item length label");
				return false;
			}
			bytes.cutLeft(6);
			bytes.stripWhiteSpace();
			bool bOk = false;
			int bLen = bytes.toInt(&bOk);
			if( !bOk ) {
				error = __tr("Unrecognized item length");
				return false;
			}
			if( (bLen < 0) || (bLen > buffer.len()) ) {
				error = __tr("Buffer is shorter than the declared length");
				return false;
			}
			KviStr pBuf = buffer.left(bLen);
			buffer.cutLeft(bLen);
			buffer.stripLeftWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("[endbutton]\n", buffer.ptr(), 12) ) {
				error = __tr("Somethng went wrong with the buffer length");
				return false;
			}
			buffer.cutLeft(12);
			insertNormalItem(name.ptr(), sbar.ptr(), image.ptr(), pBuf.ptr());
		} else if( kvi_strEqualNoLocaleCIN("[menu]\n", buffer.ptr(), 7) ) {
			buffer.cutLeft(7);
			KviStr image, name, sbar, bytes;
			if( !buffer.getLine(image) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if( !buffer.getLine(name) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if( !buffer.getLine(sbar) ){
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if( !buffer.getLine(bytes) ) {
				error = __tr("Unexpected end of buffer");
				return false;
			}

			if( !kvi_strEqualNoLocaleCIN("image=", image.ptr(), 6) ) {
				error = __tr("Unrecognized menu image label");
				return false;
			}
			image.cutLeft(6);
			image.stripWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("name=", name.ptr(), 5) ) {
				error = __tr("Unrecognized menu name label");
				return false;
			}
			name.cutLeft(5);
			name.stripWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("statusbar=", sbar.ptr(), 10) ) {
				error = __tr("Unrecognized item statusbar label");
				return false;
			}
			sbar.cutLeft(10);
			sbar.stripWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("bytes=", bytes.ptr(), 6) ) {
				error = __tr("Unrecognized menu length label");
				return false;
			}
			bytes.cutLeft(6);
			bytes.stripWhiteSpace();
			bool bOk = false;
			int bLen = bytes.toInt(&bOk);
			if( !bOk ) {
				error = __tr("Unrecognized menu length");
				return false;
			}
			if( (bLen < 0) || (bLen > buffer.len()) ) {
				error = __tr("Buffer is shorter than the declared length");
				return false;
			}
			KviStr mBuf = buffer.left(bLen);
			buffer.cutLeft(bLen);
			buffer.stripLeftWhiteSpace();
			if( !kvi_strEqualNoLocaleCIN("[endmenu]\n", buffer.ptr(), 10) ) {
				error = __tr("Something went wrong with the buffer length");
				return false;
			}
			buffer.cutLeft(10);

			KviStr yourName(KviStr::Format, "usertoolbar_script_popup_%s", name.ptr());
			KviUserPopupMenu *menu = new KviUserPopupMenu(0, yourName.ptr());
			if( !menu->loadFromDataBuffer(mBuf, error) ) {
				delete menu;
				return false;
			}

			insertNormalItem(name.ptr(), sbar.ptr(), image.ptr(), 0, menu);
		} else {
			error = __tr("Unrecognized toolbar item");
			return false;
		}
		buffer.stripLeftWhiteSpace();
	}
	return true;
}

bool KviUserToolBarTemplate::saveToolBarTemplate(const char *filename)
{
	QFile f(filename);
	if( !f.open(IO_WriteOnly | IO_Truncate) )
		return false;

	f.writeBlock(KVI_MAGIC_STRING, KVI_MAGIC_STRING_LENGTH);
	g_pUserToolBarTemplate->save(&f);

	f.close();
	return true;
}

void KviUserToolBarTemplate::save(QFile *f)
{
	f->writeBlock("[TOOLBAR", 8);
	f->writeBlock(" bytes ", 7);

	KviStr buffer;
	buildDataBuffer(buffer);

	KviStr szNBytes(KviStr::Format, "%d", buffer.len());
	f->writeBlock(szNBytes.ptr(), szNBytes.len());
	f->writeBlock("]\n", 2);
	f->writeBlock(buffer.ptr(), buffer.len());
	f->writeBlock("\n[ENDTOOLBAR]\n", 14);
}

void KviUserToolBarTemplate::buildDataBuffer(KviStr &str)
{
	for( KviUserToolBarData *d = m_pDataList->first(); d; d = m_pDataList->next() ) {
		switch( d->type ) {
			case KviUserToolBar::Separator:
				str.append("[separator]\n");
				continue;
			case KviUserToolBar::Button: {
				str.append("[button]\n");
				str.append("image=");
				if( d->image.hasData() )
					str.append(d->image);
				str.append("\nname=");
				str.append(d->text);
				str.append("\nstatusbar=");
				str.append(d->sbar);
				str.append("\nbytes=");
				KviStr tmp(KviStr::Format, "%d", d->buffer.len());
				str.append(tmp);
				str.append("\n");
				str.append(d->buffer);
				str.append("\n[endbutton]\n");
				continue;
			}
			case KviUserToolBar::Menu: {
				str.append("[menu]\n");
				str.append("image=");
				if( d->image.hasData() )
					str.append(d->image);
				str.append("\nname=");
				str.append(d->text);
				str.append("\nstatusbar=");
				str.append(d->sbar);
				KviStr auxBuf;
				d->menu->buildDataBuffer(auxBuf);
				str.append("\nbytes=");
				KviStr tmp(KviStr::Format, "%d", auxBuf.len());
				str.append(tmp);
				str.append("\n");
				str.append(auxBuf);
				str.append("\n[endmenu]\n");
				continue;
			}
		}
	}
}

/**
 * =======================================================
 */
KviUserToolBarTemplate::KviUserToolBarTemplate()
{
	m_pDataList = new QPtrList<KviUserToolBarData>;
	m_pDataList->setAutoDelete(true);
	m_id = 1;
}

KviUserToolBarTemplate::~KviUserToolBarTemplate()
{
	clearAll();
	delete m_pDataList;
	m_pDataList = 0;
}

void KviUserToolBarTemplate::clearAll()
{
	KviUserToolBarData *data;
	while( (data = m_pDataList->first()) ) {
		if( data->button ) {
			delete data->button;
			data->button = 0;
		}
		if( data->menu ) {
			delete data->menu;
			data->menu = 0;
		}
		m_pDataList->removeFirst();
	}
	clearToolBar();
	m_id = 1;
}

bool KviUserToolBarTemplate::isEmpty()
{
	return m_pDataList->isEmpty();
}

int KviUserToolBarTemplate::insertSeparatorItem()
{
	KviUserToolBarData *data = new KviUserToolBarData();
	data->id     = 0;
	data->buffer = " ";
	data->button = 0;
	data->text   = "";
	data->image  = "";
	data->sbar   = "";
	data->menu   = 0;
	data->type   = KviUserToolBar::Separator;
	m_pDataList->append(data);
	m_id++;
	return data->id;
}

int KviUserToolBarTemplate::insertNormalItem(
	const char *name, const char *sbar, const char *img, const char *buffer, KviUserPopupMenu *menu)
{
	KviUserToolBarData *data = new KviUserToolBarData();
	data->id     = m_id++;
	data->buffer = buffer;
	data->button = 0;
	data->text   = name;
	data->image  = img;
	data->sbar   = sbar;
	data->menu   = menu;
	if( menu )
		data->type = KviUserToolBar::Menu;
	else
		data->type = KviUserToolBar::Button;
	m_pDataList->append(data);
	return data->id;
}

void KviUserToolBarTemplate::clearToolBar()
{
	m_pDataList->clear();
}

void KviUserToolBarTemplate::copyFromToolBar(KviUserToolBar *toolbar)
{
	__range_valid(toolbar);
	clearToolBar();
	for( KviUserToolBarData *d = toolbar->m_pDataList->first(); d; d = toolbar->m_pDataList->next() ) {
		switch( d->type ) {
			case KviUserToolBar::Separator:
				insertSeparatorItem();
				break;
			case KviUserToolBar::Button:
				insertNormalItem(d->text.ptr(), d->sbar.ptr(), d->image.ptr(), d->buffer.ptr());
				break;
			case KviUserToolBar::Menu: {
				KviStr yourName(KviStr::Format, "usertoolbar_script_popup_%s", d->text.ptr());
				KviUserPopupMenu *m = new KviUserPopupMenu(0, yourName.ptr());
				m->copyFromPopup(d->menu);
				insertNormalItem(d->text.ptr(), d->sbar.ptr(), d->image.ptr(), 0, m);
				break;
			}
			default:
				break; // No other types
		}
	}
}
