// =============================================================================
//
//      --- kvi_options_charset_translation.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptionsCharsetTranslation"

#include <qlayout.h>

#include "kvi_app.h"
#include "kvi_boolselector.h"
#include "kvi_fileutils.h"
#include "kvi_lineedit.h"
#include "kvi_locale.h"
#include "kvi_options_charset_translation.h"
#include "kvi_options.h"
#include "kvi_pushbutton.h"
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	#include "kvi_charset_translator.h"
	extern KviCharsetTranslator *g_pCharsetTranslator;
#endif

/*
	@quickhelp: KviOptionsCharsetTranslation
	@widget: Charset translation options
		This options page is available only if the charset
		translation support has been enabled at compilation time.<br>
		Charset translations are useful for non english users that
		have more than one charset in their language.<br>
		Russian is a good example of this case: Linux users
		generally have the KOI charset, Windows users have
		the Windows native russian charset.<br>
		If you are going to speak on a channel where all other
		users are using the Windows charset, it may be a good idea
		to remap your KOI charset to the Windows one when writing
		to that channel and do the inverse mapping for the data that
		arrives from the server.<br>
		A translation map describes this operation.<br>
		KVIrc translation maps are compatible with the ones
		used by IRCII.<br><br>
		<b>translation map file format</b><br><br>
		The translation map is a text file with a sequence of 512 numbers.<br>
		Each number is specified as "0xYY" where YY are two hexadecimal digits.<br>
		Numbers are separated by commas and arranged in lines of 8 entries.<br>
		So the whole translation map has 64 lines of 8 hexadecimal numbers.<br>
		The first 256 numbers are used as <b>Server To Client map</b>
		and the remaining part as the <b>Client To Server map</b>.<br>
		Example:<br><br>
		The first four lines of a hypotetical translation map:<br>
		<i>
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, <br>
			0x21, 0x12, 0xa0, 0xee, 0x0c, 0x0d, 0x0e, 0x0f, <br>
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, <br>
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, <br>
			...<br>
		</i>
		These lines will be used as the first 32 entries
		in the Server-To-Client map.<br>
		So for example if the character 9 (decimal, not ASCII!!!)
		arrives from server, it will be mapped to 0x12 (that is 18 decimal).<br>
		<br><br>Notes:<br><br>
		It is absolutely easy to mess up everything when writing
		a translation map.<br>
		There are special characters that CANNOT be remapped,
		so be sure to know what are you doing!.<br>
		Usually the remapped characters are over 127 (decimal),
		and the standard ASCII set is left unchanged.<br>
*/
KviOptionsCharsetTranslation::KviOptionsCharsetTranslation(QWidget *parent)
	: KviOptionsWidget(parent, KVI_GENERAL_OPTIONS_WIDGET_ID_CHARSETTRANSLATION)
{
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	QGridLayout *g = new QGridLayout(this, 3, 2, KVI_GENERAL_OPTIONS_GRIDLAYOUT_BORDER, 1);

	KviBoolSelector *b = new KviBoolSelector(this,
		__tr("Use the following translation map:"), &(g_pOptions->m_bUseCharsetTranslation)
	);
	g->addMultiCellWidget(b, 0, 0, 0, 1);

	m_pMapFileEdit = new KviLineEdit(this);
	g->addWidget(m_pMapFileEdit, 1, 0);
	m_pMapFileEdit->setEnabled(g_pOptions->m_bUseCharsetTranslation);
	connect(b, SIGNAL(toggled(bool)), m_pMapFileEdit, SLOT(setEnabled(bool)));
	m_pMapFileEdit->setText(g_pOptions->m_szCharsetTranslationMapFile.ptr());

	m_pBrowseBtn = new KviPushButton(__tr("&Browse..."), this);
	g->addWidget(m_pBrowseBtn, 1, 1);
	m_pBrowseBtn->setEnabled(g_pOptions->m_bUseCharsetTranslation);
	connect(b, SIGNAL(toggled(bool)), m_pBrowseBtn, SLOT(setEnabled(bool)));
	connect(m_pBrowseBtn, SIGNAL(clicked()), this, SLOT(chooseTranslationMap()));

	QFrame *fr = new QFrame(this);
	g->addMultiCellWidget(fr, 2, 2, 0, 1);
	g->setRowStretch(2, 1);
#endif
}

KviOptionsCharsetTranslation::~KviOptionsCharsetTranslation()
{
	// Nothing here
}

void KviOptionsCharsetTranslation::chooseTranslationMap()
{
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	KviStr szPath;
	g_pApp->getGlobalKVIrcDirectory(szPath, KviApp::Charmaps);
	KviStr tmp = kvi_askForOpenFileName(szPath.ptr());
	if( tmp.hasData() ) {
		if( g_pCharsetTranslator->setTranslationMap(tmp.ptr(), true) ) {
			m_pMapFileEdit->setText(tmp.ptr());
		} else {
			g_pApp->warningBox(__tr("Invalid or corrupted charset translation map."));
		}
	}
#endif
}

#ifdef COMPILE_NEED_CHARSET_TRANSLATION
void KviOptionsCharsetTranslation::commitChanges()
{
	KviBoolSelector::commitAll(this);
	if( g_pOptions->m_bUseCharsetTranslation ) {
		KviStr tmp = m_pMapFileEdit->text();
		if( g_pCharsetTranslator->setTranslationMap(tmp.ptr(), true) ) {
			g_pOptions->m_szCharsetTranslationMapFile = tmp;
			g_pCharsetTranslator->setTranslationMap(tmp.ptr());
		} else {
			// Failed to read it
			g_pOptions->m_bUseCharsetTranslation      = false;
			g_pOptions->m_szCharsetTranslationMapFile = "";
			g_pApp->warningBox(__tr("Invalid or corrupted charset translation map."));
		}
	}
}
#endif

#include "m_kvi_options_charset_translation.moc"
