// =============================================================================
//
//      --- kvi_options_input.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptionsInput"

#include <qlayout.h>

#include "kvi_boolselector.h"
#include "kvi_charselector.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_options_input.h"
#include "kvi_pushbutton.h"
#include "kvi_string.h"
#include "kvi_stringselector.h"
#include "kvi_strsub.h"

/*
	@quickhelp: KviOptionsInput
	@widget: Input widget options
		Here you can set the options for the commandline input widget.<br>
		In channel windows the widget is capable of nick / mask completion (Tab / Shift + Tab).<br>
		The completed words can have a user defined postfix (like ": ").<br>
		It is possible to specify a personal command prefix to be used instead of '/'.<br>
		The widget is also capable to automatically substitute specified strings.<br>
		For example, you might want to substitute all occurrences of the string "AFAIK",
		with the expanded version "As far as I know".<br>
		To achieve this, just click the "Add new string" button.<br>
		A new entry will appear in the string substitution list box.<br>
		Double click on the left column and then type in the string "AFAIK".<br>
		In the same way type in the second column "As far as I know" and then click OK.<br>
		Now type in the commandline input "Hey, AFAIK<SPACE>" and voila!, the input line
		shows "Hey, As far as I know ".<br>
		The substitution is performed when you type in a space (to give you some visual feedback),
		period or you press the return key.<br>
*/
KviOptionsInput::KviOptionsInput(QWidget *parent)
	: KviOptionsWidget(parent, KVI_GENERAL_OPTIONS_WIDGET_ID_INPUT)
{
	QGridLayout *g = new QGridLayout(
		this, 11, 2, KVI_GENERAL_OPTIONS_GRIDLAYOUT_BORDER, KVI_GENERAL_OPTIONS_GRIDLAYOUT_SPACE
	);

	KviStringSelector *s = new KviStringSelector(
		this, __tr("Nick/mask completion postfix:"), &(g_pOptions->m_szStringToAddAfterCompletedNick), true, 180
	);
	g->addMultiCellWidget(s, 0, 0, 0, 1);

	KviBoolSelector *b = new KviBoolSelector(
		this, __tr("Apply completion postfix to first word only"), &(g_pOptions->m_bApplyCompletionPostfixForFirstWordOnly)
	);
	g->addMultiCellWidget(b, 1, 1, 0, 1);
	b = new KviBoolSelector(this, __tr("Replace whole word"), &(g_pOptions->m_bCompletionReplaceWholeWord));
	g->addMultiCellWidget(b, 2, 2, 0, 1);

	QFrame *f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f, 3, 3, 0, 1);

	b = new KviBoolSelector(
		this, __tr("Warp input cursor to end when browsing command history"), &(g_pOptions->m_bInputHistoryCursorAtEnd)
	);
	g->addMultiCellWidget(b, 4, 4, 0, 1);

	f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f, 5, 5, 0, 1);

	KviCharSelector *ch = new KviCharSelector(
		this, __tr("Personal command prefix:"), &(g_pOptions->m_cPersonalCommandPrefix), true, 20
	);
	g->addMultiCellWidget(ch, 6, 6, 0, 1);

	f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f, 7, 7, 0, 1);

	b = new KviBoolSelector(this, __tr("String substitution"), &(g_pOptions->m_bUseStringSubstitution));
	g->addMultiCellWidget(b, 8, 8, 0, 1);

	m_pStrSubView = new KviListView(this, "substitutions", true);
	m_pStrSubView->addColumn(_CHAR_2_QSTRING(__tr("String")));
	m_pStrSubView->addColumn(_CHAR_2_QSTRING(__tr("Substitute")));
	m_pStrSubView->setMinimumHeight((m_pStrSubView->fontMetrics().height())*4);
	m_pStrSubView->setSorting(-1);
	m_pStrSubView->setEnabled(g_pOptions->m_bUseStringSubstitution);

	connect(b, SIGNAL(toggled(bool)), m_pStrSubView, SLOT(setEnabled(bool)));
	g->addMultiCellWidget(m_pStrSubView, 9, 9, 0, 1);

	KviPushButton *pb = new KviPushButton(_CHAR_2_QSTRING(__tr("&Add String")), this);
	pb->setEnabled(g_pOptions->m_bUseStringSubstitution);
	connect(b, SIGNAL(toggled(bool)), pb, SLOT(setEnabled(bool)));
	connect(pb, SIGNAL(clicked()), this, SLOT(addNewString()));
	g->addWidget(pb, 10, 0);

	pb = new KviPushButton(_CHAR_2_QSTRING(__tr("&Remove String")), this);
	pb->setEnabled(g_pOptions->m_bUseStringSubstitution);
	connect(b, SIGNAL(toggled(bool)), pb, SLOT(setEnabled(bool)));
	connect(pb, SIGNAL(clicked()), this, SLOT(deleteSelectedStrings()));
	g->addWidget(pb, 10, 1);

	KviListViewItem *it;
	for(KviStrSubItem *m = g_pOptions->m_pStrSub->m_pList->first(); m; m = g_pOptions->m_pStrSub->m_pList->next() ) {
		it = new KviListViewItem(
			m_pStrSubView, _CHAR_2_QSTRING(m->szOriginal.ptr()), _CHAR_2_QSTRING(m->szSubstitute.ptr())
		);
	}

	g->setRowStretch(9, 1);
}

KviOptionsInput::~KviOptionsInput()
{
	// Nothing here
}

void KviOptionsInput::addNewString()
{
	(void) new KviListViewItem(m_pStrSubView,
		_CHAR_2_QSTRING(__tr("*")), _CHAR_2_QSTRING("*"),
		_CHAR_2_QSTRING(""), _CHAR_2_QSTRING(""),
		_CHAR_2_QSTRING("")
	);
}

void KviOptionsInput::deleteSelectedStrings()
{
	KviListViewItem *it = m_pStrSubView->firstChild();
	while( it ) {
		KviListViewItem *tmp = it;
		it = it->nextSibling();
		if( tmp->isSelected() ) delete tmp;
	}
}

void KviOptionsInput::commitChanges()
{
	KviBoolSelector::commitAll(this);
	KviCharSelector::commitAll(this);
	KviStringSelector::commitAll(this);

	while( g_pOptions->m_pStrSub->m_pList->first() )
		g_pOptions->m_pStrSub->m_pList->removeFirst();
	KviListViewItem *it = m_pStrSubView->firstChild();
	while( it ) {
		KviStrSubItem *item = new KviStrSubItem();
		item->szOriginal = it->text(0);
		item->szOriginal.stripWhiteSpace();
		item->szSubstitute = it->text(1);
		item->szSubstitute.stripWhiteSpace();
		if( item->szOriginal.hasData() && item->szSubstitute.hasData() )
			g_pOptions->m_pStrSub->m_pList->append(item);
		else
			delete item; // Incomplete... useless
		it = it->nextSibling();
	}
	// Disable substitution when there are no items
	if( g_pOptions->m_pStrSub->m_pList->count() == 0 )
		g_pOptions->m_bUseStringSubstitution = false;
}

#include "m_kvi_options_input.moc"
