// =============================================================================
//
//      --- kvi_popupeditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviPopupEditor"

#include <qcursor.h>
#include <qheader.h>
#include <qlayout.h>
#include <qsplitter.h>

#include "kvi_combobox.h"
#include "kvi_defines.h"
#include "kvi_label.h"
#include "kvi_lineedit.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_popupeditor.h"
#include "kvi_popuplistview.h"
#include "kvi_popupmenu.h"
#include "kvi_pushbutton.h"
#include "kvi_script_editor.h"
#include "kvi_userpopupmenu.h"

// TODO: Totally missing quick help!

extern KviUserPopupMenu *g_pChannelPopup;
extern KviUserPopupMenu *g_pUserlistPopup;
extern KviUserPopupMenu *g_pNotifylistPopup;
extern KviUserPopupMenu *g_pQueryPopup;
extern KviUserPopupMenu *g_pConsolePopup;
extern KviUserPopupMenu *g_pDccChatPopup;

extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

int g_iLastSelectedPopup = -1;

/*
	@quickhelp: KviPopupEditor
	@widget: Popups tab
		Here, you can edit your popup menus.<br>
		In the upper list box, you can select the popup to edit:<br>
		&nbsp;Channel Window - Shown when right clicking in the channel window.<br>
		&nbsp;Query Window - Shown when right clicking in a query window.<br>
		&nbsp;Console Window - Shown when right clicking in the console window.<br>
		&nbsp;DCC Chat Window - Shown when right clicking in a DCC chat window.<br>
		&nbsp;Notify List - Shown when right clicking on a nickname in the right pane of the Console window (the notify list).<br>
		&nbsp;User List - Shown when right clicking on a nickname in the right pane of a channel window (the user list).<br>
		The lower tree view displays the current popup.<br>
		A right click in the tree view shows a menu that allows adding/removing
		items from the popup.<br>
		In the lower editor, you edit the popup code associated with the selected popup item.<br>
		In the 'Icon' line edit, you can enter the number of an icon to use with the popup (see the <a href="echo.kvihelp">echo</a> -i option).<br>
		The 'Icon' can be also an absolute or relative (to the KVIrc pics directory) file name.<br>
		Right clicking in the editor displays a helper menu.<br>
		The 'Name' line edit contains the current item name.<br>
		The "Test" button allows you to test your popup in its current state, and clicking an item
		in the popup selects it in the tree view.<br>
*/

/**
 *
 * POPUP EDITOR
 *
 * This widget allows editing of the popups
 *
 */
KviPopupEditor::KviPopupEditor(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	m_pChannelPopup    = new KviUserPopupMenu(0, "dummy");
	m_pQueryPopup      = new KviUserPopupMenu(0, "dummy");
	m_pNotifylistPopup = new KviUserPopupMenu(0, "dummy");
	m_pUserlistPopup   = new KviUserPopupMenu(0, "dummy");
	m_pConsolePopup    = new KviUserPopupMenu(0, "dummy");
	m_pDccChatPopup    = new KviUserPopupMenu(0, "dummy");

	   m_pChannelPopup->copyFromPopup(g_pChannelPopup);
	     m_pQueryPopup->copyFromPopup(g_pQueryPopup);
	m_pNotifylistPopup->copyFromPopup(g_pNotifylistPopup);
	  m_pUserlistPopup->copyFromPopup(g_pUserlistPopup);
	   m_pConsolePopup->copyFromPopup(g_pConsolePopup);
	   m_pDccChatPopup->copyFromPopup(g_pDccChatPopup);

	QGridLayout *g = new QGridLayout(this, 4, 4, 4, 2);

	KviPushButton *b = new KviPushButton(_CHAR_2_QSTRING(__tr("&Test Popup")), this);
	g->addWidget(b, 0, 2);
	connect(b, SIGNAL(clicked()), this, SLOT(testCurrentPopup()));

	b = new KviPushButton(_CHAR_2_QSTRING(__tr("&Clear Popup")), this);
	g->addWidget(b, 0, 3);
	connect(b, SIGNAL(clicked()), this, SLOT(clearCurrentPopup()));

	m_pPopupsBox = new KviComboBox(false, this);
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Channel Window (channel)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Console Window (console)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("DCC Chat Window (dccchat)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Console Notify List (notifylist)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Query Window (query)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Channel Userlist (userlist)"));
	g->addMultiCellWidget(m_pPopupsBox, 0, 0, 0, 1);

	connect(m_pPopupsBox, SIGNAL(activated(int)), this, SLOT(popupComboSelectionChanged(int)));

	QSplitter *splitter = new QSplitter(Qt::Vertical, this);
	splitter->setOpaqueResize();

	m_pPopupView = new KviListView(splitter);
	m_pPopupView->addColumn(_CHAR_2_QSTRING("Item"));
	m_pPopupView->addColumn(_CHAR_2_QSTRING("Type"));
	m_pPopupView->setRootIsDecorated(true);
	m_pPopupView->setMultiSelection(false);
	m_pPopupView->header()->setClickEnabled(false);
	m_pPopupView->viewport()->installEventFilter(this); // Need to adjust the behaviour of mousePressEvent

	m_pPopupPopup = new KviPopupMenu(this);

	connect(m_pPopupView, SIGNAL(selectionChanged(KviListViewItem *)), this, SLOT(popupItemSelected(KviListViewItem *)));
	connect(m_pPopupView,
		SIGNAL(rightButtonPressed(KviListViewItem *, const QPoint &, int)),
		this,
		SLOT(popupViewRightButtonPressed(KviListViewItem *, const QPoint &, int))
	);

	m_pPopupEditor = new KviScriptEditor(splitter);
	g->addMultiCellWidget(splitter, 1, 1, 0, 3);

	m_pIconPopup = new KviPopupMenu(this);
	KviPopupMenu *ppp = m_pIconPopup;
	connect(ppp, SIGNAL(activated(int)), this, SLOT(iconPopupActivated(int)));
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		ppp->insertItem(*(g_pixViewOut[i]), i);
		if( ((i % 15) == 0) && (i != 0) ) {
			KviPopupMenu *sub = new KviPopupMenu(ppp);
			ppp->insertItem(__tr("More..."), sub);
			ppp = sub;
			connect(ppp, SIGNAL(activated(int)), this, SLOT(iconPopupActivated(int)));
		}
	}

	KviLabel *l = new KviLabel(_CHAR_2_QSTRING("Icon:"), this);
	g->addWidget(l, 2, 0);

	m_pPopupIconEditor = new KviLineEdit(this);
	m_pPopupIconEditor->installEventFilter(this);

	g->addMultiCellWidget(m_pPopupIconEditor, 2, 2, 1, 3);

	l = new KviLabel(_CHAR_2_QSTRING("Name:"), this);
	g->addWidget(l, 3, 0);

	m_pPopupNameEditor = new KviLineEdit(this);
	g->addMultiCellWidget(m_pPopupNameEditor, 3, 3, 1, 3);

	g->setColStretch(1, 10);

	m_pEditedMenu      = 0;
	m_pEditedPopupItem = 0;
	m_pClipboardItem   = 0;

	if( (g_iLastSelectedPopup < 0) || (g_iLastSelectedPopup > 5) )
		g_iLastSelectedPopup = 0;
	m_pPopupsBox->setCurrentItem(g_iLastSelectedPopup);
	popupComboSelectionChanged(g_iLastSelectedPopup);
}

KviPopupEditor::~KviPopupEditor()
{
	destroyClipboard();
	delete m_pConsolePopup;    m_pConsolePopup    = 0;
	delete m_pChannelPopup;    m_pChannelPopup    = 0;
	delete m_pQueryPopup;      m_pQueryPopup      = 0;
	delete m_pUserlistPopup;   m_pUserlistPopup   = 0;
	delete m_pNotifylistPopup; m_pNotifylistPopup = 0;
	delete m_pDccChatPopup;    m_pDccChatPopup    = 0;
}

void KviPopupEditor::destroyClipboardItem(KviPopupClipboardItem *i)
{
	if( !i ) return;
	if( i->pItemList ) {
		for( KviPopupClipboardItem *it = i->pItemList->first(); it; it = i->pItemList->next() ) {
			destroyClipboardItem(it);
		}
		delete i->pItemList;
	}
	delete i;
}

void KviPopupEditor::destroyClipboard()
{
	if( m_pClipboardItem ) {
		destroyClipboardItem(m_pClipboardItem);
		m_pClipboardItem = 0;
	}
}

void KviPopupEditor::commit()
{
	if( m_pEditedMenu )
		saveMenu(m_pEditedMenu);

	g_iLastSelectedPopup = m_pPopupsBox->currentItem();

	   g_pChannelPopup->copyFromPopup(m_pChannelPopup);
	     g_pQueryPopup->copyFromPopup(m_pQueryPopup);
	g_pNotifylistPopup->copyFromPopup(m_pNotifylistPopup);
	  g_pUserlistPopup->copyFromPopup(m_pUserlistPopup);
	   g_pConsolePopup->copyFromPopup(m_pConsolePopup);
	   g_pDccChatPopup->copyFromPopup(m_pDccChatPopup);
}

bool KviPopupEditor::eventFilter(QObject *o, QEvent *e)
{
	// Poor hack
	if( o == m_pPopupIconEditor ) {
		if( e->type() == QEvent::MouseButtonPress ) {
			if( ((QMouseEvent *) e)->button() & RightButton) {
				m_pIconPopup->popup(QCursor::pos());
				return true;
			}
		}
		return false;
	}
	if( e->type() == QEvent::MouseButtonPress ) {
		if( o == m_pPopupView->viewport() ) {
			if( !(m_pPopupView->firstChild()) ) {
				popupViewRightButtonPressed(0, m_pPopupView->mapToGlobal(((QMouseEvent *) e)->pos()), 0);
				return true;
			}
		}
	}
	return QWidget::eventFilter(o, e);
}

void KviPopupEditor::popupComboSelectionChanged(int index)
{
	switch( index ) {
		case 0: startEditPopup(m_pChannelPopup);    break;
		case 1: startEditPopup(m_pConsolePopup);    break;
		case 2: startEditPopup(m_pDccChatPopup);    break;
		case 3: startEditPopup(m_pNotifylistPopup); break;
		case 4: startEditPopup(m_pQueryPopup);      break;
		case 5: startEditPopup(m_pUserlistPopup);   break;
		default: /* Never here */ break;
	}
}

void KviPopupEditor::startEditPopup(KviUserPopupMenu *p)
{
	if( m_pEditedMenu ) saveMenu(m_pEditedMenu);

	m_pEditedMenu = p;
	m_pPopupView->clear();
	m_pPopupView->setSorting(-1);
	appendChildPopup(0, p);
	m_pEditedPopupItem = 0;
	if( !(m_pPopupView->firstChild()) )
		disablePopupEditors();
	else
		m_pPopupView->setSelected(m_pPopupView->firstChild(), true);
}

void KviPopupEditor::testCurrentPopup()
{
	if( m_pEditedMenu ) {
		saveMenu(m_pEditedMenu);
		m_pEditedMenu->doTestPopup(this, SLOT(testPopupSlot(int)));
	}
}

void KviPopupEditor::clearCurrentPopup()
{
	if( KviMessageBox::warningYesNo(
		__tr("Are you sure you want to clear this popup?"),
		__tr("Clear Popup"),
		this) == KviMessageBox::Yes
	) {
		m_pPopupView->clear();
		m_pPopupView->setSorting(-1);
	}
}

void KviPopupEditor::testPopupSlot(int id)
{
	if( !selectPopupItemById(0, id) )
		debug("Oops... popup item not found in the list view?");
}

bool KviPopupEditor::selectPopupItemById(KviPopupListViewItem *par, int id)
{
	KviPopupListViewItem *it = (KviPopupListViewItem *) (par ? par->firstChild() : m_pPopupView->firstChild());
	while( it ) {
		if( it->m_type == KviPopupListViewItem::Item ) {
			if( it->m_id == id ) {
				// Open all parents
				par = (KviPopupListViewItem *) it->parent();
				while( par ) {
					par->setOpen(true);
					par = (KviPopupListViewItem *) par->parent();
				}
				// Select it
				m_pPopupView->setSelected(it, true);
				// And ensure it is visible
				m_pPopupView->ensureItemVisible(it);
				return true;
			}
		} else if( it->m_type == KviPopupListViewItem::Submenu ) {
			if( selectPopupItemById(it, id) )
				return true;
		}
		it = (KviPopupListViewItem *) it->nextSibling();
	}
	return false;
}

void KviPopupEditor::saveMenu(KviUserPopupMenu *p)
{
	if( m_pEditedPopupItem )
		savePopupItem(m_pEditedPopupItem); // Do not forget to save the last one
	p->clearAll();
	saveSubmenu(p, 0);
}

void KviPopupEditor::saveSubmenu(KviUserPopupMenu *p, KviPopupListViewItem *par)
{
	KviPopupListViewItem *it = (KviPopupListViewItem *) (par ? par->firstChild() : m_pPopupView->firstChild());
	while( it ) {
		if( it->m_type == KviPopupListViewItem::Separator ) {
			p->insertSeparatorItem();
			it->m_id = 0;
		} else {
			if( it->m_type == KviPopupListViewItem::Item ) {
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if( tmp.isEmpty() )
					tmp = __tr("Unnamed");
				it->m_id = p->insertNormalItem(tmp.ptr(), it->m_icon.ptr(), it->m_buffer.ptr());
			} else {
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if( tmp.isEmpty() )
					tmp = __tr("Unnamed");
				KviStr yourName(KviStr::Format, "usertoolbar_script_popup_%s", tmp.ptr());
				KviUserPopupMenu *sub = new KviUserPopupMenu(0, yourName.ptr());
				saveSubmenu(sub, it);
				p->insertPopupItem(tmp.ptr(), it->m_icon.ptr(), sub);
				it->m_id = 0;
			}
		}
		it = (KviPopupListViewItem *) it->nextSibling();
	}
}

void KviPopupEditor::disablePopupEditors()
{
	m_pPopupEditor->setText(_CHAR_2_QSTRING(""));
	m_pPopupIconEditor->setText(_CHAR_2_QSTRING(""));
	m_pPopupNameEditor->setText(_CHAR_2_QSTRING(""));
	m_pPopupEditor->setEnabled(false);
	m_pPopupIconEditor->setEnabled(false);
	m_pPopupNameEditor->setEnabled(false);
}

void KviPopupEditor::iconPopupActivated(int id)
{
	if( (id >= 0) && (id <= KVI_OUT_NUM_IMAGES) && m_pPopupIconEditor->isEnabled() ) {
		QString tmp;
		tmp.setNum(id);
		m_pPopupIconEditor->setText(tmp);
		if( m_pEditedPopupItem )
			savePopupItem(m_pEditedPopupItem);
		return;
	}
}

void KviPopupEditor::appendChildPopup(KviPopupListViewItem *parent, KviUserPopupMenu *p)
{
	KviPopupListViewItem *it = 0;
	for( KviUserPopupMenuData *d = p->m_pDataList.first(); d; d = p->m_pDataList.next() ) {
		if( d->menu ) {
			KviStr tmp(p->text(d->id));
			if( parent ) {
				if( it ) { // Insert after (Qt sometimes strangely inverts the insertion order)
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
						parent, it, tmp.ptr(), "submenu", 0, d->image.ptr()
					);
				} else {
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
						parent, tmp.ptr(), "submenu", 0, d->image.ptr()
					);
				}
			} else {
				if( it ) {
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
						m_pPopupView, it, tmp.ptr(), "submenu", 0, d->image.ptr()
					);
				} else {
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
						m_pPopupView, tmp.ptr(), "submenu", 0, d->image.ptr()
					);
				}
			}
			appendChildPopup(it, d->menu);
		} else {
			if( (d->id == 0) && kvi_strEqualCS(d->buffer.ptr(), " ") ) {
				if( parent ) {
					if( it ) {
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
							parent, it, "----", "separator", 0, 0
						);
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
							parent, "----", "separator", 0, 0
						);
					}
				} else {
					if( it ) {
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
							m_pPopupView, it, "----", "separator", 0, 0
						);
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
							m_pPopupView, "----", "separator", 0, 0
						);
					}
				}
			} else {
				KviStr tmp(p->text(d->id));
				if( parent ) {
					if( it ) {
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,
							parent, it, tmp.ptr(), "item", d->buffer.ptr(), d->image.ptr()
						);
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,
							parent, tmp.ptr(), "item", d->buffer.ptr(), d->image.ptr()
						);
					}
				} else {
					if( it ) {
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,
							m_pPopupView, it, tmp.ptr(), "item", d->buffer.ptr(), d->image.ptr()
						);
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,
							m_pPopupView, tmp.ptr(), "item", d->buffer.ptr(), d->image.ptr()
						);
					}
				}
			}
		}
	}
}

void KviPopupEditor::popupItemSelected(KviListViewItem *i)
{
	if( m_pEditedPopupItem )
		savePopupItem(m_pEditedPopupItem);

	KviPopupListViewItem *it = (KviPopupListViewItem *) i;
	m_pEditedPopupItem = it;
	if( !it ) {
		disablePopupEditors();
		return;
	}

	if( it->m_type == KviPopupListViewItem::Submenu ) {
		m_pPopupEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupEditor->setEnabled(false);
		m_pPopupIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
		m_pPopupIconEditor->setEnabled(true);
		m_pPopupNameEditor->setText(it->text(0));
		m_pPopupNameEditor->setEnabled(true);
	} else if( it->m_type == KviPopupListViewItem::Separator ) {
		m_pPopupEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupEditor->setEnabled(false);
		m_pPopupIconEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupIconEditor->setEnabled(false);
		m_pPopupNameEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupNameEditor->setEnabled(false);
	} else {
		m_pPopupEditor->setText(_CHAR_2_QSTRING(it->m_buffer.ptr()));
		m_pPopupEditor->setEnabled(true);
		m_pPopupIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
		m_pPopupIconEditor->setEnabled(true);
		m_pPopupNameEditor->setText(it->text(0));
		m_pPopupNameEditor->setEnabled(true);
	}
}

void KviPopupEditor::savePopupItem(KviPopupListViewItem *p)
{
	if( m_pEditedPopupItem->m_type == KviPopupListViewItem::Item ) {
		m_pEditedPopupItem->m_buffer = m_pPopupEditor->text();
		m_pEditedPopupItem->m_icon   = m_pPopupIconEditor->text();
		QString s = m_pPopupNameEditor->text();
		s.stripWhiteSpace();
		if( s.isEmpty() )
			s = _CHAR_2_QSTRING(__tr("Unnamed"));
		m_pEditedPopupItem->setText(0, s);
		m_pEditedPopupItem->tryLoadImage();
	} else if( m_pEditedPopupItem->m_type == KviPopupListViewItem::Submenu ) {
		m_pEditedPopupItem->m_icon = m_pPopupIconEditor->text();
		QString s = m_pPopupNameEditor->text();
		s.stripWhiteSpace();
		if( s.isEmpty() )
			s = _CHAR_2_QSTRING(__tr("Unnamed"));
		m_pEditedPopupItem->setText(0, s);
		m_pEditedPopupItem->tryLoadImage();
	}
}

void KviPopupEditor::popupViewRightButtonPressed(KviListViewItem *i, const QPoint &p, int)
{
	if( i )
		m_pPopupView->setSelected(i, true);
	else {
		m_pPopupView->clearSelection();
		if( m_pEditedPopupItem )
			savePopupItem(m_pEditedPopupItem);
	}
	m_pEditedPopupItem = (KviPopupListViewItem *) i;

	if( m_pEditedPopupItem ) {
		m_pPopupPopup->clear();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Remove Item")), this, SLOT(removeEditedPopupItem()));
		m_pPopupPopup->insertSeparator();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Add Item Below")), this, SLOT(addEmptyPopupItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add &Submenu Below")), this, SLOT(addSubmenuItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&eparator Below")), this, SLOT(addSeparatorItem()));
		if( m_pEditedPopupItem->parent() == m_pEditedPopupItem->itemAbove() ) {
			m_pPopupPopup->insertSeparator();
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("A&dd Item Above")), this, SLOT(addEmptyPopupItemAbove()));
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&ubmenu Above")), this, SLOT(addSubmenuItemAbove()));
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add Separat&or Above")), this, SLOT(addSeparatorItemAbove()));
		}
		if( m_pEditedPopupItem->m_type == KviPopupListViewItem::Submenu ) {
			m_pPopupPopup->insertSeparator();
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add I&tem Inside")), this, SLOT(addEmptyPopupItemInside()));
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add Su&bmenu Inside")), this, SLOT(addSubmenuItemInside()));
		}
		m_pPopupPopup->insertSeparator();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Copy Item")), this, SLOT(copyItem()));
		if( m_pClipboardItem ) {
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Paste Item Below")), this, SLOT(pasteItemBelow()));
			if( m_pEditedPopupItem->parent() == m_pEditedPopupItem->itemAbove() )
				m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Paste &Item Above")), this, SLOT(pasteItemAbove()));
			if( m_pEditedPopupItem->m_type == KviPopupListViewItem::Submenu )
				m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Paste Item I&nside")), this, SLOT(pasteItemInside()));
		}
	} else {
		// At the end?
		disablePopupEditors();
		m_pPopupPopup->clear();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Add Item")), this, SLOT(addEmptyPopupItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add &Submenu")), this, SLOT(addSubmenuItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&eparator")), this, SLOT(addSeparatorItem()));
	}
	m_pPopupPopup->popup(p);
}

void KviPopupEditor::removeEditedPopupItem()
{
	if( !m_pEditedPopupItem ) return;

	KviListViewItem *p = m_pEditedPopupItem->parent();
	KviPopupListViewItem *tmp = m_pEditedPopupItem;
	// Needs to be 0 before the delete (the KviListViewItem destructor
	// triggers an update in the KviListView that calls selectionChanged that
	// attempts to save the popup while it is being deleted!
	m_pEditedPopupItem = 0;
	delete tmp;
	if( p )
		m_pPopupView->setSelected(p, true);
	else {
		if( m_pPopupView->firstChild() )
			m_pPopupView->setSelected(m_pPopupView->firstChild(), true);
		else
			disablePopupEditors();
	}
}

void KviPopupEditor::copyItem()
{
	if( !m_pEditedPopupItem ) return;

	destroyClipboard();
	m_pClipboardItem = m_pEditedPopupItem->getClipboardItem();
}

void KviPopupEditor::addEmptyPopupItem()
{
	KviPopupListViewItem *it;
	if( m_pEditedPopupItem ) {
		if( m_pEditedPopupItem->parent() ) {
			it = new KviPopupListViewItem(KviPopupListViewItem::Item,
				((KviPopupListViewItem *) m_pEditedPopupItem->parent()),
				m_pEditedPopupItem, "unnamed", "item",
				__tr("# Enter the script for the new item here"), ""
			);
		} else {
			it = new KviPopupListViewItem(KviPopupListViewItem::Item,
				m_pPopupView, m_pEditedPopupItem, "unnamed", "item",
				__tr("# Enter the script for the new item here"), ""
			);
		}
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Item,
			m_pPopupView, "unnamed", "item",
			__tr("# Enter the script for the new item here"), ""
		);
	}
	m_pPopupView->setSelected(it, true);
}

KviPopupListViewItem *KviPopupEditor::pasteClipboard(
	KviPopupListViewItem *par, KviPopupListViewItem *aft, KviPopupClipboardItem *it, bool bNullAfter)
{
	if( !it ) return 0;

	KviPopupListViewItem *i;
	const char *type = 0;
	static const char *types[4] =
	{
		"item", "submenu", "separator", "???"
	};

	switch( it->type ) {
		case KviPopupListViewItem::Item:      type = types[0]; break;
		case KviPopupListViewItem::Submenu:   type = types[1]; break;
		case KviPopupListViewItem::Separator: type = types[2]; break;
		default: type = types[3]; break;
	}
	if( par ) {
		if( aft || bNullAfter ) {
			i = new KviPopupListViewItem(it->type, par, aft, it->szName.ptr(), type, it->szBuffer.ptr(), it->szIcon.ptr());
		} else {
			i = new KviPopupListViewItem(it->type, par, it->szName.ptr(), type, it->szBuffer.ptr(), it->szIcon.ptr());
		}
	} else {
		if( aft || bNullAfter ) {
			i = new KviPopupListViewItem(
				it->type, m_pPopupView, aft, it->szName.ptr(), type, it->szBuffer.ptr(), it->szIcon.ptr()
			);
		} else {
			i = new KviPopupListViewItem(
				it->type, m_pPopupView, it->szName.ptr(), type, it->szBuffer.ptr(), it->szIcon.ptr()
			);
		}
	}
	KviPopupListViewItem *last = 0;
	if( it->pItemList ) {
		for( KviPopupClipboardItem *c = it->pItemList->first(); c; c = it->pItemList->next() ) {
			last = pasteClipboard(i, last, c);
		}
	}
	return i;
}

void KviPopupEditor::pasteItemBelow()
{
	KviPopupListViewItem *it;
	if( m_pEditedPopupItem ) {
		if( m_pEditedPopupItem->parent() )
			it = pasteClipboard((KviPopupListViewItem *) m_pEditedPopupItem->parent(), m_pEditedPopupItem, m_pClipboardItem);
		else
			it = pasteClipboard(0, m_pEditedPopupItem, m_pClipboardItem);
	} else it = pasteClipboard(0, 0, m_pClipboardItem);

	if( it )
		m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::pasteItemInside()
{
	if( !m_pEditedPopupItem ) return;

	if( m_pEditedPopupItem->m_type != KviPopupListViewItem::Submenu )
		return;

	KviPopupListViewItem *it = pasteClipboard(m_pEditedPopupItem, 0, m_pClipboardItem);
	if( it )
		m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::pasteItemAbove()
{
	if( !m_pEditedPopupItem ) return;

	if( m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove() )
		return;

	KviPopupListViewItem *it;
	if( m_pEditedPopupItem->parent() )
		it = pasteClipboard((KviPopupListViewItem *) m_pEditedPopupItem->parent(), 0, m_pClipboardItem, true);
	else
		it = pasteClipboard(0, 0, m_pClipboardItem, true);

	if( it )
		m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addEmptyPopupItemAbove()
{
	if( !m_pEditedPopupItem ) return;

	if( m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove() )
		return;

	KviPopupListViewItem *it;
	if( m_pEditedPopupItem->parent() ) {
		it = new KviPopupListViewItem(KviPopupListViewItem::Item,
			((KviPopupListViewItem *) m_pEditedPopupItem->parent()),
			0, "unnamed", "item",
			__tr("# Enter the script for the new item here"), ""
		);
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Item,
			m_pPopupView,
			0, "unnamed", "item",
			__tr("# Enter the script for the new item here"), ""
		);
	}
	m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addSubmenuItemAbove()
{
	if( !m_pEditedPopupItem ) return;

	if( m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove() )
		return;

	KviPopupListViewItem *sb;
	if( m_pEditedPopupItem->parent() ) {
		sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
			((KviPopupListViewItem *) m_pEditedPopupItem->parent()),
			0, "unnamed", "submenu", "", ""
		);
	} else {
		sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
			m_pPopupView,
			0, "unnamed", "submenu", "", ""
		);
	}

	KviPopupListViewItem *it = new KviPopupListViewItem(KviPopupListViewItem::Item,
		sb,
		"unnamed", "item",
		__tr("# Enter the script for the new item here"), ""
	);
	sb->setOpen(true);

	m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addSeparatorItemAbove()
{
	if( !m_pEditedPopupItem ) return;

	if( m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove() )
		return;

	KviPopupListViewItem *it;
	if( m_pEditedPopupItem->parent() ) {
		it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
			((KviPopupListViewItem *) m_pEditedPopupItem->parent()),
			0, "----", "separator", "", ""
		);
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
			m_pPopupView,
			0, "----", "separator", "", ""
		);
	}

	m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addEmptyPopupItemInside()
{
	if( !m_pEditedPopupItem ) return;

	if( m_pEditedPopupItem->m_type != KviPopupListViewItem::Submenu )
		return;

	KviPopupListViewItem *it = new KviPopupListViewItem(KviPopupListViewItem::Item,
		m_pEditedPopupItem, "unnamed", "item",
		__tr("# Enter the script for the new item here"), ""
	);
	m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addSubmenuItemInside()
{
	if( !m_pEditedPopupItem ) return;

	KviPopupListViewItem *sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
		m_pEditedPopupItem, "unnamed", "submenu", "", ""
	);
	KviPopupListViewItem *it = new KviPopupListViewItem(KviPopupListViewItem::Item,
		sb, "unnamed", "item",
		__tr("# Enter the script for the new item here"), ""
	);
	sb->setOpen(true);
	m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addSubmenuItem()
{
	KviPopupListViewItem *sb;
	if( m_pEditedPopupItem ) {
		if( m_pEditedPopupItem->parent() ) {
			sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
				((KviPopupListViewItem *) m_pEditedPopupItem->parent()),
				m_pEditedPopupItem, "unnamed", "submenu", "", ""
			);
		} else {
			sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
				m_pPopupView,
				m_pEditedPopupItem, "unnamed", "submenu", "", ""
			);
		}
	} else {
		sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,
			m_pPopupView, "unnamed", "submenu", "", ""
		);
	}

	KviPopupListViewItem *it = new KviPopupListViewItem(KviPopupListViewItem::Item,
		sb, "unnamed", "item",
		__tr("# Enter the script for the new item here"), ""
	);
	sb->setOpen(true);

	m_pPopupView->setSelected(it, true);
}

void KviPopupEditor::addSeparatorItem()
{
	KviPopupListViewItem *it;
	if( m_pEditedPopupItem ) {
		if( m_pEditedPopupItem->parent() ) {
			it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
				((KviPopupListViewItem *) m_pEditedPopupItem->parent()),
				m_pEditedPopupItem, "----", "separator", "", ""
			);
		} else {
			it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
				m_pPopupView,
				m_pEditedPopupItem, "----", "separator", "", ""
			);
		}
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Separator,
			m_pPopupView, "separator", "separator", "", ""
		);
	}

	m_pPopupView->setSelected(it, true);
}

#include "m_kvi_popupeditor.moc"
