// =============================================================================
//
//      --- kvi_script_editor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptEditor"

#include <qhbox.h>
#include <qlayout.h>
#include <qtoolbutton.h>

#include "kvi_app.h"
#include "kvi_fileutils.h"
#include "kvi_label.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_popupmenu.h"
#include "kvi_script_editor.h"
#include "kvi_script_editorwidget.h"
#include "kvi_string.h"

KviScriptEditor::KviScriptEditor(QWidget *parent)
	: QWidget(parent)
{
	QGridLayout *g = new QGridLayout(this, 2, 4, 0, 0);
	m_pEditor = new KviScriptEditorWidget(this);
	g->addMultiCellWidget(m_pEditor, 0, 0, 0, 3);
	g->setRowStretch(0, 1);

	QToolButton *b = new QToolButton(UpArrow, this);
	g->addWidget(b, 1, 0);

	KviPopupMenu *pop = new KviPopupMenu(b);
	pop->insertItem(__tr("&Save to File..."),   this, SLOT(saveToFile()));
	pop->insertItem(__tr("&Load from File..."), this, SLOT(loadFromFile()));
	b->setPopup(pop);
	b->setPopupDelay(0);

	KviLabel *l = new KviLabel(__tr("Ctrl+W completes commands and functions"), this);
	l->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	g->setColStretch(1, 1);
	g->addWidget(l, 1, 1);

	QHBox *box;
	box = new QHBox(this);
	box->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	g->addWidget(box, 1, 2);
	m_pAux1 = new KviLabel(__tr("Row: "), box);
	m_pRowLabel = new KviLabel("0", box);

	box = new QHBox(this);
	box->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	g->addWidget(box, 1, 3);
	m_pAux2 = new KviLabel(__tr("Col: "), box);
	m_pColLabel = new KviLabel("0", box);

	QObject::connect(m_pEditor, SIGNAL(cursorRowChanged(int)), this, SLOT(cursorRowChanged(int)));
	QObject::connect(m_pEditor, SIGNAL(cursorColChanged(int)), this, SLOT(cursorColChanged(int)));
	QObject::connect(m_pEditor, SIGNAL(message(const QString &)), l, SLOT(setText(const QString &)));
}

KviScriptEditor::~KviScriptEditor()
{
	//  Nothing here
}

void KviScriptEditor::cursorRowChanged(int row)
{
	QString tmp;
	tmp.setNum(row);
	m_pRowLabel->setText(tmp);
}

void KviScriptEditor::cursorColChanged(int col)
{
	QString tmp;
	tmp.setNum(col);
	m_pColLabel->setText(tmp);
}

void KviScriptEditor::setEnabled(bool bEnabled)
{
	QWidget::setEnabled(bEnabled);
	m_pEditor->setEnabled(bEnabled);
	m_pAux1->setEnabled(bEnabled);
	m_pAux2->setEnabled(bEnabled);
	m_pColLabel->setEnabled(bEnabled);
	m_pRowLabel->setEnabled(bEnabled);
}

void KviScriptEditor::saveToFile()
{
	KviStr fName = kvi_askForSaveFileName();
	if( fName.hasData() ) {
		if( kvi_fileExists(fName.ptr()) ) {
			if( KviMessageBox::warningYesNo(
				__tr("The file already exists. Do you wish to overwrite it?"),
				__tr("Script Center")) == KviMessageBox::No
			) {
				return;
			}
		}
		KviStr buffer = text();
		if( !kvi_writeFile(fName.ptr(), buffer) )
			g_pApp->warningBox(__tr("Cannot open file %s for writing. Save failed."), fName.ptr());
	}
}

void KviScriptEditor::loadFromFile()
{
	KviStr fName = kvi_askForOpenFileName();
	if( fName.hasData() ) {
		KviStr buffer;
		if( kvi_loadFile(fName.ptr(), buffer) )
			setText(buffer.ptr());
		else
			g_pApp->warningBox(__tr("Unable to to open file %s"), fName.ptr());
	}
}

KviScriptEditorWidget *KviScriptEditor::editor()
{
	return m_pEditor;
};

void KviScriptEditor::setText(const QString &text)
{
	m_pEditor->setText(text);
}

QString KviScriptEditor::text()
{
	return m_pEditor->text();
}

#include "m_kvi_script_editor.moc"
